// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.function.*;
import de.caff.generics.range.Range;

import java.util.*;
import java.util.function.Consumer;

import static de.caff.generics.Order.Ascending;
import static de.caff.generics.Order.Descending;

/**
 * Class which allows readonly access of raw bytes by index.
 * <p>
 * The {@link #EMPTY} constant or {@link #emptyIndexable()} method provide the same
 * useful return value to indicate emptiness.
 * <p>
 * Simple implementations should extend {@link Base} because
 * that provides useful implementations for standard Object methods and implements
 * {@link Comparable}.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since July 7, 2020
 * @see de.caff.generics.algorithm.FastByteSequenceSearch
 */
public interface ByteIndexable
        extends PrimitiveByteIterable,
                Sizeable
{
  /** Byte indexable with only one element with value 0. */
  ByteIndexable SINGLE_0 = singleton((byte)0);

  /** Indexable with the 256 hexadecimal 2-digit codes for the (unsigned) byte values defined by the index (uppercase). */
  Indexable<String> UPPERCASE_2DIGIT_HEXCODES = Indexable.viewByIndex(256, idx -> String.format("%02X", idx)).frozen();
  /** Indexable with the 256 hexadecimal 2-digit codes for the (unsigned) byte values defined by the index (lowercase). */
  Indexable<String> LOWERCASE_2DIGIT_HEXCODES = Indexable.viewByIndex(256, idx -> String.format("%02x", idx)).frozen();
  
  /**
   * Get the element at the given index.
   * @param index index between {@code 0} and {@code size() - 1}
   * @return element at the given index
   */
  byte get(int index);

  /**
   * Pythonesque get.
   * This allows accessing elements from the back by using negative indexes,
   * e.g. {@code -1} references the last element, {@code -2} its predecessor, and so on.
   * @param index index between {@code -size()} and {@code size() - 1}
   * @return element at the given index
   */
  default byte gyt(int index)
  {
    return get(Pythonesque.mapX(index, this));
  }

  /**
   * Get an element modulo size.
   * <p>
   * This maps the given {@code index} into the range of this indexable
   * by applying a modulo {@link #size()} operation. For empty indexable
   * this will throw an {@link IndexOutOfBoundsException} as there is no
   * possible index to get.
   * @param index index, possibly out of range, possibly even negative
   *              for Pythonesque access
   * @return element at the given, possibly modulated, index
   */
  default byte getMod(int index)
  {
    final int size = size();
    if (size == 0) {
      throw new IndexOutOfBoundsException("No element for empty indexable!");
    }
    return gyt(index % size);
  }

  /**
   * Get this indexable but with inverted order.
   * @return a view to this indexable which accesses the elements in reverse order
   */
  @NotNull
  default Base reverse()
  {
    return new Base()
    {
      @Override
      public int size()
      {
        return ByteIndexable.this.size();
      }

      @Override
      public byte get(int index)
      {
        return ByteIndexable.this.get(ByteIndexable.this.size() - index - 1);
      }

      @NotNull
      @Override
      public Base reverse()
      {
        return ByteIndexable.this.asBase();
      }
    };
  }

  /**
   * Is this indexable empty?
   * @return {@code true}: there are no elements in this indexable<br>
   *         {@code false}: this indexable has elements
   */
  default boolean isEmpty()
  {
    return size() == 0;
  }

  /**
   * Get a primitive int iterator.
   * @return {@code int} iterator which iterates over this indexable
   */
  @NotNull
  default PrimitiveIterator.OfInt intIterator()
  {
    return new PrimitiveIterator.OfInt()
    {
      private int index = 0;

      @Override
      public int nextInt()
      {
        if (index >= size()) {
          throw new NoSuchElementException("index: "+index);
        }
        return get(index++);
      }

      @Override
      public boolean hasNext()
      {
        return index < size();
      }
    };
  }

  /**
   * Get an primitive byte iterator.
   * @return {@code byte} iterator which iterates over this indexable
   */
  @NotNull
  default PrimitiveByteIterator byteIterator()
  {
    return byteIterator(0, size());
  }

  /**
   * Get a byte iterator from the given sub set.
   * @param from first index of iterator
   * @param to   one past last index of iterator
   * @return int iterator which iterates over the given part of this indexable
   */
  @NotNull
  default PrimitiveByteIterator byteIterator(final int from, final int to)
  {
    return new PrimitiveByteIterator()
    {
      private int index = from;

      @Override
      public byte nextByte()
      {
        if (index >= to) {
          throw new NoSuchElementException(String.format("Index %s out of allowed range [%d, %d[!",
                                                         index, from, to));
        }
        return get(index++);
      }

      @Override
      public boolean hasNext()
      {
        return index < to;
      }
    };
  }

  /**
   * Returns an iterator over elements of type {@code T}.
   *
   * @return an Iterator.
   */
  @NotNull
  @Override
  default Iterator<Byte> iterator()
  {
    return listIterator();
  }

  /**
   * Returns a list iterator over elements of type {@code T}.
   *
   * @return a list iterator.
   */
  @NotNull
  default ListIterator<Byte> listIterator()
  {
    return new ListIterator<Byte>() {
      private int index = 0;

      @Override
      public boolean hasNext()
      {
        return index < size();
      }

      @Override
      public Byte next()
      {
        if (index >= size()) {
          throw new NoSuchElementException("index: "+index);
        }
        return get(index++);
      }

      @Override
      public boolean hasPrevious()
      {
        return index > 0;
      }

      @Override
      public Byte previous()
      {
        if (index == 0) {
          throw new NoSuchElementException("index: -1");
        }
        return get(--index);
      }

      @Override
      public int nextIndex()
      {
        return index;
      }

      @Override
      public int previousIndex()
      {
        return index - 1;
      }

      @Override
      public void remove()
      {
        throw new UnsupportedOperationException();
      }

      @Override
      public void set(Byte t)
      {
        throw new UnsupportedOperationException();
      }

      @Override
      public void add(Byte t)
      {
        throw new UnsupportedOperationException();
      }
    };
  }

  /**
   * Get an indexable subset.
   * The subset includes indices {@code fromIndex} to {@code toIndex - 1}.
   * @param fromIndex start index of sub set
   * @param toIndex   index after last index
   * @return indxable subset view of this indexable
   */
  @NotNull
  default Base subSet(int fromIndex, int toIndex)
  {
    if (fromIndex < 0)
      throw new IndexOutOfBoundsException("fromIndex = " + fromIndex);
    if (toIndex > size())
      throw new IndexOutOfBoundsException("toIndex = " + toIndex);
    if (fromIndex > toIndex)
      throw new IllegalArgumentException("fromIndex(" + fromIndex +
                                         ") > toIndex(" + toIndex + ")");
    final int length = toIndex - fromIndex;
    return length == 0
            ? EMPTY
            : new Base()
    {
      @Override
      public int size()
      {
        return length;
      }

      @Override
      public byte get(int index)
      {
        if (index >= length) {
          throw new IndexOutOfBoundsException("No such element: "+index);
        }
        return ByteIndexable.this.get(index + fromIndex);
      }

      @NotNull
      @Override
      public Base subSet(int fromIdx, int toIdx)
      {
        if (fromIdx < 0) {
          throw new IndexOutOfBoundsException("fromIndex = " + fromIdx);
        }
        if (toIdx > length) {
          throw new IndexOutOfBoundsException("toIndex = " + toIdx);
        }
        if (fromIdx > toIdx) {
          throw new IllegalArgumentException("fromIndex(" + fromIdx +
                                             ") > toIndex(" + toIdx + ")");
        }
        return ByteIndexable.this.subSet(fromIndex + fromIdx,
                                         fromIndex + toIdx);
      }
    };
  }

  /**
   * Get an indexable subset.
   * This is the Pythonesque version which allows negative indexes.
   * @param fromIndex start index of sub set
   * @param toIndex   end index of sub set
   * @return indexable subset view of this indexable set
   */
  @NotNull
  default Base sybSet(int fromIndex, int toIndex)
  {
    return subSet(Pythonesque.mapX(fromIndex, this),
                  Pythonesque.mapX(toIndex, this));
  }

  /**
   * Create an indexable subset from the last elements of this indexable set.
   * This already allows Pythonesquew indexing.
   * @param fromIndex index to start with, negative counts from the back
   * @return indexable subset view of this indexable set
   */
  @NotNull
  default Base tailSet(int fromIndex)
  {
    return subSet(Pythonesque.mapX(fromIndex, this),
                  size());
  }

  /**
   * Create an indexable subset from the last elements of this indexable set.
   * This already allows Pythonesquew indexing.
   * @param toIndex index one after the end (equal to the length of the returned set),
   *                negative counts from the back
   * @return indexable subset view of this indexable set
   */
  @NotNull
  default Base headSet(int toIndex)
  {
    return subSet(0,
                  Pythonesque.mapX(toIndex, this));
  }

  /**
   * Return a mapped view on this BYTE indexable.
   * @param mapper mapper
   * @param <T> result type pf mapper
   * @return mapped indexable
   */
  @NotNull
  default <T> Indexable<T> view(@NotNull ByteFunction1<? extends T> mapper)
  {
    return new Indexable.Base<T>()
    {
      @Override
      public int size()
      {
        return ByteIndexable.this.size();
      }

      @Override
      public T get(int index)
      {
        return mapper.applyAsByte(ByteIndexable.this.get(index));
      }
    };
  }

  /**
   * Get a view on this indexable as an unmodifiable collection.
   * This is especially useful for conversion to standard collection classes, eg for adding all
   * elements via {@link Collection#addAll(Collection)}.
   * @return collection view
   */
  @NotNull
  default Collection<Byte> asCollection()
  {
    return new AbstractCollection<Byte>()
    {
      @NotNull
      @Override
      public Iterator<Byte> iterator()
      {
        return ByteIndexable.this.iterator();
      }

      @Override
      public int size()
      {
        return ByteIndexable.this.size();
      }
    };
  }

  /**
   * Get a view on this indexable as an unmodifiable list.
   * This is especially useful for conversion to standard collection classes, eg for adding all
   * elements via {@link Collection#addAll(Collection)}.
   * @return collection view
   * @see #toList()
   */
  @NotNull
  default List<Byte> asList()
  {
    return new AbstractList<Byte>()
    {
      @Override
      public Byte get(int index)
      {
        return ByteIndexable.this.get(index);
      }

      @Override
      public int size()
      {
        return ByteIndexable.this.size();
      }

      @NotNull
      @Override
      public Iterator<Byte> iterator()
      {
        return ByteIndexable.this.iterator();
      }
    };
  }

  /**
   * Create a list from this bye indexable.
   * This creates an independent list to which the elements of this
   * indexable are copied.
   * @return list copy of this indexable
   * @see #asList()
   */
  @NotNull
  default ArrayList<Byte> toList()
  {
    final ArrayList<Byte> list = new ArrayList<>(size());
    addAllTo(list);
    return list;
  }

  /**
   * Add all elements of this indexable to the given collection.
   * @param collection collection to add to
   */
  default void addAllTo(@NotNull Collection<? super Byte> collection)
  {
    for (Byte entry : this) {
      collection.add(entry);
    }
  }

  /**
   * Get a view of this indexable as a standard object-based indexable.
   * @return indexable with non-null {@code Byte} values
   */
  @NotNull
  default Indexable<Byte> asIndexable()
  {
    return new Indexable.Base<Byte>()
    {
      @Override
      public Byte get(int index)
      {
        return ByteIndexable.this.get(index);
      }

      @Override
      public int size()
      {
        return ByteIndexable.this.size();
      }
    };
  }

  /**
   * Convert this indexable into a Java array.
   * @return array copy containing all values of this indexable
   */
  @NotNull
  default byte[] toArray()
  {
    if (isEmpty()) {
      return Empty.BYTE_ARRAY;
    }
    final byte[] array = new byte[size()];
    addToArray(array, 0);
    return array;
  }


  /**
   * Add the content of this indexable to the given array.
   * @param array array where the content is added
   * @param pos   position in the array where the content of this indexable is added
   * @return array position after this indexable
   */
  default int addToArray(@NotNull byte[] array, int pos)
  {
    return addToArray(array, pos, 0, size());
  }

  /**
   * Add a part of the content of this indexable to the given array.
   * @param array     array where the content is added
   * @param arrayPos  position in hte array where the content is added
   * @param index     start index of this indexable which is added first
   * @param length    number of entries of this indexable added to the array
   * @return array position after the added content
   */
  default int addToArray(@NotNull byte[] array, int arrayPos,
                         int index, int length)
  {
    for (PrimitiveByteIterator it = byteIterator(index, index + length);  it.hasNext();  ) {
      array[arrayPos++] = it.nextByte();
    }
    return arrayPos;
  }

  /**
   * Convert this indexable into an int array.
   * @return array copy containing all values of this indexable
   */
  @NotNull
  default int[] toIntArray()
  {
    if (isEmpty()) {
      return Empty.INT_ARRAY;
    }
    final int[] array = new int[size()];
    addToArray(array, 0);
    return array;
  }


  /**
   * Add the content of this indexable to the given int array.
   * @param array array where the content is added
   * @param pos   position in the array where the content of this indexable is added
   * @return array position after this indexable
   */
  default int addToArray(@NotNull int[] array, int pos)
  {
    return addToArray(array, pos, 0, size());
  }

  /**
   * Add a part of the content of this indexable to the given intarray.
   * @param array     array where the content is added
   * @param arrayPos  position in hte array where the content is added
   * @param index     start index of this indexable which is added first
   * @param length    number of entries of this indexable added to the array
   * @return array position after the added content
   */
  default int addToArray(@NotNull int[] array, int arrayPos,
                         int index, int length)
  {
    for (PrimitiveByteIterator it = byteIterator(index, index + length);  it.hasNext();  ) {
      array[arrayPos++] = it.nextByte();
    }
    return arrayPos;
  }

  /**
   * Get the indexes of this indexable as an iterable range.
   * @return iterable indexes
   * @see #intIndexes()
   * @see Range#indexes(int)
   */
  @NotNull
  default Iterable<Integer> indexes()
  {
    return Range.indexes(size());
  }

  /**
   * Get the indexes of this indexable as an integer indexable.
   * @return indexes as integer indexable
   * @see #indexes()
   * @see IntIndexable#rangeFromSize(int)
   */
  @NotNull
  default IntIndexable intIndexes()
  {
    return IntIndexable.rangeFromSize(size());
  }

  /**
   * View this byte indexable as an integer indexable.
   * @return integer indexable view
   */
  @NotNull
  default IntIndexable.Base asIntIndexable()
  {
    return new IntIndexable.Base()
    {
      @Override
      public int size()
      {
        return ByteIndexable.this.size();
      }

      @Override
      public int get(int index)
      {
        return ByteIndexable.this.get(index);
      }
    };
  }

  /**
   * View this byte indexable as an integer indexable using unsigned values.
   * @return integer indexable view
   */
  @NotNull
  default IntIndexable.Base asUnsignedIndexable()
  {
    return new IntIndexable.Base()
    {
      @Override
      public int size()
      {
        return ByteIndexable.this.size();
      }

      @Override
      public int get(int index)
      {
        return ByteIndexable.this.get(index) & 0xFF;
      }
    };
  }

  /**
   * Get a byte array copy of this byte indexable.
   * @return byte array
   */
  @NotNull
  default byte[] toByteArray()
  {
    final int length = size();
    if (length == 0) {
      return Empty.BYTE_ARRAY;
    }
    final byte[] result = new byte[length];
    int index = 0;
    for (byte b : this) {
      result[index++] = b;
    }
    return result;
  }

  /**
   * Copy the content of this indexable into a byte array.
   * @param targetArray target byte array
   * @param targetStart start index in target
   */
  default void copyTo(@NotNull byte[] targetArray,
                      int targetStart)
  {
    copyTo(targetArray, targetStart, 0, size());
  }

  /**
   * Copy the content of this indexable (or a part of it)
   * into a byte array.
   * @param targetArray target byte array
   * @param targetStart start index in target
   * @param sourceStart start index in this indexable
   * @param length number of bytes to be copied
   */
  default void copyTo(@NotNull byte[] targetArray,
                      int targetStart,
                      int sourceStart,
                      int length)
  {
    for (int i = 0;  i < length;  ++i) {
      targetArray[targetStart + i] = get(sourceStart + i);
    }
  }

  /**
   * Go over all values and do a cumulative calculation.
   *
   * @param initialValue  start value for the operation, will be returned if this iterable is empty
   * @param foldOperation operation applied to each value, will get the accumulated value as its
   *                      first and the current element value as its second argument
   * @return accumulated value, result of applying fold operation to all values of this indexable
   *         iteratively
   */
  default byte foldLeft(byte initialValue,
                        @NotNull ByteOperator2 foldOperation)
  {
    byte result = initialValue;
    for (PrimitiveByteIterator iterator = byteIterator();
         iterator.hasNext();  ) {
      result = foldOperation.applyAsByte(result, iterator.nextByte());
    }
    return result;
  }

  /**
   * Convert this byte indexable into a string of hexadecimal numbers, 2 per byte, using uppercase.
   * @return hexadecimal string
   * @see #toHexString(boolean)
   */
  @NotNull
  default String toHexString()
  {
    return toHexString(true);
  }

  /**
   * Convert this byte indexable into a string of hexadecimal numbers, 2 per byte.
   * @param uppercase if {@code true} the returned string will use uppercase letters for non-decimal digits<br>
   *                  if {@code false} it will use lowercase letters
   * @return hexadecimal string
   */
  @NotNull
  default String toHexString(boolean uppercase)
  {
    if (isEmpty()) {
      return Empty.STRING;
    }
    final Indexable<String> hexcodes = uppercase
            ? UPPERCASE_2DIGIT_HEXCODES
            : LOWERCASE_2DIGIT_HEXCODES;
    final StringBuilder sb = new StringBuilder(2 * size());
    forEachByte(b -> sb.append(hexcodes.get(b & 0xFF)));
    return sb.toString();
  }

  /**
   * Create a view of this byte indexable with an inserted value.
   * Note that this is not efficient, but handy in certain situations
   * where only 1 or a few values are inserted.
   * Use {@link ExpandableIntIndexable} if you want to carry out more
   * stuff like this.
   * <p>
   * As this creates a view of this indexable, changes to this indexable might 
   * result in a disaster when using the returned indexable!
   *
   * @param index index where the value is inserted before the current index
   *              (<b>not Pythonesque</b>, because it is possible to add an value at {@code index == size()},
   *              and it is expected that most insertion will happen at 0 or at the end)
   * @param value inserted value at that index
   * @return a view of this indexable with 1 more value at the given index,
   *         all values beyond are moved one index forward
   * @see #withAppendedValue(byte)
   */
  @NotNull
  default ByteIndexable withInsertedValueAt(int index, byte value)
  {
    final int insertIndex = index;
    final int newSize = size() + 1;
    if (insertIndex == 0) {
      return new ByteIndexable.Base() {
        @Override
        public int size()
        {
          return newSize;
        }

        @Override
        public byte get(int index)
        {
          return index == 0
                  ? value
                  : ByteIndexable.this.get(index - 1);
        }
      };
    }
    if (insertIndex == newSize - 1) {
      return new ByteIndexable.Base() {

        @Override
        public int size()
        {
          return newSize;
        }

        @Override
        public byte get(int index)
        {
          return index == newSize - 1
                  ? value
                  : ByteIndexable.this.get(index);
        }
      };
    }
    return new ByteIndexable.Base()
    {
      @Override
      public int size()
      {
        return newSize;
      }

      @Override
      public byte get(int index)
      {
        if (index == insertIndex) {
          return value;
        }
        return ByteIndexable.this.get(index < insertIndex
                                                ? index
                                                : index - 1);
      }
    };
  }

  /**
   * Create a view with of this indexable with another value added at the end.
   * Note that this is not efficient, but handy in certain situations
   * where only 1 or a few items are added.
   * @param value value to add
   * @return indexable view of this indexable, with the given value added at the end,
   *         so size is larger by 1
   * @see #withInsertedValueAt(int, byte)
   */
  @NotNull
  default ByteIndexable withAppendedValue(byte value)
  {
    return withInsertedValueAt(size(), value);
  }

  /**
   * Create a view of this byte indexable with an exchanged value.
   * Note that this is not efficient, but handy in certain situations
   * where only 1 or a few values are exchanged.
   * Use {@link ExpandableIntIndexable} if you want to carry out more
   * stuff like this.
   * <p>
   * As this creates a view of this indexable, changes to this indexable could 
   * lead to unexpected results when using the returned indexable!
   *
   * @param index index where the element is exchanged, hiding the current element.
   *              (Pythonesque)
   * @param value exchanged value at that index
   * @return a view of this indexable with 1 more element at the given index,
   *         all elements beyond are moved one index forward
   */
  @NotNull
  default ByteIndexable withExchangedValueAt(int index, byte value)
  {
    final int insertIndex = Pythonesque.mapX(index, this);
    return new ByteIndexable.Base()
    {
      @Override
      public byte get(int index)
      {
        return index == insertIndex
                ? value
                : ByteIndexable.this.get(index);
      }

      @Override
      public int size()
      {
        return ByteIndexable.this.size();
      }
    };
  }

  /**
   * Create a view of this indexable with one value removed.
   * Note that this is not efficient, but handy in certain situations
   * where only one or a few values are removed.
   *
   * @param index index of the removed value (Pythonesque)
   * @return a view of this indexable where the value at the given index
   *         is removed
   */
  @NotNull
  default ByteIndexable withRemovedValueAt(int index)
  {
    final int removeIndex = Pythonesque.mapX(index, this);
    if (removeIndex == 0) {
      // remove first
      return tailSet(1);
    }
    final int sz = size() - 1;
    if (removeIndex == sz) {
      // remove last
      return headSet(-1);
    }
    return new ByteIndexable.Base()
    {
      @Override
      public byte get(int index)
      {
        return index < removeIndex
                ? ByteIndexable.this.get(index)
                : ByteIndexable.this.get(index + 1);
      }

      @Override
      public int size()
      {
        return sz;
      }
    };
  }

  /**
   * View this byte indexable transformed by an operator.
   * @param op transforming operator
   * @return view of this indexable with transformed values
   */
  @NotNull
  default ByteIndexable viewOp(@NotNull ByteOperator1 op)
  {
    return ByteIndexable.viewByIndex(size(),
                                       idx -> op.applyAsByte(ByteIndexable.this.get(idx)));
  }

  /**
   * View this byte integer indexable as a double indexable.
   * This will cast the values of this indexable on each request,
   * use {@link #frozen()} on the result to create an independent copy
   * if this seems like a problem to you.
   * @return a double indexable view of this indexable
   */
  @NotNull
  default DoubleIndexable viewAsDouble()
  {
    return new DoubleIndexable.Base()
    {
      @Override
      public double get(int index)
      {
        return ByteIndexable.this.get(index);
      }

      @Override
      public int size()
      {
        return ByteIndexable.this.size();
      }
    };
  }

  /**
   * View this byte integer indexable as a float indexable.
   * This will cast the values of this indexable on each request,
   * use {@link #frozen()} on the result to create an independent copy
   * if this seems like a problem to you.
   * @return a float indexable view of this indexable
   */
  @NotNull
  default FloatIndexable viewAsFloat()
  {
    return new FloatIndexable.Base()
    {
      @Override
      public float get(int index)
      {
        return ByteIndexable.this.get(index);
      }

      @Override
      public int size()
      {
        return ByteIndexable.this.size();
      }
    };
  }

  /**
   * View this byte integer indexable as a long integer indexable.
   * This will cast the values of this indexable on each request,
   * use {@link #frozen()} on the result to create an independent copy
   * if this seems like a problem to you.
   * @return an integer indexable view of this indexable
   */
  @NotNull
  default LongIndexable viewAsLong()
  {
    return new LongIndexable.Base()
    {
      @Override
      public long get(int index)
      {
        return ByteIndexable.this.get(index);
      }

      @Override
      public int size()
      {
        return ByteIndexable.this.size();
      }
    };
  }

  /**
   * View this byte integer indexable as an unsigned long integer indexable.
   * This will assume that the values of this indexable don't have a sign bit
   * and range from 0 to 255.
   * This will cast the values of this indexable on each request,
   * use {@link #frozen()} on the result to create an independent copy
   * if this seems like a problem to you.
   * @return an integer indexable view of this indexable
   */
  @NotNull
  default LongIndexable viewAsUnsignedLong()
  {
    return new LongIndexable.Base()
    {
      @Override
      public long get(int index)
      {
        return ByteIndexable.this.get(index)  & 0xffL;
      }

      @Override
      public int size()
      {
        return ByteIndexable.this.size();
      }
    };
  }

  /**
   * View this byte integer indexable as an integer indexable.
   * This will cast the values of this indexable on each request,
   * use {@link #frozen()} on the result to create an independent copy
   * if this seems like a problem to you.
   * @return an integer indexable view of this indexable
   */
  @NotNull
  default IntIndexable viewAsInt()
  {
    return new IntIndexable.Base()
    {
      @Override
      public int get(int index)
      {
        return ByteIndexable.this.get(index);
      }

      @Override
      public int size()
      {
        return ByteIndexable.this.size();
      }
    };
  }

  /**
   * View this byte integer indexable as an unsigned integer indexable.
   * This will assume that the values of this indexable don't have a sign bit
   * and range from 0 to 255.
   * This will cast the values of this indexable on each request,
   * use {@link #frozen()} on the result to create an independent copy
   * if this seems like a problem to you.
   * @return an integer indexable view of this indexable
   */
  @NotNull
  default IntIndexable viewAsUnsignedInt()
  {
    return new IntIndexable.Base()
    {
      @Override
      public int get(int index)
      {
        return ByteIndexable.this.get(index) & 0xff;
      }

      @Override
      public int size()
      {
        return ByteIndexable.this.size();
      }
    };
  }

  /**
   * View this byte integer indexable as a short integer indexable.
   * This will cast the values of this indexable on each request,
   * use {@link #frozen()} on the result to create an independent copy
   * if this seems like a problem to you.
   * @return a byte integer indexable view of this indexable
   */
  @NotNull
  default ShortIndexable viewAsShort()
  {
    return new ShortIndexable.Base()
    {
      @Override
      public short get(int index)
      {
        return ByteIndexable.this.get(index);
      }

      @Override
      public int size()
      {
        return ByteIndexable.this.size();
      }
    };
  }

  /**
   * View this byte integer indexable as an unsigned short integer indexable.
   * This will assume that the values of this indexable don't have a sign bit
   * and range from 0 to 255.
   * This will cast the values of this indexable on each request,
   * use {@link #frozen()} on the result to create an independent copy
   * if this seems like a problem to you.
   * @return a byte integer indexable view of this indexable
   */
  @NotNull
  default ShortIndexable viewAsUnsignedShort()
  {
    return new ShortIndexable.Base()
    {
      @Override
      public short get(int index)
      {
        return (short)(ByteIndexable.this.get(index) & 0xff);
      }

      @Override
      public int size()
      {
        return ByteIndexable.this.size();
      }
    };
  }

  /**
   * Create a view of this byte indexable with two values swapped.
   * Note that this is not efficient, but handy in certain situations
   * where only a few items are swapped.
   *
   * @param index1 index of the first value (Pythonesque)
   * @param index2 index of the second value (Pythonesque)
   * @return a view of this indexable where the first and second value
   *         have exchanged places
   */
  @NotNull
  default ByteIndexable withSwappedValuesAt(int index1, int index2)
  {
    final int swap1 = Pythonesque.mapX(index1, this);
    final int swap2 = Pythonesque.mapX(index2, this);
    if (swap1 == swap2) {
      return this;
    }
    return new ByteIndexable.Base()
    {
      @Override
      public byte get(int index)
      {
        if (index == swap1) {
          return ByteIndexable.this.get(swap2);
        }
        if (index == swap2) {
          return ByteIndexable.this.get(swap1);
        }
        return ByteIndexable.this.get(index);
      }

      @Override
      public int size()
      {
        return ByteIndexable.this.size();
      }
    };
  }

  /**
   * Get a rotated view of this byte indexable.
   * The returned indexable will show this indexable with rotated indexes.
   * A positive {@code steps} will rotate left, i.e. any index will be accessed
   * as if the number of steps is added before extraction (modulo length).
   * E.g. a rotation of {@code -1} will return the last element when index {@code 0} is
   * requested, and the first element for index {@code 1}.
   *
   * @param steps steps to rotate
   * @return rotated view of this indexable, use  {@link #frozen()}
   *         to create an indexable which no longer depends on this one
   */
  @NotNull
  default ByteIndexable rotated(int steps)
  {
    steps = steps % size();
    if (steps == 0) {
      return this;
    }
    if (steps < 0) {
      steps += size();
    }
    final int rotate = steps;
    return new ByteIndexable.Base()
    {
      @Override
      public byte get(int index)
      {
        return ByteIndexable.this.get((index + rotate) % size());
      }

      @Override
      public int size()
      {
        return ByteIndexable.this.size();
      }

      @NotNull
      @Override
      public ByteIndexable rotated(int steps)
      {
        return ByteIndexable.this.rotated(rotate + steps);
      }
    };
  }

  /**
   * View this indexable as a boolean indexable while using a conversion for
   * each value.
   * <p>
   *   Use {@link #frozen()} on the result to decouple it from this indexable.
   * </p>
   * @param convert converter applied to each value of this indexable before
   *                it is returned from the resulting indexable
   * @return boolean indexable view of this indexable
   */
  @NotNull
  default BooleanIndexable viewAsBoolean(@NotNull BytePredicate1 convert)
  {
    return new BooleanIndexable.Base()
    {
      @Override
      public boolean get(int index)
      {
        return convert.testByte(ByteIndexable.this.get(index));
      }

      @Override
      public int size()
      {
        return ByteIndexable.this.size();
      }
    };
  }

  /**
   * View a list as an Indexable.
   * This requires a list of non-null numbers as a base.
   * @param list list used as a base, required to have only non-null elements
   * @return indexable view of the given list, so any changes of the list will be reflected by the returned Indexable
   */
  @NotNull
  static Base viewList(@NotNull List<? extends Number> list)
  {
    return new Base()
    {
      @Override
      public int size()
      {
        return list.size();
      }

      @Override
      public byte get(int index)
      {
        return list.get(index).byteValue();
      }

      @Override
      public boolean isEmpty()
      {
        return list.isEmpty();
      }
    };
  }

  /**
   * View a list as an Indexable.
   * This allows to provide a fallback for null elements in the list.
   * @param list list used as a base, required to have only non-null elements
   * @param nullValue value returned for {@code null} elements in the list
   * @return indexable view of the given list, so any changes of the list will be reflected by the returned Indexable
   */
  @NotNull
  static Base viewList(@NotNull List<? extends Number> list, byte nullValue)
  {
    return new Base()
    {
      @Override
      public int size()
      {
        return list.size();
      }

      @Override
      public byte get(int index)
      {
        final Number number = list.get(index);
        return number != null
                ? number.byteValue()
                : nullValue;
      }

      @Override
      public boolean isEmpty()
      {
        return list.isEmpty();
      }
    };
  }

  /**
   * View a generic list as a double indexable.
   * @param list generic list
   * @param extractor extractor function which extracts a double from the elements of {@code list}
   * @param <T> element type of list
   * @return a double indexable view of {@code list}
   */
  @NotNull
  static <T> Base viewList(@NotNull List<T> list,
                           @NotNull ToByteFunction<? super T> extractor)
  {
    return new Base()
    {
      @Override
      public byte get(int index)
      {
        return extractor.applyAsByte(list.get(index));
      }

      @Override
      public int size()
      {
        return list.size();
      }

      @Override
      public boolean isEmpty()
      {
        return list.isEmpty();
      }
    };
  }

  /**
   * View a generic Number indexable as byte indexable.
   * @param indexable generic indexable with Number elements, required to have only non-null elements
   * @return byte indexable view
   */
  @NotNull
  static Base viewIndexable(@NotNull Indexable<? extends Number> indexable)
  {
    return new Base()
    {
      @Override
      public int size()
      {
        return indexable.size();
      }

      @Override
      public byte get(int index)
      {
        return indexable.get(index).byteValue();
      }
    };
  }

  /**
   * View a generic Number indexable as byte indexable.
   * @param indexable generic indexable with Number elements
   * @param nullFallback fallback for {@code null} elements
   * @return byte indexable view
   */
  @NotNull
  static Base viewIndexable(@NotNull Indexable<? extends Number> indexable,
                            byte nullFallback)
  {
    return new Base()
    {
      @Override
      public int size()
      {
        return indexable.size();
      }

      @Override
      public byte get(int index)
      {
        final Number number = indexable.get(index);
        return number != null
                ? number.byteValue()
                : nullFallback;
      }
    };
  }

  /**
   * View a generic indexable as a byte indexable.
   * @param indexable generic indexable
   * @param extractor extractor function which extracts a byte from the elements of {@code indexable}
   * @param <T> element type of indexable
   * @return a byte indexable view of {@code indexable}
   */
  @NotNull
  static <T> Base viewIndexable(@NotNull Indexable<T> indexable,
                                @NotNull ToByteFunction<? super T> extractor)
  {
    return new Base()
    {
      @Override
      public byte get(int index)
      {
        return extractor.applyAsByte(indexable.get(index));
      }

      @Override
      public int size()
      {
        return indexable.size();
      }

      @Override
      public boolean isEmpty()
      {
        return indexable.isEmpty();
      }
    };
  }

  /**
   * View a Number array as an Indexable.
   * To view only part of the array use {@link #subSet(int, int)}.
   * @param array array used as base, required to hold only non-null values
   * @return indexable view of the given array, any changes to the underlying array will be reflected by the returned Indexable
   */
  @NotNull
  static Base viewNumberArray(@NotNull Number ... array)
  {
    return new Base()
    {
      @Override
      public int size()
      {
        return array.length;
      }

      @Override
      public byte get(int index)
      {
        return array[index].byteValue();
      }
    };
  }

  /**
   * View a byte array as an Indexable.
   * To view only part of the array use {@link #viewArray(byte[], int, int)}.
   * @param array array used as base
   * @return indexable view of the given array, any changes to the underlying array will be reflected by the returned Indexable
   */
  @NotNull
  static Base viewArray(@NotNull byte ... array)
  {
    return array.length == 0
            ? EMPTY
            : new Base()
    {
      @Override
      public int size()
      {
        return array.length;
      }

      @Override
      public byte get(int index)
      {
        return array[index];
      }
    };
  }

  /**
   * View part of a byte array as an Indexable.
   * @param array array used as base
   * @param start index of first byte to use from the array
   * @param length number of elements to use from the array
   * @return indexable view of the given array, any changes to the underlying array will be reflected by the returned Indexable
   */
  @NotNull
  static Base viewArray(@NotNull byte[] array,
                        int start,
                        int length)
  {
    if (start < 0  ||  length < 0  ||  start + length > array.length) {
      throw new IndexOutOfBoundsException("Start or end outside of range!");
    }
    if (length == 0) {
      return EMPTY;
    }
    return new Base()
    {
      @Override
      public int size()
      {
        return length;
      }

      @Override
      public byte get(int index)
      {
        if (index < 0  ||  index >= length) {
          throw new IndexOutOfBoundsException(String.format("Index %d outside range [0, %d[!",
                                                            index, length));
        }
        return array[index + start];
      }
    };
  }

  /**
   * View an object array as a byte indexable.
   * @param array         viewed array 
   * @param valueExtract  converter from array elements to the byte values of this indexable 
   * @return indexable view of the given array, any changes to the underlying array will be reflected by the returned Indexable 
   * @param <T> array element type
   */
  @NotNull
  static <T> ByteIndexable.Base viewArray(@NotNull T[] array,
                                          @NotNull ToByteFunction<? super T> valueExtract)
  {
    if (array.length == 0) {
      return EMPTY;
    }
    return new ByteIndexable.Base()
    {
      @Override
      public byte get(int index)
      {
        return valueExtract.applyAsByte(array[index]);
      }

      @Override
      public int size()
      {
        return array.length;
      }
    };
  }

  /**
   * View a single byte value as a byte indexable of size 1.
   * @param value single value
   * @return indexable with one value
   */
  @NotNull
  static ByteIndexable.Base singleton(byte value)
  {
    return new ByteIndexable.Base() {
      @Override
      public int size()
      {
        return 1;
      }

      @Override
      public byte get(int index)
      {
        if (index != 0) {
          throw new IndexOutOfBoundsException(String.format("Index %d for indexable of size 1!", index));
        }
        return value;
      }

      @Override
      public boolean isEmpty()
      {
        return false;
      }

      @NotNull
      @Override
      public Base reverse()
      {
        return this;
      }

      @NotNull
      @Override
      public ByteIndexable rotated(int steps)
      {
        return this;
      }

      @NotNull
      @Override
      public Collection<Byte> asCollection()
      {
        return Collections.singleton(value);
      }

      @NotNull
      @Override
      public List<Byte> asList()
      {
        return Collections.singletonList(value);
      }

      @NotNull
      @Override
      public byte[] toArray()
      {
        return new byte[] { value };
      }

      @Override
      public int addToArray(@NotNull byte[] array, int pos)
      {
        array[pos] = value;
        return pos + 1;
      }

      @NotNull
      @Override
      public Iterable<Integer> indexes()
      {
        return IntIndexable.SINGLE_0;
      }

      @NotNull
      @Override
      public IntIndexable intIndexes()
      {
        return IntIndexable.SINGLE_0;
      }
    };
  }

  /**
   * Get a byte indexable of a given size which always returns the same value.
   * @param size  size of the returned indexable
   * @param value value returned for each element
   * @return indexable of size {@code size} with always the same element
   */
  @NotNull
  static ByteIndexable.Base init(int size, byte value)
  {
    if (size == 0) {
      return EMPTY;
    }
    if (size < 0) {
      throw new IndexOutOfBoundsException("Indexables with negative size are impossible: "+size);
    }
    return new ByteIndexable.Base()
    {
      @Override
      public byte get(int index)
      {
        if (index < 0  ||  index >= size) {
          throw new IndexOutOfBoundsException(String.format("Element %d requested from indexable with size %d!",
                                                            index, size));
        }
        return value;
      }

      @Override
      public int size()
      {
        return 0;
      }

      @NotNull
      @Override
      public ByteIndexable frozen()
      {
        return this;
      }
    };
  }

  /**
   * Get a byte indexable of a given size which returns elements created by index.
   * This will always call the {@code producer} when an element is requested.
   * Use {@link #frozen()} on the result to create an indexable which avoids this.
   * @param size size of the returned indexable
   * @param producer producer which is called with an index and expected to return the associated value              
   * @return an indexable with the given size which will return elements by calling the {@code producer}
   */
  @NotNull
  static ByteIndexable.Base viewByIndex(int size, @NotNull IntToByteFunction1 producer)
  {
    if (size == 0) {
      return EMPTY;
    }
    if (size < 0) {
      throw new IndexOutOfBoundsException("Indexables with negative size are impossible: "+size);
    }
    return new ByteIndexable.Base()
    {
      @Override
      public byte get(int index)
      {
        if (index < 0  ||  index >= size) {
          throw new IndexOutOfBoundsException(String.format("Element %d requested from indexable with size %d!",
                                                            index, size));
        }
        return producer.applyAsByte(index);
      }

      @Override
      public int size()
      {
        return size;
      }
    };
  }

  /**
   * Get a byte indexable of a given size which returns elements created by index.
   * This will call the producer only once for each element during the call of this method,
   * and return the results on later requests.
   * @param size size of the returned indexable
   * @param producer producer which is called with an index and expected to return the associated value
   * @return indexable with elements initialized by the {@code producer}
   * @see #viewByIndex(int, IntToByteFunction1)
   */
  @NotNull
  static ByteIndexable initByIndex(int size, @NotNull IntToByteFunction1 producer)
  {
    return viewByIndex(size, producer).frozen();
  }

  /**
   * An empty indexable.
   */
  Base EMPTY = new Base()
  {
    @Override
    public int size()
    {
      return 0;
    }

    @Override
    public byte get(int index)
    {
      throw new IndexOutOfBoundsException("Empty indexable has no elements!");
    }

    @Override
    public boolean isEmpty()
    {
      return true;
    }

    @Override
    public int addToArray(@NotNull byte[] array, int pos)
    {
      return pos;
    }

    @Override
    public int addToArray(@NotNull byte[] array, int arrayPos, int index, int length)
    {
      return arrayPos;
    }

    @NotNull
    @Override
    public byte[] toArray()
    {
      return Empty.BYTE_ARRAY;
    }

    @NotNull
    @Override
    public Iterable<Integer> indexes()
    {
      return IntIndexable.EMPTY;
    }

    @NotNull
    @Override
    public IntIndexable intIndexes()
    {
      return IntIndexable.EMPTY;
    }

    @NotNull
    @Override
    public ByteIndexable.Base reverse()
    {
      return this;
    }

    @NotNull
    @Override
    public ByteIndexable rotated(int steps)
    {
      return this;
    }

    @NotNull
    @Override
    public ByteIndexable frozen()
    {
      return this;
    }

    @NotNull
    @Override
    public Iterator<Byte> iterator()
    {
      return Types.emptyIterator();
    }

    @NotNull
    @Override
    public ListIterator<Byte> listIterator()
    {
      return Types.emptyListIterator();
    }

    @NotNull
    @Override
    public PrimitiveByteIterator byteIterator()
    {
      return Types.EMPTY_BYTE_ITERATOR;
    }

    @Override
    public int compareTo(@NotNull ByteIndexable o)
    {
      return o.isEmpty() ? 0 : -1;
    }

    @Override
    public String toString()
    {
      return Indexable.EMPTY_INDEXABLE_STRING;
    }

    @Override
    public int hashCode()
    {
      return 1;
    }

    @Override
    public boolean equals(Object obj)
    {
      return (obj instanceof ByteIndexable  &&  ((ByteIndexable)obj).isEmpty());
    }

    @NotNull
    @Override
    public Collection<Byte> asCollection()
    {
      return Collections.emptyList();
    }

    @NotNull
    @Override
    public List<Byte> asList()
    {
      return Collections.emptyList();
    }

    @NotNull
    @Override
    public IntIndexable.Base asIntIndexable()
    {
      return IntIndexable.EMPTY;
    }

    @NotNull
    @Override
    public IntIndexable.Base asUnsignedIndexable()
    {
      return IntIndexable.EMPTY;
    }

    @NotNull
    @Override
    public byte[] toByteArray()
    {
      return Empty.BYTE_ARRAY;
    }

    @Override
    public void copyTo(@NotNull byte[] targetArray, int targetStart)
    {
    }

    @Override
    public void copyTo(@NotNull byte[] targetArray, int targetStart, int sourceStart, int length)
    {
    }

    @Override
    public byte foldLeft(byte initialValue, @NotNull ByteOperator2 foldOperation)
    {
      return initialValue;
    }

    @Override
    public void forEachByte(@NotNull ByteConsumer consumer)
    {
    }

    @Override
    public boolean containsByte(byte value)
    {
      return false;
    }

    @Override
    public void forEach(Consumer<? super Byte> action)
    {
    }

    @NotNull
    @Override
    public String toHexString()
    {
      return Empty.STRING;
    }

    @NotNull
    @Override
    public String toHexString(boolean uppercase)
    {
      return Empty.STRING;
    }
  };

  /**
   * Get an empty indexable set.
   * @return empty indexable
   */
  @NotNull
  static Base emptyIndexable()
  {
    return EMPTY;
  }

  /**
   * Create a string representation of the given indexable.
   * @param indexable indexable
   * @return string representation
   */
  @NotNull
  static String toString(@NotNull ByteIndexable indexable)
  {
    if (indexable.isEmpty()) {
      return "[]";
    }
    final StringBuilder sb = new StringBuilder();
    sb.append('[').append(indexable.get(0));
    for (PrimitiveByteIterator it = indexable.tailSet(1).byteIterator();  it.hasNext();  ) {
      sb.append(',').append(it.nextByte());
    }
    sb.append(']');
    return sb.toString();
  }

  /**
   * Are two byte indexables equal?
   *
   * @param indexable1 first indexable
   * @param indexable2 second indexable
   * @return {@code true} if both indexables contain the same values in the same sequence<br>
   *         {@code false} if sizes or values differ
   */
  static boolean equal(@NotNull ByteIndexable indexable1,
                       @NotNull ByteIndexable indexable2)
  {
    if (indexable1 == indexable2) {
      return true;
    }
    if (indexable1.size() != indexable2.size()) {
      return false;
    }
    final PrimitiveByteIterator it1 = indexable1.byteIterator();
    final PrimitiveByteIterator it2 = indexable2.byteIterator();
    while (it1.hasNext()  &&  it2.hasNext()) {
      if (it1.nextByte() != it2.nextByte()) {
        return false;
      }
    }
    return !(it1.hasNext()  ||  it2.hasNext());
  }

  /**
   * Are two byte indexables equal?
   * This is a convenience method which may be called from
   * implementations for their {@link Object#equals(Object)}
   * method.
   * @param indexable1  first indexable
   * @param indexable2  object expected to be an indexable itself
   * @return {@code true} if both indexables contain the same values in the same sequence<br>
   *         {@code false} if the second object is not an indexable, of if sizes or values differ
   */
  static boolean equal(@NotNull ByteIndexable indexable1,
                       @Nullable Object indexable2)
  {
    if (indexable2 instanceof ByteIndexable) {
      return equal(indexable1, (ByteIndexable)indexable2);
    }
    return false;
  }

  /**
   * Compare two byte indexables.
   * This compares the two indexable lexically element by element
   * until either the first difference is found (smaller of the two
   * values define the lower indexable) or the end of one is met
   * (shorter indexable is less).
   *
   * @param indexable1 first indexable
   * @param indexable2 second indexable
   * @return {@code < 0>} if {@code indexable1 < indxable2},
   *         {@code 0} if {@code indexable1 == indexable2}, or
   *         {@code > 0} if {@code indexable1 > indexable2}
   */
  static int compare(@NotNull ByteIndexable indexable1,
                     @NotNull ByteIndexable indexable2)
  {
    final PrimitiveByteIterator it1 = indexable1.byteIterator();
    final PrimitiveByteIterator it2 = indexable2.byteIterator();
    while (it1.hasNext()  &&  it2.hasNext()) {
      final int compare = Byte.compare(it1.nextByte(),
                                        it2.nextByte());
      if (compare != 0) {
        return compare;
      }
    }
    if (it1.hasNext()) {
      return 1;
    }
    if (it2.hasNext()) {
      return -1;
    }
    return 0;
  }

  /**
   * Compare two byte indexables as if they contain unsigned byte values.
   * This compares the two indexable lexically element by element
   * until either the first difference is found (smaller of the two
   * values define the lower indexable) or the end of one is met
   * (shorter indexable is less).
   *
   * @param indexable1 first indexable
   * @param indexable2 second indexable
   * @return {@code < 0>} if {@code indexable1 < indxable2},
   *         {@code 0} if {@code indexable1 == indexable2}, or
   *         {@code > 0} if {@code indexable1 > indexable2}
   */
  static int compareUnsigned(@NotNull ByteIndexable indexable1,
                             @NotNull ByteIndexable indexable2)
  {
    final PrimitiveByteIterator it1 = indexable1.byteIterator();
    final PrimitiveByteIterator it2 = indexable2.byteIterator();
    while (it1.hasNext()  &&  it2.hasNext()) {
      final int compare = Primitives.compareUnsigned(it1.nextByte(),
                                                     it2.nextByte());
      if (compare != 0) {
        return compare;
      }
    }
    if (it1.hasNext()) {
      return 1;
    }
    if (it2.hasNext()) {
      return -1;
    }
    return 0;
  }

  /**
   * Calculate a hashcode for a byte indexable.
   * @param indexable indexable for which the hash code is required
   * @return hash code for the given indexable
   */
  static int hash(@NotNull ByteIndexable indexable)
  {
    // this follows Arrays.hashCode()
    int result = 1;

    for (PrimitiveByteIterator it = indexable.byteIterator();  it.hasNext();  ) {
      result = 31 * result + Byte.hashCode(it.next());
    }

    return result;
  }

  /**
   * Wrap an indexable with one which caches the hash value.
   * This is useful if indexables are used as hash keys because hash value calculation
   * is expensive.
   * <p>
   * The wrapped indexable must not change after it is wrapped, otherwise strange things are expected to happen.
   * @param indexable wrapped indexable, must not change
   * @return indexable which forwards most methods to {@code indexable}, but also provides
   *         useful implementations for {@link Object#hashCode()}, {@link Object#equals(Object)},
   *         and {@link Object#toString()}
   */
  @NotNull
  static ByteIndexable.Base withCachedHash(@NotNull ByteIndexable indexable)
  {
    final int hashCode = hash(indexable);
    return new ByteIndexable.Base()
    {
      @Override
      public int size()
      {
        return indexable.size();
      }

      @Override
      public byte get(int index)
      {
        return indexable.get(index);
      }

      @NotNull
      @Override
      public PrimitiveByteIterator byteIterator()
      {
        return indexable.byteIterator();
      }

      @NotNull
      @Override
      public PrimitiveByteIterator byteIterator(int from, int to)
      {
        return indexable.byteIterator(from, to);
      }

      @NotNull
      @Override
      public Iterator<Byte> iterator()
      {
        return indexable.iterator();
      }

      @NotNull
      @Override
      public ListIterator<Byte> listIterator()
      {
        return indexable.listIterator();
      }

      @NotNull
      @Override
      public Base subSet(int fromIndex, int toIndex)
      {
        return indexable.subSet(fromIndex, toIndex);
      }

      @Override
      public int addToArray(@NotNull byte[] array, int pos)
      {
        return indexable.addToArray(array, pos);
      }

      @Override
      public int addToArray(@NotNull byte[] array, int arrayPos, int index, int length)
      {
        return indexable.addToArray(array, arrayPos, index, length);
      }

      @Override
      public int hashCode()
      {
        return hashCode;
      }

      @Override
      public boolean equals(Object obj)
      {
        return equal(indexable, obj);
      }

      @Override
      public String toString()
      {
        return ByteIndexable.toString(indexable);
      }
    };
  }

  /**
   * Get a byte indexable from a string of hexcoded bytes.
   * This converts a string like {@code "cafebabe"} to
   * a byte indexable with the 4 bytes 0xCA, 0xFE, 0xBA, and 0cBE.
   * @param hexString string with hexadecimal digits, 2 per byte
   * @return byte indexable
   * @throws IllegalArgumentException if the string has odd size
   * @throws NumberFormatException if the string contains characters which doe't appear in hexcadecimal digits
   * @see #bytesFromHexString(String)
   */
  @NotNull
  static ByteIndexable fromHexString(@NotNull String hexString)
  {
    return viewArray(bytesFromHexString(hexString));
  }

  /**
   * Get a byte array from a string of hexcoded bytes.
   * This converts a string like {@code "cafebabe"} to
   * a byte array with the 4 bytes 0xCA, 0xFE, 0xBA, and 0cBE.
   * @param hexString string with hexadecimal digits, 2 per byte
   * @return byte array
   * @throws IllegalArgumentException if the string has odd size
   * @throws NumberFormatException if the string contains characters which doe't appear in hexcadecimal digits
   */
  @NotNull
  static byte[] bytesFromHexString(@NotNull String hexString)
  {
    if (hexString.isEmpty()) {
      return Empty.BYTE_ARRAY;
    }
    if (hexString.length() % 2 == 1) {
      throw new IllegalArgumentException("String has to contain an even number of characters!");
    }
    final byte[] result = new byte[hexString.length() / 2];
    for (int b = 0;  b < result.length;  ++b) {
      result[b] = (byte)Integer.parseInt(hexString.substring(2 * b, 2 * b + 2), 16);
    }
    return result;
  }

  /**
   * Get the next index for which the given check is fulfilled.
   * @param startIndex start index for checking, {@link Pythonesque} indexing supported
   * @param check      check to perform on elements of this indexable until {@code true}
   * @return next index starting with {@code startIndex} for which the check returns {@code true},
   *         {@link Pythonesque#UNMAPPABLE} if none
   */
  default int nextMatch(int startIndex, @NotNull BytePredicate1 check)
  {
    final int size = size();
    for (int i = Pythonesque.mapLB(startIndex, this);  i < size;  ++i) {
      if (check.testByte(get(i))) {
        return i;
      }
    }
    return Pythonesque.UNMAPPABLE;
  }

  /**
   * Get the first index for which the given check is fulfilled.
   * @param check check to perform on elements of this indexable until {@code true}
   * @return first index for which the check returns {@code true},
   *         {@code -1} if none
   */
  default int firstMatch(@NotNull BytePredicate1 check)
  {
    return nextMatch(0, check);
  }

  /**
   * Get the previous index for which the given check is fulfilled.
   * @param startIndex start index for checking, {@link Pythonesque} indexing supported
   * @param check      check to perform on elements of this indexable until {@code true}
   * @return next index starting with {@code startIndex} for which the check returns {@code true},
   *         {@link Pythonesque#UNMAPPABLE} if none
   */
  default int previousMatch(int startIndex, @NotNull BytePredicate1 check)
  {
    for (int i = Pythonesque.mapUB(startIndex, this);  i >= 0;  --i) {
      if (check.testByte(get(i))) {
        return i;
      }
    }
    return Pythonesque.UNMAPPABLE;
  }

  /**
   * Get the last index for which the given check is fulfilled.
   * @param check check to perform on elements of this indexable until {@code true}
   * @return last index for which the check returns {@code true},
   *         {@code -1} if none
   */
  default int lastMatch(@NotNull BytePredicate1 check)
  {
    return previousMatch(-1, check);
  }

  /**
   * Get a frozen version of this indexable.
   * <p>
   * Often Indexables are used as a view to underlying collections.
   * Although this interface is immutable, the underlying colelction might
   * nevertheless change. This copies the current state of this indexable
   * into an unmodifiable state, and returns an Indexable which is
   * stable in size and will return always the same element for a given index.
   * Beware: it is still possible that any element itself changes when the
   * elements are mutable.
   * <p>
   * Calling {@code frozen()} again on the returned object will just return
   * the object itself, so you can safely call this method more than once.
   *
   * @return frozen version of this Indexable
   */
  @NotNull
  default ByteIndexable frozen()
  {
    return IndexableHelper.frozenFromArray(toArray());
  }

  /**
   * Is this indexable sorted according to the given ordering?
   * This method tests that consecutive elements in this indexable
   * or either ascending or the same, but never descending.
   * <p>
   * See {@linkplain #isStrictlyOrdered(ByteOrdering)} for a stricter alternative.
   * <p>
   * Prefer {@linkplain #isOrdered()} to check for natural ordering because it is faster.
   *
   * @param order expected ordering
   * @return {@code true} if this indexable is sorted as defined by {@code order}<br>
   *         {@code false} if not
   */
  default boolean isOrdered(@NotNull ByteOrdering order)
  {
    if (size() < 2) {
      return true;
    }
    byte last = gyt(-1);
    for (int i = size() - 2;  i >= 0;  --i) {
      final byte value = get(i);
      if (order.checkByte(value, last) == Descending) {
        return false;
      }
      last = value;
    }
    return true;
  }

  /**
   * Is this indexable sorted naturally?
   * This method tests that consecutive elements in this indexable
   * or either ascending or the same, but never descending.
   * <p>
   * See {@linkplain #isStrictlyOrdered()} for a stricter alternative.
   * @return {@code true} if this indexable is sorted naturally<br>
   *         {@code false} if not
   */
  default boolean isOrdered()
  {
    if (size() < 2) {
      return true;
    }
    int last = gyt(-1);
    for (int i = size() - 2;  i >= 0;  --i) {
      final int value = get(i);
      if (value < last) {
        return false;
      }
      last = value;
    }
    return true;
  }

  /**
   * Is this indexable strictly sorted according to the given ordering?
   * This method tests that consecutive elements in this indexable
   * or strictly ascending or the same, but never descending.
   * See {@linkplain #isOrdered(ByteOrdering)} for a more relaxed alternative.
   * @param order expected ordering
   * @return {@code true} if this indexable is sorted as defined by {@code order}<br>
   *         {@code false} if not
   */
  default boolean isStrictlyOrdered(@NotNull ByteOrdering order)
  {
    if (size() < 2) {
      return true;
    }
    byte last = gyt(-1);
    for (int i = size() - 2;  i >= 0;  --i) {
      final byte value = get(i);
      if (order.checkByte(value, last) != Ascending) {
        return false;
      }
      last = value;
    }
    return true;
  }

  /**
   * Is this indexable strictly sorted naturally?
   * This method tests that consecutive elements in this indexable
   * or strictly ascending or the same, but never descending.
   * See {@linkplain #isOrdered()} for a more relaxed alternative.
   * @return {@code true} if this indexable is sorted naturally<br>
   *         {@code false} if not
   */
  default boolean isStrictlyOrdered()
  {
    if (size() < 2) {
      return true;
    }
    int last = gyt(-1);
    for (int i = size() - 2;  i >= 0;  --i) {
      final int value = get(i);
      if (value >= last) {
        return false;
      }
      last = value;
    }
    return true;
  }

  /**
   * Do a binary search in an indexable ordered in natural ascending order.
   * This requires this indexable to be ordered in
   * {@link ByteOrdering#ASCENDING natural ascending order},
   * i.e. {@linkplain #isOrdered(ByteOrdering)} has to return {@code true}
   * for this ordering.
   * For efficiency this prerequisite is not checked, but not fulfilling
   * it will make this method return bogus results. If this indexable is only ordered,
   * but not {@link #isStrictlyOrdered(ByteOrdering) strictly ordered} it is not defined
   * which index is returned when the looked up value appears in a sequence of equal values.
   *
   * @param value value to look up
   * @return a positive integer defining the index where the given value was found,
   *         and a negative integer if the value is not contained. The latter defines
   *         the insertion point where the looked up value would match into this
   *         sorted indexable in the form {@code -(insertIndex + 1)}.
   */
  default int binarySearch(byte value)
  {
    final int iValue = value; // use int as bytecode cannot compare bytes
    int low = 0;
    int hi  = size() - 1;

    while (low <= hi) {
      final int mid = (low + hi) >>> 1;
      final int dp = get(mid);

      if (dp < iValue) {
        low = mid + 1;
      }
      else if (dp > iValue) {
        hi = mid - 1;
      }
      else {
        return mid;
      }
    }
    return -(low + 1);
  }

  /**
   * Do a binary search in an ordered indexable.
   * This requires this indexable to be ordered in non-descending order
   * as defined by the given {@code order},
   * i.e. {@linkplain #isOrdered(ByteOrdering)} has to return {@code true}
   * for this order.
   * For efficiency this prerequisite is not checked, but not fulfilling
   * it will make this method return bogus results. If this indexable is only ordered,
   * but not {@link #isStrictlyOrdered(ByteOrdering) strictly ordered} it is not defined
   * which index is returned when the looked up value appears in a sequence of equal values.
   *
   * @param value value to look up
   * @param order sort order
   * @return a positive integer defining the index where the given value was found,
   * and a negative integer if the value is not contained. The latter defines
   * the insertion point where the looked up value would match into this
   * sorted indexable in the form {@code -(insertIndex + 1)}.
   */
  default int binarySearch(byte value, @NotNull ByteOrdering order)
  {
    int low = 0;
    int hi  = size() - 1;

    while (low <= hi) {
      final int mid = (low + hi) >>> 1;
      final byte dp = get(mid);

      switch (order.checkByte(dp, value)) {
      case Ascending:
        low = mid + 1;
        break;
      case Descending:
        low = mid - 1;
        break;
      default:
        return mid;
      }
    }
    return -(low + 1);
  }

  /**
   * Create a (mutable) byte indexable from the values of this indexable
   * which is ordered as defined by the given ordering.
   * @param order sort order
   * @return independent int indexable with sorted values
   */
  @NotNull
  default MutableByteIndexable ordered(@NotNull ByteOrdering order)
  {
    final MutableByteIndexable result = MutableByteIndexable.fromByteIndexable(this);
    result.order(order);
    return result;
  }

  /**
   * Make this byte indexable usable as a class with defined standard Object methods.
   * @return based version of this indexable
   */
  @NotNull
  default Base asBase()
  {
    return new ByteIndexable.Base() {
      @Override
      public int size()
      {
        return ByteIndexable.this.size();
      }

      @Override
      public byte get(int index)
      {
        return ByteIndexable.this.get(index);
      }

      @NotNull
      @Override
      public Base reverse()
      {
        return ByteIndexable.this.reverse();
      }

      @NotNull
      @Override
      public PrimitiveIterator.OfInt intIterator()
      {
        return ByteIndexable.this.intIterator();
      }

      @NotNull
      @Override
      public PrimitiveByteIterator byteIterator()
      {
        return ByteIndexable.this.byteIterator();
      }

      @NotNull
      @Override
      public PrimitiveByteIterator byteIterator(int from, int to)
      {
        return ByteIndexable.this.byteIterator(from ,to);
      }

      @NotNull
      @Override
      public Iterator<Byte> iterator()
      {
        return ByteIndexable.this.iterator();
      }

      @NotNull
      @Override
      public ListIterator<Byte> listIterator()
      {
        return ByteIndexable.this.listIterator();
      }

      @NotNull
      @Override
      public Base subSet(int fromIndex, int toIndex)
      {
        return ByteIndexable.this.subSet(fromIndex, toIndex);
      }

      @NotNull
      @Override
      public byte[] toArray()
      {
        return ByteIndexable.this.toArray();
      }

      @Override
      public int addToArray(@NotNull byte[] array, int pos)
      {
        return ByteIndexable.this.addToArray(array, pos);
      }

      @Override
      public int addToArray(@NotNull byte[] array, int arrayPos, int index, int length)
      {
        return ByteIndexable.this.addToArray(array, arrayPos, index, length);
      }

      @NotNull
      @Override
      public IntIndexable.Base asIntIndexable()
      {
        return ByteIndexable.this.asIntIndexable();
      }

      @NotNull
      @Override
      public IntIndexable.Base asUnsignedIndexable()
      {
        return ByteIndexable.this.asUnsignedIndexable();
      }
    };
  }

  /**
   * Abstract base class which provides useful implementations
   * for {@link Object#equals(Object)}, {@link Object#hashCode()},
   * {@link Object#toString()}. It also provides
   * {@link Comparable#compareTo(Object)}.
   * @see ByteIndexable#asBase()
   */
  abstract class Base implements ByteIndexable,
                                 Comparable<ByteIndexable>
  {
    @Override
    public int compareTo(@NotNull ByteIndexable o)
    {
      Objects.requireNonNull(o);
      return compare(this, o);
    }

    @Override
    public int hashCode()
    {
      return hash(this);
    }

    @Override
    public boolean equals(Object obj)
    {
      return equal(this, obj);
    }

    @Override
    public String toString()
    {
      return ByteIndexable.toString(this);
    }

    @NotNull
    @Override
    public Base asBase()
    {
      return this;
    }
  }
}
