// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.function.*;
import de.caff.generics.range.Range;

import java.util.*;
import java.util.function.Predicate;

/**
 * Class which allows readonly access of raw booleans by index.
 * <p>
 * The {@link #EMPTY} constant or {@link #emptyIndexable()} method provide the same
 * useful return value to indicate emptiness.
 * <p>
 * Simple implementations should extend {@link Base} because
 * that provides useful implementations for standard Object methods and implements
 * {@link Comparable}.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since July 7, 2020
 */
public interface BooleanIndexable
        extends PrimitiveBooleanIterable,
                Sizeable
{
  /** Boolean indexable with a single {@code false} value. */
  BooleanIndexable SINGLE_FALSE = singleton(false);
  /** Boolean indexable with a single {@code false} value. */
  BooleanIndexable SINGLE_TRUE = singleton(true);

  /**
   * Get the element at the given index.
   * @param index index between {@code 0} and {@code size() - 1}
   * @return element at the given index
   */
  boolean get(int index);

  /**
   * Pythonesque get.
   * This allows accessing elements from the back by using negative indexes,
   * e.g. {@code -1} references the last element, {@code -2} its predecessor, and so on.
   * @param index index between {@code -size()} and {@code size() - 1}
   * @return element at the given index
   */
  default boolean gyt(int index)
  {
    return get(Pythonesque.mapX(index, this));
  }

  /**
   * Get an element modulo size.
   * <p>
   * This maps the given {@code index} into the range of this indexable
   * by applying a modulo {@link #size()} operation. For empty indexable
   * this will throw an {@link IndexOutOfBoundsException} as there is no
   * possible index to get.
   * @param index index, possibly out of range, possibly even negative
   *              for Pythonesque access
   * @return element at the given, possibly modulated, index
   */
  default boolean getMod(int index)
  {
    final int size = size();
    if (size == 0) {
      throw new IndexOutOfBoundsException("No element for empty indexable!");
    }
    return gyt(index % size);
  }

  /**
   * Get this indexable but with inverted order.
   * @return a view to this indexable which accesses the elements in reverse order
   */
  @NotNull
  default Base reverse()
  {
    return new Base()
    {
      @Override
      public int size()
      {
        return BooleanIndexable.this.size();
      }

      @Override
      public boolean get(int index)
      {
        return BooleanIndexable.this.get(BooleanIndexable.this.size() - index - 1);
      }

      @NotNull
      @Override
      public Base reverse()
      {
        return BooleanIndexable.this.asBase();
      }
    };
  }

  /**
   * Is this indexable empty?
   * @return {@code true}: there are no elements in this indexable<br>
   *         {@code false}: this indexable has elements
   */
  default boolean isEmpty()
  {
    return size() == 0;
  }

  /**
   * Get an primitive boolean iterator.
   * @return {@code boolean} iterator which iterates over this indexable
   */
  @NotNull
  default PrimitiveBooleanIterator booleanIterator()
  {
    return booleanIterator(0, size());
  }

  /**
   * Get a boolean iterator from the given sub set.
   * @param from first index of iterator
   * @param to   one past last index of iterator
   * @return int iterator which iterates over the given part of this indexable
   */
  @NotNull
  default PrimitiveBooleanIterator booleanIterator(final int from, final int to)
  {
    return new PrimitiveBooleanIterator()
    {
      private int index = from;

      @Override
      public boolean nextBool()
      {
        if (index >= to) {
          throw new NoSuchElementException(String.format("Index %s out of allowed range [%d, %d[!",
                                                         index, from, to));
        }
        return get(index++);
      }

      @Override
      public boolean hasNext()
      {
        return index < to;
      }
    };
  }

  /**
   * Returns an iterator over elements of type {@code T}.
   *
   * @return an Iterator.
   */
  @NotNull
  @Override
  default Iterator<Boolean> iterator()
  {
    return listIterator();
  }

  /**
   * Returns a list iterator over elements of type {@code T}.
   *
   * @return a list iterator.
   */
  @NotNull
  default ListIterator<Boolean> listIterator()
  {
    return new ListIterator<Boolean>() {
      private int index = 0;

      @Override
      public boolean hasNext()
      {
        return index < size();
      }

      @Override
      public Boolean next()
      {
        if (index >= size()) {
          throw new NoSuchElementException("index: "+index);
        }
        return get(index++);
      }

      @Override
      public boolean hasPrevious()
      {
        return index > 0;
      }

      @Override
      public Boolean previous()
      {
        if (index == 0) {
          throw new NoSuchElementException("index: -1");
        }
        return get(--index);
      }

      @Override
      public int nextIndex()
      {
        return index;
      }

      @Override
      public int previousIndex()
      {
        return index - 1;
      }

      @Override
      public void remove()
      {
        throw new UnsupportedOperationException();
      }

      @Override
      public void set(Boolean t)
      {
        throw new UnsupportedOperationException();
      }

      @Override
      public void add(Boolean t)
      {
        throw new UnsupportedOperationException();
      }
    };
  }

  /**
   * Get an indexable subset.
   * The subset includes indices {@code fromIndex} to {@code toIndex - 1}.
   * @param fromIndex start index of sub set
   * @param toIndex   index after last index
   * @return indxable subset view of this indexable
   */
  @NotNull
  default Base subSet(int fromIndex, int toIndex)
  {
    if (fromIndex < 0)
      throw new IndexOutOfBoundsException("fromIndex = " + fromIndex);
    if (toIndex > size())
      throw new IndexOutOfBoundsException("toIndex = " + toIndex);
    if (fromIndex > toIndex)
      throw new IllegalArgumentException("fromIndex(" + fromIndex +
                                         ") > toIndex(" + toIndex + ")");
    final int length = toIndex - fromIndex;
    return length == 0
            ? EMPTY
            : new Base()
    {
      @Override
      public int size()
      {
        return length;
      }

      @Override
      public boolean get(int index)
      {
        if (index < 0  ||  index >= length) {
          throw new IndexOutOfBoundsException("No such element: "+index);
        }
        return BooleanIndexable.this.get(index + fromIndex);
      }

      @NotNull
      @Override
      public Base subSet(int fromIdx, int toIdx)
      {
        if (fromIdx < 0) {
          throw new IndexOutOfBoundsException("fromIndex = " + fromIdx);
        }
        if (toIdx > length) {
          throw new IndexOutOfBoundsException("toIndex = " + toIdx);
        }
        if (fromIdx > toIdx) {
          throw new IllegalArgumentException("fromIndex(" + fromIdx +
                                             ") > toIndex(" + toIdx + ")");
        }
        return BooleanIndexable.this.subSet(fromIndex + fromIdx,
                                            fromIndex + toIdx);
      }
    };
  }

  /**
   * Get an indexable subset.
   * This is the Pythonesque version which allows negative indexes.
   * @param fromIndex start index of sub set
   * @param toIndex   end index of sub set
   * @return indexable subset view of this indexable set
   */
  @NotNull
  default Base sybSet(int fromIndex, int toIndex)
  {
    return subSet(Pythonesque.mapX(fromIndex, this),
                  Pythonesque.mapX(toIndex, this));
  }

  /**
   * Create an indexable subset from the last elements of this indexable set.
   * This already allows Pythonesquew indexing.
   * @param fromIndex index to start with, negative counts from the back
   * @return indexable subset view of this indexable set
   */
  @NotNull
  default Base tailSet(int fromIndex)
  {
    return subSet(Pythonesque.mapX(fromIndex, this),
                  size());
  }

  /**
   * Create an indexable subset from the last elements of this indexable set.
   * This already allows Pythonesquew indexing.
   * @param toIndex index one after the end (equal to the length of the returned set),
   *                negative counts from the back
   * @return indexable subset view of this indexable set
   */
  @NotNull
  default Base headSet(int toIndex)
  {
    return subSet(0,
                  Pythonesque.mapX(toIndex, this));
  }

  /**
   * Return a mapped view on this boolean indexable.
   * @param mapper mapper
   * @param <T> result type pf mapper
   * @return mapped indexable
   */
  @NotNull
  default <T> Indexable<T> view(@NotNull BooleanFunction1<? extends T> mapper)
  {
    return new Indexable.Base<T>()
    {
      @Override
      public int size()
      {
        return BooleanIndexable.this.size();
      }

      @Override
      public T get(int index)
      {
        return mapper.applyAsBool(BooleanIndexable.this.get(index));
      }
    };
  }

  /**
   * Get a view on this indexable as an unmodifiable collection.
   * This is especially useful for conversion to standard collection classes, eg for adding all
   * elements via {@link Collection#addAll(Collection)}.
   * @return collection view
   */
  @NotNull
  default Collection<Boolean> asCollection()
  {
    return new AbstractCollection<Boolean>()
    {
      @NotNull
      @Override
      public Iterator<Boolean> iterator()
      {
        return BooleanIndexable.this.iterator();
      }

      @Override
      public int size()
      {
        return BooleanIndexable.this.size();
      }
    };
  }

  /**
   * Get a view on this indexable as an unmodifiable list.
   * This is especially useful for conversion to standard collection classes, eg for adding all
   * elements via {@link Collection#addAll(Collection)}.
   * @return collection view
   * @see #toList()
   */
  @NotNull
  default List<Boolean> asList()
  {
    return new AbstractList<Boolean>()
    {
      @Override
      public Boolean get(int index)
      {
        return BooleanIndexable.this.get(index);
      }

      @Override
      public int size()
      {
        return BooleanIndexable.this.size();
      }

      @NotNull
      @Override
      public Iterator<Boolean> iterator()
      {
        return BooleanIndexable.this.iterator();
      }
    };
  }

  /**
   * Create a list from this boolean indexable.
   * This creates an independent list to which the elements of this
   * indexable are copied.
   * @return list copy of this indexable
   * @see #asList()
   */
  @NotNull
  default ArrayList<Boolean> toList()
  {
    final ArrayList<Boolean> list = new ArrayList<>(size());
    addAllTo(list);
    return list;
  }

  /**
   * Add all elements of this indexable to the given collection.
   * @param collection collection to add to
   */
  default void addAllTo(@NotNull Collection<? super Boolean> collection)
  {
    for (Boolean entry : this) {
      collection.add(entry);
    }
  }

  /**
   * Get a view of this indexable as a standard object-based indexable.
   * @return indexable with non-null {@code Boolean} values
   */
  @NotNull
  default Indexable<Boolean> asIndexable()
  {
    return new Indexable.Base<Boolean>()
    {
      @Override
      public Boolean get(int index)
      {
        return BooleanIndexable.this.get(index);
      }

      @Override
      public int size()
      {
        return BooleanIndexable.this.size();
      }
    };
  }

  /**
   * Convert this indexable into a Java array.
   * @return array copy containing all values of this indexable
   */
  @NotNull
  default boolean[] toArray()
  {
    if (isEmpty()) {
      return Empty.BOOLEAN_ARRAY;
    }
    final boolean[] array = new boolean[size()];
    addToArray(array, 0);
    return array;
  }

  /**
   * Add the content of this indexable to the given array.
   * @param array array where the content is added
   * @param pos   position in the array where the content of this indexable is added
   * @return array position after this indexable
   */
  default int addToArray(@NotNull boolean[] array, int pos)
  {
    return addToArray(array, pos, 0, size());
  }

  /**
   * Add a part of the content of this indexable to the given array.
   * @param array     array where the content is added
   * @param arrayPos  position in hte array where the content is added
   * @param index     start index of this indexable which is added first
   * @param length    number of entries of this indexable added to the array
   * @return array position after the added content
   */
  default int addToArray(@NotNull boolean[] array, int arrayPos,
                         int index, int length)
  {
    for (PrimitiveBooleanIterator it = booleanIterator(index, index + length);  it.hasNext();  ) {
      array[arrayPos++] = it.nextBool();
    }
    return arrayPos;
  }

  /**
   * Get the indexes of this indexable as an iterable range.
   * @return iterable indexes
   * @see #intIndexes()
   * @see Range#indexes(int)
   */
  @NotNull
  default Iterable<Integer> indexes()
  {
    return Range.indexes(size());
  }

  /**
   * Get the indexes of this indexable as an integer indexable.
   * @return indexes as integer indexable
   * @see #indexes()
   * @see IntIndexable#rangeFromSize(int)
   */
  @NotNull
  default IntIndexable intIndexes()
  {
    return IntIndexable.rangeFromSize(size());
  }

  /**
   * Go over all values and do a cumulative calculation.
   *
   * @param initialValue  start value for the operation, will be returned if this iterable is empty
   * @param foldOperation operation applied to each value, will get the accumulated value as its
   *                      first and the current element value as its second argument
   * @return accumulated value, result of applying fold operation to all values of this indexable
   *         iteratively
   */
  default boolean foldLeft(boolean initialValue,
                           @NotNull BooleanOperator2 foldOperation)
  {
    boolean result = initialValue;
    for (PrimitiveBooleanIterator iterator = booleanIterator();
         iterator.hasNext();  ) {
      result = foldOperation.applyAsBoolean(result, iterator.nextBool());
    }
    return result;
  }

  /**
   * Create a view of this boolean indexable with an inserted value.
   * Note that this is not efficient, but handy in certain situations
   * where only 1 or a few values are inserted.
   * <p>
   * As this creates a view of this indexable, changes to this indexable might 
   * result in a disaster when using the returned indexable!
   *
   * @param index index where the value is inserted before the current index
   *              (<b>not Pythonesque</b>, because it is possible to add an value at {@code index == size()},
   *              and it is expected that most insertion will happen at 0 or at the end)
   * @param value inserted value at that index
   * @return a view of this indexable with 1 more value at the given index,
   *         all values beyond are moved one index forward
   * @see #withAppendedValue(boolean)
   */
  @NotNull
  default BooleanIndexable withInsertedValueAt(int index, boolean value)
  {
    final int insertIndex = index;
    final int newSize = size() + 1;
    if (insertIndex == 0) {
      return new BooleanIndexable.Base() {
        @Override
        public int size()
        {
          return newSize;
        }

        @Override
        public boolean get(int index)
        {
          return index == 0
                  ? value
                  : BooleanIndexable.this.get(index - 1);
        }
      };
    }
    if (insertIndex == newSize - 1) {
      return new BooleanIndexable.Base() {

        @Override
        public int size()
        {
          return newSize;
        }

        @Override
        public boolean get(int index)
        {
          return index == newSize - 1
                  ? value
                  : BooleanIndexable.this.get(index);
        }
      };
    }
    return new BooleanIndexable.Base()
    {
      @Override
      public int size()
      {
        return newSize;
      }

      @Override
      public boolean get(int index)
      {
        if (index == insertIndex) {
          return value;
        }
        return BooleanIndexable.this.get(index < insertIndex
                                                ? index
                                                : index - 1);
      }
    };
  }

  /**
   * Create a view with of this indexable with another value added at the end.
   * Note that this is not efficient, but handy in certain situations
   * where only 1 or a few items are added.
   * @param value value to add
   * @return indexable view of this indexable, with the given value added at the end,
   *         so size is larger by 1
   * @see #withInsertedValueAt(int, boolean)
   */
  @NotNull
  default BooleanIndexable withAppendedValue(boolean value)
  {
    return withInsertedValueAt(size(), value);
  }

  /**
   * Create a view of this boolean indexable with an exchanged value.
   * Note that this is not efficient, but handy in certain situations
   * where only 1 or a few values are exchanged.
   * <p>
   * As this creates a view of this indexable, changes to this indexable could 
   * lead to unexpected results when using the returned indexable!
   *
   * @param index index where the element is exchanged, hiding the current element.
   *              (Pythonesque)
   * @param value exchanged value at that index
   * @return a view of this indexable with 1 more element at the given index,
   *         all elements beyond are moved one index forward
   */
  @NotNull
  default BooleanIndexable withExchangedValueAt(int index, boolean value)
  {
    final int insertIndex = Pythonesque.mapX(index, this);
    return new BooleanIndexable.Base()
    {
      @Override
      public boolean get(int index)
      {
        return index == insertIndex
                ? value
                : BooleanIndexable.this.get(index);
      }

      @Override
      public int size()
      {
        return BooleanIndexable.this.size();
      }
    };
  }

  /**
   * Create a view of this indexable with one value removed.
   * Note that this is not efficient, but handy in certain situations
   * where only one or a few values are removed.
   *
   * @param index index of the removed value (Pythonesque)
   * @return a view of this indexable where the value at the given index
   *         is removed
   */
  @NotNull
  default BooleanIndexable withRemovedValueAt(int index)
  {
    final int removeIndex = Pythonesque.mapX(index, this);
    if (removeIndex == 0) {
      // remove first
      return tailSet(1);
    }
    final int sz = size() - 1;
    if (removeIndex == sz) {
      // remove last
      return headSet(-1);
    }
    return new BooleanIndexable.Base()
    {
      @Override
      public boolean get(int index)
      {
        return index < removeIndex
                ? BooleanIndexable.this.get(index)
                : BooleanIndexable.this.get(index + 1);
      }

      @Override
      public int size()
      {
        return sz;
      }
    };
  }

  /**
   * Create a view of this boolean indexable with two values swapped.
   * Note that this is not efficient, but handy in certain situations
   * where only a few items are swapped.
   *
   * @param index1 index of the first value (Pythonesque)
   * @param index2 index of the second value (Pythonesque)
   * @return a view of this indexable where the first and second value
   *         have exchanged places
   */
  @NotNull
  default BooleanIndexable withSwappedValuesAt(int index1, int index2)
  {
    final int swap1 = Pythonesque.mapX(index1, this);
    final int swap2 = Pythonesque.mapX(index2, this);
    if (swap1 == swap2) {
      return this;
    }
    return new BooleanIndexable.Base()
    {
      @Override
      public boolean get(int index)
      {
        if (index == swap1) {
          return BooleanIndexable.this.get(swap2);
        }
        if (index == swap2) {
          return BooleanIndexable.this.get(swap1);
        }
        return BooleanIndexable.this.get(index);
      }

      @Override
      public int size()
      {
        return BooleanIndexable.this.size();
      }
    };
  }

  /**
   * Get a rotated view of this boolean indexable.
   * The returned indexable will show this indexable with rotated indexes.
   * A positive {@code steps} will rotate left, i.e. any index will be accessed
   * as if the number of steps is added before extraction (modulo length).
   * E.g. a rotation of {@code -1} will return the last element when index {@code 0} is
   * requested, and the first element for index {@code 1}.
   *
   * @param steps steps to rotate
   * @return rotated view of this indexable, use  {@link #frozen()}
   *         to create an indexable which no longer depends on this one
   */
  @NotNull
  default BooleanIndexable rotated(int steps)
  {
    steps = steps % size();
    if (steps == 0) {
      return this;
    }
    if (steps < 0) {
      steps += size();
    }
    final int rotate = steps;
    return new BooleanIndexable.Base()
    {
      @Override
      public boolean get(int index)
      {
        return BooleanIndexable.this.get((index + rotate) % size());
      }

      @Override
      public int size()
      {
        return BooleanIndexable.this.size();
      }

      @NotNull
      @Override
      public BooleanIndexable rotated(int steps)
      {
        return BooleanIndexable.this.rotated(rotate + steps);
      }
    };
  }

  /**
   * View a list as an Indexable.
   * This requires a list of non-null numbers as a base.
   * @param list list used as a base, required to have only non-null elements
   * @return indexable view of the given list, so any changes of the list will be reflected by the returned Indexable
   */
  @NotNull
  static Base viewList(@NotNull List<? extends Boolean> list)
  {
    if (list.isEmpty()) {
      return EMPTY;
    }
    return new Base()
    {
      @Override
      public int size()
      {
        return list.size();
      }

      @Override
      public boolean get(int index)
      {
        return list.get(index);
      }

      @Override
      public boolean isEmpty()
      {
        return list.isEmpty();
      }
    };
  }

  /**
   * View a list as an Indexable.
   * This allows to provide a fallback for null elements in the list.
   * @param list list used as a base, required to have only non-null elements
   * @param nullFallback value returned for {@code null} elements in the list
   * @return indexable view of the given list, so any changes of the list will be reflected by the returned Indexable
   */
  @NotNull
  static Base viewList(@NotNull List<? extends Boolean> list, boolean nullFallback)
  {
    if (list.isEmpty()) {
      return EMPTY;
    }
    return new Base()
    {
      @Override
      public int size()
      {
        return list.size();
      }

      @Override
      public boolean get(int index)
      {
        final Boolean value = list.get(index);
        return value != null
                ? value
                : nullFallback;
      }

      @Override
      public boolean isEmpty()
      {
        return list.isEmpty();
      }
    };
  }

  /**
   * View a generic list as a boolean indexable.
   * @param list generic list
   * @param extractor extractor function which extracts a boolean from the elements of {@code list}
   * @param <T> element type of list
   * @return a boolean indexable view of {@code list}
   */
  @NotNull
  static <T> Base viewList(@NotNull List<T> list,
                           @NotNull Predicate<? super T> extractor)
  {
    if (list.isEmpty()) {
      return EMPTY;
    }
    return new Base()
    {
      @Override
      public boolean get(int index)
      {
        return extractor.test(list.get(index));
      }

      @Override
      public int size()
      {
        return list.size();
      }

      @Override
      public boolean isEmpty()
      {
        return list.isEmpty();
      }
    };
  }

  /**
   * View a generic Number indexable as integer indexable.
   * @param indexable generic indexable with Number elements, required to have only non-null elements
   * @return integer indexable view
   */
  @NotNull
  static Base viewIndexable(@NotNull Indexable<? extends Boolean> indexable)
  {
    if (indexable.isEmpty()) {
      return EMPTY;
    }
    return new Base()
    {
      @Override
      public int size()
      {
        return indexable.size();
      }

      @Override
      public boolean get(int index)
      {
        return indexable.get(index);
      }
    };
  }

  /**
   * View a generic Number indexable as integer indexable.
   * @param indexable generic indexable with Number elements
   * @param nullFallback fallback for {@code null} elements
   * @return integer indexable view
   */
  @NotNull
  static Base viewIndexable(@NotNull Indexable<? extends Boolean> indexable,
                            boolean nullFallback)
  {
    if (indexable.isEmpty()) {
      return EMPTY;
    }
    return new Base()
    {
      @Override
      public int size()
      {
        return indexable.size();
      }

      @Override
      public boolean get(int index)
      {
        final Boolean value = indexable.get(index);
        return value != null
                ? value
                : nullFallback;
      }
    };
  }

  /**
   * View a generic indexable as a boolean indexable.
   * @param indexable generic indexable
   * @param extractor extractor function which extracts a boolean from the elements of {@code indexable}
   * @param <T> element type of indexable
   * @return a boolean indexable view of {@code indexable}
   */
  @NotNull
  static <T> Base viewIndexable(@NotNull Indexable<T> indexable,
                                @NotNull Predicate<? super T> extractor)
  {
    if (indexable.isEmpty()) {
      return EMPTY;
    }
    return new Base()
    {
      @Override
      public boolean get(int index)
      {
        return extractor.test(indexable.get(index));
      }

      @Override
      public int size()
      {
        return indexable.size();
      }

      @Override
      public boolean isEmpty()
      {
        return indexable.isEmpty();
      }
    };
  }

  /**
   * View an int array as an Indexable.
   * To view only part of the array use {@link #viewArray(boolean[], int, int)}.
   * @param array array used as base
   * @return indexable view of the given array, any changes to the underlying array will be reflected by the returned Indexable
   */
  @NotNull
  static Base viewArray(@NotNull boolean ... array)
  {
    if (array.length == 0) {
      return EMPTY;
    }
    return new Base()
    {
      @Override
      public int size()
      {
        return array.length;
      }

      @Override
      public boolean get(int index)
      {
        return array[index];
      }
    };
  }

  /**
   * View part of an integer array as an Indexable.
   * @param array array used as base
   * @param start start index into the array
   * @param length number of booleans use from the array
   * @return indexable view of the given array, any changes to the underlying array will be reflected by the returned Indexable
   */
  @NotNull
  static Base viewArray(@NotNull boolean[] array,
                        int start,
                        int length)
  {
    if (start < 0  ||  length < 0  ||  start + length > array.length) {
      throw new IndexOutOfBoundsException("Start or end outside of range!");
    }
    if (length == 0) {
      return EMPTY;
    }
    return new Base()
    {
      @Override
      public int size()
      {
        return length;
      }

      @Override
      public boolean get(int index)
      {
        if (index < 0  ||  index >= length) {
          throw new IndexOutOfBoundsException(String.format("Index %d outside range [0, %d[!",
                                                            index, length));
        }
        return array[index + start];
      }
    };
  }

  /**
   * View an object array as a boolean indexable.
   * @param array         viewed array 
   * @param valueExtract  converter from array elements to the boolean values of this indexable 
   * @return indexable view of the given array, any changes to the underlying array will be reflected by the returned Indexable 
   * @param <T> array element type
   */
  @NotNull
  static <T> BooleanIndexable.Base viewArray(@NotNull T[] array,
                                             @NotNull Predicate<? super T> valueExtract)
  {
    if (array.length == 0) {
      return EMPTY;
    }
    return new BooleanIndexable.Base()
    {
      @Override
      public boolean get(int index)
      {
        return valueExtract.test(array[index]);
      }

      @Override
      public int size()
      {
        return array.length;
      }
    };
  }

  /**
   * View a single boolean value as a boolean indexable of size 1.
   * @param value single value
   * @return indexable with one value
   */
  @NotNull
  static BooleanIndexable.Base singleton(boolean value)
  {
    return new BooleanIndexable.Base() {
      @Override
      public int size()
      {
        return 1;
      }

      @Override
      public boolean get(int index)
      {
        if (index != 0) {
          throw new IndexOutOfBoundsException(String.format("Index %d for indexable of size 1!", index));
        }
        return value;
      }

      @Override
      public boolean isEmpty()
      {
        return false;
      }

      @NotNull
      @Override
      public Base reverse()
      {
        return this;
      }

      @NotNull
      @Override
      public BooleanIndexable rotated(int steps)
      {
        return this;
      }

      @NotNull
      @Override
      public Collection<Boolean> asCollection()
      {
        return Collections.singleton(value);
      }

      @NotNull
      @Override
      public List<Boolean> asList()
      {
        return Collections.singletonList(value);
      }

      @NotNull
      @Override
      public boolean[] toArray()
      {
        return new boolean[] { value };
      }

      @Override
      public int addToArray(@NotNull boolean[] array, int pos)
      {
        array[pos] = value;
        return pos + 1;
      }

      @NotNull
      @Override
      public Iterable<Integer> indexes()
      {
        return IntIndexable.SINGLE_0;
      }

      @NotNull
      @Override
      public IntIndexable intIndexes()
      {
        return IntIndexable.SINGLE_0;
      }
    };
  }

  /**
   * Get a boolean indexable of a given size which always returns the same value.
   * @param size  size of the returned indexable
   * @param value value returned for each element
   * @return indexable of size {@code size} with always the same element
   */
  @NotNull
  static BooleanIndexable.Base init(int size, boolean value)
  {
    if (size == 0) {
      return EMPTY;
    }
    if (size < 0) {
      throw new IndexOutOfBoundsException("Indexables with negative size are impossible: "+size);
    }
    return new BooleanIndexable.Base()
    {
      @Override
      public boolean get(int index)
      {
        if (index < 0  ||  index >= size) {
          throw new IndexOutOfBoundsException(String.format("Element %d requested from indexable with size %d!",
                                                            index, size));
        }
        return value;
      }

      @Override
      public int size()
      {
        return 0;
      }

      @NotNull
      @Override
      public BooleanIndexable frozen()
      {
        return this;
      }
    };
  }

  /**
   * Get a boolean indexable of a given size which returns elements created by index.
   * This will always call the {@code producer} when an element is requested.
   * Use {@link #frozen()} on the result to create an indexable which avoids this.
   * @param size size of the returned indexable
   * @param producer producer which is called with an index and expected to return the associated value
   * @return an indexable with the given size which will return elements by calling the {@code producer}
   * @see #initByIndex(int, IntPredicate1)
   */
  @NotNull
  static BooleanIndexable.Base viewByIndex(int size, @NotNull IntPredicate1 producer)
  {
    if (size == 0) {
      return EMPTY;
    }
    if (size < 0) {
      throw new IndexOutOfBoundsException("Indexables with negative size are impossible: "+size);
    }
    return new BooleanIndexable.Base()
    {
      @Override
      public boolean get(int index)
      {
        if (index < 0  ||  index >= size) {
          throw new IndexOutOfBoundsException(String.format("Element %d requested from indexable with size %d!",
                                                            index, size));
        }
        return producer.test(index);
      }

      @Override
      public int size()
      {
        return size;
      }
    };
  }

  /**
   * Get a boolean indexable of a given size which returns elements created by index.
   * This will call the producer only once for each element during the call of this method,
   * and return the results on later requests.
   * @param size size of the returned indexable
   * @param producer producer which is called with an index and expected to return the associated value
   * @return indexable with elements initialized by the {@code producer}
   * @see #viewByIndex(int, IntPredicate1)
   */
  @NotNull
  static BooleanIndexable initByIndex(int size, @NotNull IntPredicate1 producer)
  {
    return viewByIndex(size, producer).frozen();
  }

  /**
   * An empty indexable.
   */
  @NotNull
  Base EMPTY = new Base()
  {
    @Override
    public int size()
    {
      return 0;
    }

    @Override
    public boolean get(int index)
    {
      throw new IndexOutOfBoundsException("Empty indexable has no elements!");
    }

    @Override
    public boolean isEmpty()
    {
      return true;
    }

    @Override
    public int addToArray(@NotNull boolean[] array, int pos)
    {
      return pos;
    }

    @Override
    public int addToArray(@NotNull boolean[] array, int arrayPos, int index, int length)
    {
      return arrayPos;
    }

    @NotNull
    @Override
    public boolean[] toArray()
    {
      return Empty.BOOLEAN_ARRAY;
    }

    @NotNull
    @Override
    public Iterable<Integer> indexes()
    {
      return IntIndexable.EMPTY;
    }

    @NotNull
    @Override
    public IntIndexable intIndexes()
    {
      return IntIndexable.EMPTY;
    }

    @Override
    public boolean foldLeft(boolean initialValue, @NotNull BooleanOperator2 foldOperation)
    {
      return initialValue;
    }

    @NotNull
    @Override
    public Base reverse()
    {
      return this;
    }

    @NotNull
    @Override
    public BooleanIndexable rotated(int steps)
    {
      return this;
    }

    @NotNull
    @Override
    public BooleanIndexable.Base frozen()
    {
      return this;
    }

    @NotNull
    @Override
    public Iterator<Boolean> iterator()
    {
      return Types.emptyIterator();
    }

    @NotNull
    @Override
    public ListIterator<Boolean> listIterator()
    {
      return Types.emptyListIterator();
    }

    @NotNull
    @Override
    public PrimitiveBooleanIterator booleanIterator()
    {
      return Types.EMPTY_BOOLEAN_ITERATOR;
    }

    @Override
    public void forEachBool(@NotNull BooleanConsumer action)
    {
    }

    @Override
    public boolean containsBool(boolean value)
    {
      return false;
    }

    @Override
    public boolean containsTrue()
    {
      return false;
    }

    @Override
    public boolean containsFalse()
    {
      return false;
    }

    @Override
    public boolean or()
    {
      return false;
    }

    @Override
    public boolean and()
    {
      return false;
    }

    @Override
    public boolean xor()
    {
      return false;
    }

    @Override
    public int compareTo(@NotNull BooleanIndexable o)
    {
      return o.isEmpty() ? 0 : -1;
    }

    @Override
    public String toString()
    {
      return Indexable.EMPTY_INDEXABLE_STRING;
    }

    @Override
    public int hashCode()
    {
      return 1;
    }

    @Override
    public boolean equals(Object obj)
    {
      return (obj instanceof BooleanIndexable  &&  ((BooleanIndexable)obj).isEmpty());
    }
  };

  /**
   * Get an empty indexable set.
   * @return empty indexable
   */
  @NotNull
  static Base emptyIndexable()
  {
    return EMPTY;
  }

  /**
   * Create a string representation of the given indexable.
   * @param indexable indexable
   * @return string representation
   */
  @NotNull
  static String toString(@NotNull BooleanIndexable indexable)
  {
    if (indexable.isEmpty()) {
      return Indexable.EMPTY_INDEXABLE_STRING;
    }
    final StringBuilder sb = new StringBuilder();
    sb.append('[').append(indexable.get(0));
    for (PrimitiveBooleanIterator it = indexable.tailSet(1).booleanIterator();  it.hasNext();  ) {
      sb.append(',').append(it.nextBool());
    }
    sb.append(']');
    return sb.toString();
  }

  /**
   * Are two boolean indexables equal?
   *
   * @param indexable1 first indexable
   * @param indexable2 second indexable
   * @return {@code true} if both indexables contain the same values in the same sequence<br>
   *         {@code false} if sizes or values differ
   */
  static boolean equal(@NotNull BooleanIndexable indexable1,
                       @NotNull BooleanIndexable indexable2)
  {
    if (indexable1 == indexable2) {
      return true;
    }
    if (indexable1.size() != indexable2.size()) {
      return false;
    }
    final PrimitiveBooleanIterator it1 = indexable1.booleanIterator();
    final PrimitiveBooleanIterator it2 = indexable2.booleanIterator();
    while (it1.hasNext()  &&  it2.hasNext()) {
      if (it1.nextBool() != it2.nextBool()) {
        return false;
      }
    }
    return !(it1.hasNext()  ||  it2.hasNext());
  }

  /**
   * Are two boolean indexables equal?
   * This is a convenience method which may be called from
   * implementations for their {@link Object#equals(Object)}
   * method.
   * @param indexable1  first indexable
   * @param indexable2  object expected to be an indexable itself
   * @return {@code true} if both indexables contain the same values in the same sequence<br>
   *         {@code false} if the second object is not an indexable, of if sizes or values differ
   */
  static boolean equal(@NotNull BooleanIndexable indexable1,
                       @Nullable Object indexable2)
  {
    if (indexable2 instanceof BooleanIndexable) {
      return equal(indexable1, (BooleanIndexable)indexable2);
    }
    return false;
  }

  /**
   * Compare two boolean indexables.
   * This compares the two indexable lexically element by element
   * until either the first difference is found (smaller of the two
   * values define the lower indexable) or the end of one is met
   * (shorter indexable is less).
   *
   * @param indexable1 first indexable
   * @param indexable2 second indexable
   * @return {@code < 0>} if {@code indexable1 < indxable2},
   *         {@code 0} if {@code indexable1 == indexable2}, or
   *         {@code > 0} if {@code indexable1 > indexable2}
   */
  static int compare(@NotNull BooleanIndexable indexable1,
                     @NotNull BooleanIndexable indexable2)
  {
    final PrimitiveBooleanIterator it1 = indexable1.booleanIterator();
    final PrimitiveBooleanIterator it2 = indexable2.booleanIterator();
    while (it1.hasNext()  &&  it2.hasNext()) {
      final int compare = Boolean.compare(it1.nextBool(),
                                          it2.nextBool());
      if (compare != 0) {
        return compare;
      }
    }
    if (it1.hasNext()) {
      return 1;
    }
    if (it2.hasNext()) {
      return -1;
    }
    return 0;
  }

  /**
   * Calculate a hashcode for a boolean indexable.
   * @param indexable indexable for which the hash code is required
   * @return hash code for the given indexable
   */
  static int hash(@NotNull BooleanIndexable indexable)
  {
    // this follows Arrays.hashCode()
    int result = 1;

    for (PrimitiveBooleanIterator it = indexable.booleanIterator();  it.hasNext();  ) {
      result = 31 * result + Boolean.hashCode(it.next());
    }

    return result;
  }

  /**
   * Wrap an indexable with one which caches the hash value.
   * This is useful if indexables are used as hash keys because hash value calculation
   * is expensive.
   * <p>
   * The wrapped indexable must not change after it is wrapped, otherwise strange things are expected to happen.
   * @param indexable wrapped indexable, must not change
   * @return indexable which forwards most methods to {@code indexable}, but also provides
   *         useful implementations for {@link Object#hashCode()}, {@link Object#equals(Object)},
   *         and {@link Object#toString()}
   */
  @NotNull
  static BooleanIndexable.Base withCachedHash(@NotNull BooleanIndexable indexable)
  {
    final int hashCode = hash(indexable);
    return new BooleanIndexable.Base()
    {
      @Override
      public int size()
      {
        return indexable.size();
      }

      @Override
      public boolean get(int index)
      {
        return indexable.get(index);
      }

      @NotNull
      @Override
      public PrimitiveBooleanIterator booleanIterator()
      {
        return indexable.booleanIterator();
      }

      @NotNull
      @Override
      public PrimitiveBooleanIterator booleanIterator(int from, int to)
      {
        return indexable.booleanIterator(from, to);
      }

      @NotNull
      @Override
      public Iterator<Boolean> iterator()
      {
        return indexable.iterator();
      }

      @NotNull
      @Override
      public ListIterator<Boolean> listIterator()
      {
        return indexable.listIterator();
      }

      @NotNull
      @Override
      public Base subSet(int fromIndex, int toIndex)
      {
        return indexable.subSet(fromIndex, toIndex);
      }

      @Override
      public int addToArray(@NotNull boolean[] array, int pos)
      {
        return indexable.addToArray(array, pos);
      }

      @Override
      public int addToArray(@NotNull boolean[] array, int arrayPos, int index, int length)
      {
        return indexable.addToArray(array, arrayPos, index, length);
      }

      @Override
      public int hashCode()
      {
        return hashCode;
      }

      @Override
      public boolean equals(Object obj)
      {
        return equal(indexable, obj);
      }

      @Override
      public String toString()
      {
        return BooleanIndexable.toString(indexable);
      }
    };
  }

  /**
   * Get the next index for which the given check is fulfilled.
   * @param startIndex start index for checking, {@link Pythonesque} indexing supported
   * @param check      check to perform on elements of this indexable until {@code true}
   * @return next index starting with {@code startIndex} for which the check returns {@code true},
   *         {@link Pythonesque#UNMAPPABLE} if none
   */
  default int nextMatch(int startIndex, @NotNull BooleanPredicate1 check)
  {
    final int size = size();
    for (int i = Pythonesque.mapLB(startIndex, this);  i < size;  ++i) {
      if (check.testBool(get(i))) {
        return i;
      }
    }
    return Pythonesque.UNMAPPABLE;
  }

  /**
   * Get the first index for which the given check is fulfilled.
   * @param check check to perform on elements of this indexable until {@code true}
   * @return first index for which the check returns {@code true},
   *         {@code -1} if none
   */
  default int firstMatch(@NotNull BooleanPredicate1 check)
  {
    return nextMatch(0, check);
  }

  /**
   * Get the previous index for which the given check is fulfilled.
   * @param startIndex start index for checking, {@link Pythonesque} indexing supported
   * @param check      check to perform on elements of this indexable until {@code true}
   * @return next index starting with {@code startIndex} for which the check returns {@code true},
   *         {@link Pythonesque#UNMAPPABLE} if none
   */
  default int previousMatch(int startIndex, @NotNull BooleanPredicate1 check)
  {
    for (int i = Pythonesque.mapUB(startIndex, this);  i >= 0;  --i) {
      if (check.testBool(get(i))) {
        return i;
      }
    }
    return Pythonesque.UNMAPPABLE;
  }

  /**
   * Get the last index for which the given check is fulfilled.
   * @param check check to perform on elements of this indexable until {@code true}
   * @return last index for which the check returns {@code true},
   *         {@code -1} if none
   */
  default int lastMatch(@NotNull BooleanPredicate1 check)
  {
    return previousMatch(-1, check);
  }

  /**
   * Get a frozen version of this indexable.
   * <p>
   * Often Indexables are used as a view to underlying collections.
   * Although this interface is immutable, the underlying colelction might
   * nevertheless change. This copies the current state of this indexable
   * into an unmodifiable state, and returns an Indexable which is
   * stable in size and will return always the same element for a given index.
   * Beware: it is still possible that any element itself changes when the
   * elements are mutable.
   * <p>
   * Calling {@code frozen()} again on the returned object will just return
   * the object itself, so you can safely call this method more than once.
   *
   * @return frozen version of this Indexable
   */
  @NotNull
  default BooleanIndexable frozen()
  {
    return IndexableHelper.frozenFromArray(toArray());
  }

  /**
   * Call an entry consumer foreach entry in this indexable.
   * @param consumer consumer to be called
   */
  default void forEachBoolEntry(@NotNull EntryConsumer consumer)
  {
    for (int idx = 0;  idx < size();  ++idx) {
      consumer.accept(idx, get(idx));
    }
  }

  /**
   * Make this boolean indexable usable as a class with defined standard Object methods.
   * @return based version of this indexable
   */
  @NotNull
  default Base asBase()
  {
    return new BooleanIndexable.Base() {
      @Override
      public int size()
      {
        return BooleanIndexable.this.size();
      }

      @Override
      public boolean get(int index)
      {
        return BooleanIndexable.this.get(index);
      }

      @NotNull
      @Override
      public Base reverse()
      {
        return BooleanIndexable.this.reverse();
      }

      @NotNull
      @Override
      public PrimitiveBooleanIterator booleanIterator()
      {
        return BooleanIndexable.this.booleanIterator();
      }

      @NotNull
      @Override
      public PrimitiveBooleanIterator booleanIterator(int from, int to)
      {
        return BooleanIndexable.this.booleanIterator(from ,to);
      }

      @NotNull
      @Override
      public Iterator<Boolean> iterator()
      {
        return BooleanIndexable.this.iterator();
      }

      @NotNull
      @Override
      public ListIterator<Boolean> listIterator()
      {
        return BooleanIndexable.this.listIterator();
      }

      @NotNull
      @Override
      public Base subSet(int fromIndex, int toIndex)
      {
        return BooleanIndexable.this.subSet(fromIndex, toIndex);
      }

      @NotNull
      @Override
      public boolean[] toArray()
      {
        return BooleanIndexable.this.toArray();
      }

      @Override
      public int addToArray(@NotNull boolean[] array, int pos)
      {
        return BooleanIndexable.this.addToArray(array, pos);
      }

      @Override
      public int addToArray(@NotNull boolean[] array, int arrayPos, int index, int length)
      {
        return BooleanIndexable.this.addToArray(array, arrayPos, index, length);
      }
    };
  }

  /**
   * Abstract base class which provides useful implementations
   * for {@link Object#equals(Object)}, {@link Object#hashCode()},
   * {@link Object#toString()}. It also provides
   * {@link Comparable#compareTo(Object)}.
   * @see BooleanIndexable#asBase()
   */
  abstract class Base implements BooleanIndexable,
                                 Comparable<BooleanIndexable>
  {
    @Override
    public int compareTo(@NotNull BooleanIndexable o)
    {
      Objects.requireNonNull(o);
      return compare(this, o);
    }

    @Override
    public int hashCode()
    {
      return hash(this);
    }

    @Override
    public boolean equals(Object obj)
    {
      return equal(this, obj);
    }

    @Override
    public String toString()
    {
      return BooleanIndexable.toString(this);
    }

    @NotNull
    @Override
    public Base asBase()
    {
      return this;
    }
  }


  /**
   * Consumer for index-value pairs with {@code int} values.
   */
  @FunctionalInterface
  interface EntryConsumer
  {
    /**
     * Do whatever is necessary for the given index-value combination.
     * @param index index of the given {@code value}
     * @param value value at the given {@code index}
     */
    void accept(int index, boolean value);
  }
}
