// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2020-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.function.Function3;

import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.function.*;

/**
 * Wrapper around a double linked list which forms a loop.
 * <p>
 * This basic implementation allows to use user-defined double linked items
 * based on {@link BasicLoop.Item} because it makes some algorithms easier
 * to add behavior to the items themselves. For loop with items holding
 * generic values see {@link Loop}.
 * <p>
 * This class is not thread-safe.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 02, 2022
 */
public class BasicLoop<T extends BasicLoop.Item<T>>
        implements Countable<T>
{
  /**
   * Base implementation of a double linked loop item.
   * @param <V> item type
   */
  public abstract static class Item<V extends BasicLoop.Item<V>>
  {
    /**
     * Loop containing this item.
     * May be {@code null} if this item is removed.
     */
    @Nullable
    BasicLoop<V> loop;
    /** Previous item in the loop (possibly {@code this}). */
    @NotNull
    V previous;
    /** Next item in the loop (possibly {@code this}). */
    @NotNull
    V next;

    /**
     * Standard constructor..
     * @param loop     loop to which this item belongs
     * @param previous previous item
     * @param next     next item
     */
    @SuppressWarnings("unchecked")  // implementation should take care that this is always a V
    protected Item(@NotNull BasicLoop<V> loop,
                   @Nullable V previous,
                   @Nullable V next)
    {
      this.loop = loop;
      if (previous != null) {
        assert next != null;
        this.previous = previous;

        this.next = next;

        assert previous.loop == loop;
        assert next.loop == loop;

        previous.next = (V)this;
        next.previous = (V)this;

      }
      else {
        assert next == null;
        assert loop.size == 0;

        this.previous = (V)this;
        this.next = (V)this;
      }

      ++loop.size;
    }

    /**
     * Constructor for first item.
     * @param loop loop to which this item belongs
     */
    protected Item(@NotNull BasicLoop<V> loop)
    {
      this(loop, null, null);
    }

    /**
     * Get the previous item.
     * @return previous item
     */
    @NotNull
    public V getPrevious()
    {
      return previous;
    }

    /**
     * Get the next item.
     * @return next item
     */
    @NotNull
    public V getNext()
    {
      return next;
    }

    /**
     * Advance to the item {@code steps} steps away.
     * @param steps steps to advance, forward if positive, backward if negative
     * @return item which is the given number of steps away
     */
    @SuppressWarnings("unchecked")  // implementation should take care that this is always a T
    public V advanced(int steps)
    {
      if (loop != null) {
        steps = steps % loop.size;
      }
      if (steps == 0) {
        return (V)this;
      }
      V run = (V)this;
      if (steps < 0) {
        while (++steps <= 0) {
          run = run.previous;
        }
      }
      else {
        while (--steps >= 0) {
          run = run.next;
        }
      }
      return run;
    }

    /**
     * Get the number of steps from this item to another item.
     * @param otherItem another item which has to be in the same loop as this item
     * @return number of forward steps to the given item, {@code 0} or positive
     * @see #minStepsTo(BasicLoop.Item)
     */
    public int stepsTo(@NotNull V otherItem)
    {
      if (otherItem == this) {
        return 0;
      }
      if (loop == null) {
        throw new IllegalStateException("Trying to find the steps to an item from an item outside of loops!");
      }
      if (otherItem.loop != loop) {
        throw new IllegalArgumentException("Trying to find the steps to an item in another loop!");
      }
      int count = 1;
      for (V item = next;  item != this;  item = item.next) {
        if (item == otherItem) {
          return count;
        }
        ++count;
      }
      throw new IllegalStateException("No match in loop!");
    }

    /**
     * Get the number of steps from this item to another item.
     * @param otherItem other item which has to be in the same loop as this item
     * @return minimal number of steps to get from this to the given item, positive for forward
     *         iteration, negative for backward iteration
     */
    public int minStepsTo(@NotNull V otherItem)
    {
      final int steps = stepsTo(otherItem);
      assert loop != null;  // checked in stepsTo()
      return (steps > loop.size / 2)
              ? steps - loop.size
              : steps;
    }

    /**
     * Get the loop to which this item belongs.
     * @return loop of this item, {@code null} if this item is removed from its loop
     */
    @Nullable
    public BasicLoop<V> getLoop()
    {
      return loop;
    }

    /**
     * Does this item still belong to a loop.
     * @return {@code true} if it belongs to a loop<br>
     *         {@code false} if it is already removed
     */
    public boolean isValid()
    {
      return loop != null;
    }

    /**
     * Insert an item before this one.
     * @param creator item creator
     * @return inserted item
     */
    @NotNull
    @SuppressWarnings("unchecked")  // implementation should take care that this is always a V
    public V insertBefore(final Function3<? extends V, BasicLoop<V>, V, V> creator)
    {
      if (loop == null) {
        throw new IllegalStateException("Cannot insert before a removed item!");
      }
      return creator.apply(loop, previous, (V)this);
    }

    /**
     * Insert an item after this one.
     * @param creator item creator
     * @return inserted item
     */
    @NotNull
    @SuppressWarnings("unchecked")  // implementation should take care that this is always a V
    public V insertAfter(final Function3<? extends V, BasicLoop<V>, V, V> creator)
    {
      if (loop == null) {
        throw new IllegalStateException("Cannot insert before a removed item!");
      }
      return creator.apply(loop, (V)this, next);
    }

    /**
     * Exchange the loop position of this item with the one of the given item
     * and vice versa.
     * <p>
     * Note that this method also allows to exchange items between different loops
     * and even removed items.
     * @param item item which will exchange its chaining properties with this one
     */
    @SuppressWarnings("unchecked")
    public void swapPlaces(@NotNull V item)
    {
      // this is surprisingly complex
      if (item == this) {
        return;
      }
      final BasicLoop<V> thisLoop = loop;
      final BasicLoop<V> thatLoop = item.loop;
      final V thisPrev = previous;
      final V thisNext = next;
      final V thatPrev = item.previous;
      final V thatNext = item.next;
      if (thisLoop == null) {
        if (thatLoop == null) {
          // Both are outside loops.
          return;
        }
        else {
          if (thatLoop.size == 1) {
            thatLoop.first = (V)this;
            this.previous = this.next = (V)this;
          }
          else {
            thatPrev.next = thatNext.previous = (V)this;
            this.previous = thatPrev;
            this.next     = thatNext;
            if (thatLoop.first == item) {
              thatLoop.first = (V)this;
            }
          }
          this.loop = thatLoop;
          item.loop = null;
        }
      }
      else {
        if (thatLoop == null) {
          if (thisLoop.size == 1) {
            thisLoop.first = item;
            item.previous = item.next = item;
          }
          else {
            thisPrev.next = thisNext.previous = item;
            item.previous = thisPrev;
            item.next     = thisNext;
            if (thisLoop.first == this) {
              thisLoop.first = item;
            }
          }
          item.loop = thisLoop;
          this.loop = null;
        }
        else {
          if (thisLoop == thatLoop) {
            // both belong to the same loop, maybe they are neighbors
            if (thisPrev == item) {
              assert thatNext == this;
              if (thisLoop.size == 2) {
                thisLoop.rotate(1);
                return;
              }
              thisNext.previous = item;
              item.next = thisNext;

              item.previous = (V)this;
              next = item;

              thatPrev.next = (V)this;
              previous = thatPrev;
            }
            else if (thisNext == item) {
              assert thatPrev == this;
              if (thisLoop.size == 2) {
                thisLoop.rotate(1);
                return;
              }
              thisPrev.next = item;
              item.previous = thisPrev;

              item.next = (V)this;
              previous  = item;

              thatNext.previous = (V)this;
              next = thatNext;
            }
            else {
              thisPrev.next = item;
              item.previous = thisPrev;

              thisNext.previous = item;
              item.next     = thisNext;

              thatPrev.next = (V)this;
              previous = thatPrev;

              thatNext.previous = (V)this;
              next     = thatNext;
            }
            if (thisLoop.first == this) {
              thisLoop.first = item;
            }
            else if (thatLoop.first == item) {
              thatLoop.first = (V)this;
            }
          }
          else {
            thisPrev.next = item;
            item.previous = thisPrev;

            thisNext.previous = item;
            item.next     = thisNext;

            item.loop     = thisLoop;

            thatPrev.next = (V)this;
            previous = thatPrev;

            thatNext.previous = (V)this;
            next     = thatNext;

            loop     = thatLoop;

            if (thisLoop.first == this) {
              thisLoop.first = item;
            }
            if (thatLoop.first == item) {
              thatLoop.first = (V)this;
            }
          }
        }
      }
    }

    /**
     * Exchange this item with a new item.
     * <p>
     * Note that this item is removed and {@link #isValid() invalid} after the call.
     * @param creator creator for new item
     * @return newly created item
     */
    @SuppressWarnings("unchecked")  // implementation should take care that this is always a V
    public V exchangeWith(@NotNull Function3<? extends V, BasicLoop<V>, V, V> creator)
    {
      final V newItem = creator.apply(loop, (V)this, next);
      remove();
      return newItem;
    }

    /**
     * Remove this item from the loop.
     */
    public void remove()
    {
      if (loop == null) {
        throw new IllegalStateException("Must not remove an item more than once!");
      }
      --loop.size;
      assert loop.size >= 0;
      previous.next = next;
      next.previous = previous;
      if (loop.first == this) {
        if (loop.size == 0) {
          loop.first = null;
        }
        else {
          loop.first = next;
        }
      }
      loop = null;  // mark as removed
    }

    /**
     * Find the next item for which its value fulfills the given condition, starting with this item.
     * This searches forward through the loop until the condition is met, or the loop
     * is finished.
     * @param condition condition to be checked
     * @return item found or {@code null} if no item matches the condition
     */
    @Nullable
    @SuppressWarnings("unchecked")  // implementation should take care that this is always a V
    public V findNext(@NotNull Predicate<? super V> condition)
    {
      if (condition.test((V)this)) {
        return (V)this;
      }
      for (V run = next; run != this; run = run.next) {
        if (condition.test(run)) {
          return run;
        }
      }
      return null;
    }

   /**
     * Find the previous item for which its value fulfills the given condition, starting with this item.
     * This searches backward through the loop until the condition is met, or the loop
     * is finished.
     * @param condition condition to be checked
     * @return item found or {@code null} if no item matches the condition
     */
    @Nullable
    @SuppressWarnings("unchecked")  // implementation should take care that this is always a V
    public V findPrevious(@NotNull Predicate<? super V> condition)
    {
      if (condition.test((V)this)) {
        return (V)this;
      }
      for (V run = previous; run != this; run = run.next) {
        if (condition.test(run)) {
          return run;
        }
      }
      return null;
    }

    @SuppressWarnings("unchecked")  // implementation should take care that this is always a V
    @NotNull
    Iterator<V> iteratorTo(@NotNull V otherItem)
    {
      if (loop == null) {
        throw new IllegalStateException("Cannot iterate because item is outside loop!");
      }
      if (loop != otherItem.loop) {
        throw new IllegalArgumentException("Cannot iterate to an item in another loop!");
      }
      if (otherItem == this) {
        return loop.closedItemView().iterator();
      }
      return new ItemIterator<>((V)this, otherItem.next);
    }
  }

  /** First item in this loop, {@code null} if loop is empty. */
  private T first;
  /** Size of this loop. */
  private int size;

  /**
   * Create an empty loop.
   */
  public BasicLoop()
  {
  }

  /**
   * Create a loop from an iterable.
   * @param elements elements to insert into this loop, in the given order
   * @param creatorCreator function which maps the elements to an item creator
   * @param <V> element type of the iterable
   */
  public <V> BasicLoop(@NotNull Iterable<V> elements,
                       @NotNull Function<? super V, Function3<T, BasicLoop<T>, T, T>> creatorCreator)
  {
    addAll(elements, creatorCreator);
  }

  /**
   * Create a loop from an array.
   * @param creatorCreator function which maps the array elements to an item creator
   * @param elements elements to add to this loop
   * @param <V> element type of the array
   */
  @SafeVarargs
  @SuppressWarnings("varargs")
  public <V> BasicLoop(@NotNull  Function<? super V, Function3<T, BasicLoop<T>, T, T>> creatorCreator,
                       V ... elements)
  {
    addAll(Indexable.viewArray(elements), creatorCreator);
  }

  /**
   * Get the first item in the loop.
   * @return first item
   * @throws EmptyLoopError when loop is empty
   */
  @NotNull
  public T getFirstItem()
  {
    if (first == null) {
      throw new EmptyLoopError();
    }
    return first;
  }

  /**
   * Get the first item in the loop, or {@code null} if the loop is empty.
   * @return first item, or {@code null}
   */
  @Nullable
  public T getFirstItemOrNull()
  {
    return first;
  }

  /**
   * Get the last item in the loop.
   * @return last item
   * @throws EmptyLoopError when loop is empty
   */
  @NotNull
  public T getLastItem()
  {
    return getFirstItem().previous;
  }

  /**
   * Get the last item in the loop, or {@code null} if the loop is empty.
   * @return last item, or {@code null}
   */
  @Nullable
  public T getLastItemOrNull()
  {
    return first == null
            ? null
            : first.previous;
  }

  /**
   * Rotate through the loop so the first item is exchanged
   * @param steps steps to rotate,
   */
  public void rotate(int steps)
  {
    if (size < 2) {
      return;
    }
    steps = steps % size;
    if (steps < -size/2) {
      steps += size;
    }
    else if (steps > size/2) {
      steps -= size;
    }
    assert first != null; // because size is not 0

    if (steps == 0) {
      return;
    }
    if (steps < 0) {
      while (steps++ < 0) {
        first = first.previous;
      }
    }
    else {
      while (steps-- > 0) {
        first = first.next;
      }
    }
  }

  /**
   * Rotate forward until a condition is fulfilled for the first item.
   * If the condition is {@code true} for the first value no rotation occurs.
   * @param condition condition tested on value
   * @return {@code true} if the condition is now fulfilled for the {@link #getFirstItem() first} value,
   *         {@code false} if not or if this loop is empty
   */
  public boolean rotateForwardUntil(@NotNull Predicate<? super T> condition)
  {
    return rotateUntil(condition, Item<T>::getNext);
  }

  /**
   * Rotate backward until a condition is fulfilled for the first item.
   * If the condition is {@code true} for the first value no rotation occurs.
   * @param condition condition tested on value
   * @return {@code true} if the condition is now fulfilled for the {@link #getFirstItem() first} value,
   *         {@code false} if not or if this loop is empty
   */
  public boolean rotateBackwardUntil(@NotNull Predicate<? super T> condition)
  {
    return rotateUntil(condition, Item<T>::getPrevious);
  }

  /**
   * Rotate until a condition is fulfilled for the first item.
   * If the condition is already {@code true} for the first value no rotation occurs.
   * @param condition condition tested on value
   * @param rotator   generalized rotation, either forward or backward
   * @return {@code true} if the condition is now fulfilled for the {@link #getFirstItem() first} value,
   *         {@code false} if not or if this loop is empty
   */
  private boolean rotateUntil(@NotNull Predicate<? super T> condition,
                              @NotNull UnaryOperator<T> rotator)
  {
    if (first == null) {
      return false;
    }
    int tests = 0;
    while (tests++ < size) {
      if (condition.test(first)) {
        return true;
      }
      first = rotator.apply(first);
    }
    return false;
  }

  /**
   * Find the first item in this loop which fulfills a condition.
   * <p>
   * Use {@link Item#findNext(Predicate)} on the result to continue searching.
   * @param condition condition to be checked
   * @return item found or {@code null} if no item matches the condition
   */
  @Nullable
  public T findFirst(@NotNull Predicate<? super T> condition)
  {
    if (first == null) {
      return null;
    }
    return first.findNext(condition);
  }

  /**
   * Find the first item in this loop which fulfills a condition.
   * <p>
   * Use {@link Item#findPrevious(Predicate)} on the result to continue searching.
   * @param condition condition to be checked
   * @return item found or {@code null} if no item matches the condition
   */
  @Nullable
  public T findLast(@NotNull Predicate<? super T> condition)
  {
    if (first == null) {
      return null;
    }
    return first.findNext(condition);
  }

  /**
   * Add an item.
   * @param creator item creator, called with 2nd and 3rd arguments being {@code null}
   *                for the initial loop element
   * @return added item
   */
  @NotNull
  public T add(@NotNull Function3<T, BasicLoop<T>, T, T> creator)
  {
    if (first == null) {
      return first = creator.apply(this, null, null);
    }
    else {
      return first.insertBefore(creator);
    }
  }

  /**
   * Add all the elements to the loop.
   * This will add the given elements in the given order into the loop before the first element.
   * @param elements elements to add to this loop
   * @param creatorCreator function which creates the loop element creators from the incoming elements
   * @param <V> incoming value type
   */
  public <V> void addAll(@NotNull Iterable<V> elements,
                         @NotNull Function<? super V, Function3<T, BasicLoop<T>, T, T>> creatorCreator)
  {
    elements.forEach(elem -> add(creatorCreator.apply(elem)));
  }

  @Override
  public int size()
  {
    return size;
  }

  @Override
  public boolean isEmpty()
  {
    return first == null;
  }

  /**
   * Clear this loop.
   */
  public void clear()
  {
    first = null;
    size = 0;
  }

  /**
   * Get an iterator over the items of this loop.
   * @return item iterator
   */
  @NotNull
  public Iterator<T> iterator()
  {
    return first == null
            ? Types.emptyIterator()
            : new ItemIterator<>(first);
  }

  /**
   * View this loop's items as a countable which is "closed".
   * The returned countable will virtually close the loop by adding the first
   * item of the loop again at the end. This simplifies some usages of the loop.
   * @return closed item view of this loop
   */
  @NotNull
  public Countable<T> closedItemView()
  {
    if (isEmpty()) {
      return this;
    }
    return Countable.combined(this, Countable.singleton(first));
  }

  /**
   * Remove duplicate items which are neighbors.
   * @param equality equality check for items, expected to return {@code true} for duplicates
   */
  public void removeSuccessiveDuplicates(@NotNull BiPredicate<? super T, ? super T> equality)
  {
    if (size < 2) {
      return;
    }
    while (first.getPrevious() != first  &&  equality.test(first, first.getPrevious())) {
      first.getPrevious().remove();
    }
    for (T run = first.getNext();  run != first;  ) {
      final T next = run.getNext();
      if (equality.test(run.getPrevious(), run)) {
        run.remove();
      }
      run = next;
    }
  }

  /**
   * Remove duplicate items which are neighbors.
   * This method calls {@link #removeSuccessiveDuplicates(BiPredicate)} with
   * {@link Objects#deepEquals(Object, Object)} as equality check.
   */
  public void removeSuccessiveDuplicates()
  {
    removeSuccessiveDuplicates(Objects::deepEquals);
  }

  @Override
  public String toString()
  {
    return toString(" ");
  }

  /**
   * Create a string containing all items, connect with the given connector string.
   * @param connect connector string
   * @return string representation of this loop, items separated by {@code connect}
   */
  @NotNull
  public String toString(@NotNull String connect)
  {
    if (isEmpty()) {
      return "\u21BB";  // circular arrow
    }
    final StringBuilder sb = new StringBuilder("\u2924 "); // north-east arrow w/ hook
    boolean first = true;
    for (T item : this) {
      if (first) {
        first = false;
      }
      else {
        sb.append(connect).append("\u2192 ");  // right arrow
      }
      sb.append(item);
    }
    sb.append(connect).append("\u2926");  // south-west arrow w/ hook
    return sb.toString();
  }

  /**
   * Create a multi-line string from this loop.
   * @return multi-line string
   */
  @NotNull
  public String toMultiLineString()
  {
    return toString("\n");
  }

  /**
   * Item iterator.
   * @param <I> item type
   */
  private static class ItemIterator<I extends Item<I>>
          implements Iterator<I>
  {
    @NotNull
    private final I last;

    @Nullable
    private I next;

    ItemIterator(@NotNull I first)
    {
      this.last = first;
      next = first;
    }

    ItemIterator(@NotNull I first, @NotNull I last)
    {
      this.last = last.getNext();
      next = first;
    }

    @Override
    public boolean hasNext()
    {
      return next != null;
    }

    @Override
    public I next()
    {
      if (next == null) {
        throw new NoSuchElementException();
      }
      final I result = next;
      next = next.next;
      if (next == last) {
        next = null;
      }
      return result;
    }
  }
}
