// ============================================================================
// File:               ArrayIterator
//
// Project:            CAFF
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   © 2020-2024  Rammi (rammi@caff.de)
//                     The usage of this source code in commercial or open 
//                     source projects is not allowed without explicit 
//                     permission.
//
// Created:            24.12.20 12:08
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;

import java.util.Iterator;
import java.util.NoSuchElementException;

/**
 * An iterator over an array.
 * @param <T> element type of this iterator
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since December 24, 2020
 */
public class ArrayIterator<T>
        implements Iterator<T>
{
  // note: code assumes that Java array length cannot be Integer.MAX_VALUE which currently is the case
  @NotNull
  private final T[] array;
  private int run;
  private final int end;

  /**
   * Constructor.
   * This iterates over the complete array.
   * @param array array to iterate over
   */
  @SafeVarargs
  @SuppressWarnings("varargs") // necessary for Java 8
  public ArrayIterator(@NotNull T... array)
  {
    this(array, 0, array.length);
  }

  /**
   * Constructor.
   * This iterates over the given part of the array.
   * @param array  array to iterate over
   * @param start  start of iteration
   * @param length number of iterations
   */
  public ArrayIterator(@NotNull T[] array, int start, int length)
  {
    if (start < 0) {
      throw new IllegalArgumentException("start has to be non-negative, but is "+start);
    }
    if (length < 0) {
      throw new IllegalArgumentException("length has to be non-negative, but is "+length);
    }
    if (start > array.length - length) {
      throw new IllegalArgumentException(String.format("Cannot iterate to %d+%d=%d when array has only %d elements!",
                                                       start, length, (long)start + length, array.length));
    }
    this.array = array;
    this.run = start;
    this.end = start + length;
  }

  @Override
  public boolean hasNext()
  {
    return run < end;
  }

  @Override
  public T next()
  {
    if (run >= end) {
      throw new NoSuchElementException("Iterating over the end!");
    }
    return array[run++];
  }
}
