// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2014-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;

import java.util.*;

/**
 * An active lists sends events on changes.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class ActiveList<T>
        extends AbstractList<T>
{
  @NotNull
  private final List<T> wrapped;
  @NotNull
  private final Collection<ActiveListListener<? super T>> listeners =
          new SynchronizedCollection<>(new LinkedList<ActiveListListener<? super T>>());

  /**
   * Constructor.
   * This will allow all operations on the list.
   * @param wrapped wrapped list, don't use directly any further or add a clone here!
   */
  public ActiveList(@NotNull List<T> wrapped)
  {
    this.wrapped = wrapped;
  }

  /**
   * Wrap a basic list to make it active.
   * @param basicList basic list, don't use directly any further or add a clone here!
   * @param <S> list type
   * @return active list
   */
  @NotNull
  public static <S> ActiveList<S> wrap(@NotNull List<S> basicList)
  {
    return new ActiveList<>(basicList);
  }

  /**
   * Add an active list listener.
   * @param listener listener to add
   */
  public void addActiveListListener(@NotNull ActiveListListener<? super T> listener)
  {
    listeners.add(listener);
  }

  /**
   * Remove an active list listener.
   * @param listener listener to remove
   */
  public void removeActiveListListener(@NotNull ActiveListListener<? super T> listener)
  {
    listeners.remove(listener);
  }

  /**
   * Inform all listeners of an added item.
   * @param index index where item was added
   * @param item  added item
   */
  private void fireAdded(int index, T item)
  {
    for (ActiveListListener<? super T> l : listeners) {
      l.itemAdded(index, item);
    }
  }

  /**
   * Inform all listeners of a removed item.
   * @param index index where item was removed
   * @param item  removed item
   */
  private void fireRemoved(int index, T item)
  {
    for (ActiveListListener<? super T> l : listeners) {
      l.itemRemoved(index, item);
    }
  }

  /**
   * Inform all listeners of an exchanged item.
   * @param index   index where item was removed
   * @param oldItem removed item
   * @param newItem added item
   */
  private void fireExchanged(int index, T oldItem, T newItem)
  {
    for (ActiveListListener<? super T> l : listeners) {
      l.itemExchanged(index, oldItem, newItem);
    }
  }

  /**
   * Returns the number of elements in this list.  If this list contains
   * more than <tt>Integer.MAX_VALUE</tt> elements, returns
   * <tt>Integer.MAX_VALUE</tt>.
   *
   * @return the number of elements in this list
   */
  @Override
  public int size()
  {
    return wrapped.size();
  }

  /**
   * Returns <tt>true</tt> if this list contains no elements.
   *
   * @return <tt>true</tt> if this list contains no elements
   */
  @Override
  public boolean isEmpty()
  {
    return wrapped.isEmpty();
  }

  /**
   * Returns <tt>true</tt> if this list contains the specified element.
   * More formally, returns <tt>true</tt> if and only if this list contains
   * at least one element <tt>e</tt> such that
   * <tt>(o==null&nbsp;?&nbsp;e==null&nbsp;:&nbsp;o.equals(e))</tt>.
   *
   * @param o element whose presence in this list is to be tested
   * @return <tt>true</tt> if this list contains the specified element
   * @throws ClassCastException   if the type of the specified element
   *                              is incompatible with this list (optional)
   * @throws NullPointerException if the specified element is null and this
   *                              list does not permit null elements (optional)
   */
  @Override
  public boolean contains(Object o)
  {
    return wrapped.contains(o);
  }

  /**
   * Returns an array containing all of the elements in this list in proper
   * sequence (from first to last element).
   *
   * <p>The returned array will be "safe" in that no references to it are
   * maintained by this list.  (In other words, this method must
   * allocate a new array even if this list is backed by an array).
   * The caller is thus free to modify the returned array.
   *
   * <p>This method acts as bridge between array-based and collection-based
   * APIs.
   *
   * @return an array containing all of the elements in this list in proper
   * sequence
   * @see java.util.Arrays#asList(Object[])
   */
  @NotNull
  @Override
  public Object[] toArray()
  {
    return wrapped.toArray();
  }

  /**
   * Returns an array containing all of the elements in this list in
   * proper sequence (from first to last element); the runtime type of
   * the returned array is that of the specified array.  If the list fits
   * in the specified array, it is returned therein.  Otherwise, a new
   * array is allocated with the runtime type of the specified array and
   * the size of this list.
   *
   * <p>If the list fits in the specified array with room to spare (i.e.,
   * the array has more elements than the list), the element in the array
   * immediately following the end of the list is set to <tt>null</tt>.
   * (This is useful in determining the length of the list <i>only</i> if
   * the caller knows that the list does not contain any null elements.)
   *
   * <p>Like the {@link #toArray()} method, this method acts as bridge between
   * array-based and collection-based APIs.  Further, this method allows
   * precise control over the runtime type of the output array, and may,
   * under certain circumstances, be used to save allocation costs.
   *
   * <p>Suppose <tt>x</tt> is a list known to contain only strings.
   * The following code can be used to dump the list into a newly
   * allocated array of <tt>String</tt>:
   *
   * <pre>
   *     String[] y = x.toArray(new String[0]);</pre>
   *
   * Note that <tt>toArray(new Object[0])</tt> is identical in function to
   * <tt>toArray()</tt>.
   *
   * @param a the array into which the elements of this list are to
   *          be stored, if it is big enough; otherwise, a new array of the
   *          same runtime type is allocated for this purpose.
   * @return an array containing the elements of this list
   * @throws ArrayStoreException  if the runtime type of the specified array
   *                              is not a supertype of the runtime type of every element in
   *                              this list
   * @throws NullPointerException if the specified array is null
   */
  @NotNull
  @Override
  public <T1> T1[] toArray(@NotNull T1[] a)
  {
    return wrapped.toArray(a);
  }

  private boolean doRemove(int index, T item)
  {
    if (wrapped.remove(item)) {
      fireRemoved(index, item);
      return true;
    }
    return false;
  }

  /**
   * Appends the specified element to the end of this list (optional
   * operation).
   *
   * <p>Lists that support this operation may place limitations on what
   * elements may be added to this list.  In particular, some
   * lists will refuse to add null elements, and others will impose
   * restrictions on the type of elements that may be added.  List
   * classes should clearly specify in their documentation any restrictions
   * on what elements may be added.
   *
   * @param t element to be appended to this list
   * @return <tt>true</tt> (as specified by {@link java.util.Collection#add})
   * @throws UnsupportedOperationException if the <tt>add</tt> operation
   *                                       is not supported by this list
   * @throws ClassCastException            if the class of the specified element
   *                                       prevents it from being added to this list
   * @throws NullPointerException          if the specified element is null and this
   *                                       list does not permit null elements
   * @throws IllegalArgumentException      if some property of this element
   *                                       prevents it from being added to this list
   */
  @Override
  public boolean add(T t)
  {
    final int index = size();
    if (wrapped.add(t)) {
      fireAdded(index, t);
      return true;
    }
    return false;
  }

  /**
   * Removes the first occurrence of the specified element from this list,
   * if it is present (optional operation).  If this list does not contain
   * the element, it is unchanged.  More formally, removes the element with
   * the lowest index <tt>i</tt> such that
   * <tt>(o==null&nbsp;?&nbsp;get(i)==null&nbsp;:&nbsp;o.equals(get(i)))</tt>
   * (if such an element exists).  Returns <tt>true</tt> if this list
   * contained the specified element (or equivalently, if this list changed
   * as a result of the call).
   *
   * @param o element to be removed from this list, if present
   * @return <tt>true</tt> if this list contained the specified element
   * @throws ClassCastException            if the type of the specified element
   *                                       is incompatible with this list (optional)
   * @throws NullPointerException          if the specified element is null and this
   *                                       list does not permit null elements (optional)
   * @throws UnsupportedOperationException if the <tt>remove</tt> operation
   *                                       is not supported by this list
   */
  @Override
  @SuppressWarnings("unchecked")
  public boolean remove(Object o)
  {
    int index = indexOf(o);
    if (index < 0) {
      return false;
    }
    return doRemove(index, (T)o);
  }

  /**
   * Returns <tt>true</tt> if this list contains all of the elements of the
   * specified collection.
   *
   * @param c collection to be checked for containment in this list
   * @return <tt>true</tt> if this list contains all of the elements of the
   * specified collection
   * @throws ClassCastException   if the types of one or more elements
   *                              in the specified collection are incompatible with this
   *                              list (optional)
   * @throws NullPointerException if the specified collection contains one
   *                              or more null elements and this list does not permit null
   *                              elements (optional), or if the specified collection is null
   * @see #contains(Object)
   */
  @Override
  public boolean containsAll(@NotNull Collection<?> c)
  {
    return wrapped.containsAll(c);
  }

  /**
   * Removes all of the elements from this list (optional operation).
   * The list will be empty after this call returns.
   *
   * @throws UnsupportedOperationException if the <tt>clear</tt> operation
   *                                       is not supported by this list
   */
  @Override
  public void clear()
  {
    if (wrapped.isEmpty()) {
      return;
    }
    final List<T> tmp = new ArrayList<>(wrapped);
    for (ListIterator<T> it = tmp.listIterator(tmp.size());
            it.hasPrevious();
            ) {
      int index = it.previousIndex();
      T item = it.previous();
      doRemove(index, item);
    }
  }

  /**
   * Returns the element at the specified position in this list.
   *
   * @param index index of the element to return
   * @return the element at the specified position in this list
   * @throws IndexOutOfBoundsException if the index is out of range
   *                                   (<tt>index &lt; 0 || index &gt;= size()</tt>)
   */
  @Override
  public T get(int index)
  {
    return wrapped.get(index);
  }

  /**
   * Replaces the element at the specified position in this list with the
   * specified element (optional operation).
   *
   * @param index   index of the element to replace
   * @param element element to be stored at the specified position
   * @return the element previously at the specified position
   * @throws UnsupportedOperationException if the <tt>set</tt> operation
   *                                       is not supported by this list
   * @throws ClassCastException            if the class of the specified element
   *                                       prevents it from being added to this list
   * @throws NullPointerException          if the specified element is null and
   *                                       this list does not permit null elements
   * @throws IllegalArgumentException      if some property of the specified
   *                                       element prevents it from being added to this list
   * @throws IndexOutOfBoundsException     if the index is out of range
   *                                       (<tt>index &lt; 0 || index &gt;= size()</tt>)
   */
  @Override
  public T set(int index, T element)
  {
    final T oldElement = wrapped.set(index, element);
    fireExchanged(index, oldElement, element);
    return oldElement;
  }

  /**
   * Inserts the specified element at the specified position in this list
   * (optional operation).  Shifts the element currently at that position
   * (if any) and any subsequent elements to the right (adds one to their
   * indices).
   *
   * @param index   index at which the specified element is to be inserted
   * @param element element to be inserted
   * @throws UnsupportedOperationException if the <tt>add</tt> operation
   *                                       is not supported by this list
   * @throws ClassCastException            if the class of the specified element
   *                                       prevents it from being added to this list
   * @throws NullPointerException          if the specified element is null and
   *                                       this list does not permit null elements
   * @throws IllegalArgumentException      if some property of the specified
   *                                       element prevents it from being added to this list
   * @throws IndexOutOfBoundsException     if the index is out of range
   *                                       (<tt>index &lt; 0 || index &gt; size()</tt>)
   */
  @Override
  public void add(int index, T element)
  {
    wrapped.add(index, element);
    fireAdded(index, element);
  }


  /**
   * Removes the element at the specified position in this list (optional
   * operation).  Shifts any subsequent elements to the left (subtracts one
   * from their indices).  Returns the element that was removed from the
   * list.
   *
   * @param index the index of the element to be removed
   * @return the element previously at the specified position
   * @throws UnsupportedOperationException if the <tt>remove</tt> operation
   *                                       is not supported by this list
   * @throws IndexOutOfBoundsException     if the index is out of range
   *                                       (<tt>index &lt; 0 || index &gt;= size()</tt>)
   */
  @Override
  public T remove(int index)
  {
    final T item = wrapped.remove(index);
    fireRemoved(index, item);
    return item;
  }

  /**
   * Returns the index of the first occurrence of the specified element
   * in this list, or -1 if this list does not contain the element.
   * More formally, returns the lowest index <tt>i</tt> such that
   * <tt>(o==null&nbsp;?&nbsp;get(i)==null&nbsp;:&nbsp;o.equals(get(i)))</tt>,
   * or -1 if there is no such index.
   *
   * @param o element to search for
   * @return the index of the first occurrence of the specified element in
   * this list, or -1 if this list does not contain the element
   * @throws ClassCastException   if the type of the specified element
   *                              is incompatible with this list (optional)
   * @throws NullPointerException if the specified element is null and this
   *                              list does not permit null elements (optional)
   */
  @Override
  public int indexOf(Object o)
  {
    return wrapped.indexOf(o);
  }

  /**
   * Returns the index of the last occurrence of the specified element
   * in this list, or -1 if this list does not contain the element.
   * More formally, returns the highest index <tt>i</tt> such that
   * <tt>(o==null&nbsp;?&nbsp;get(i)==null&nbsp;:&nbsp;o.equals(get(i)))</tt>,
   * or -1 if there is no such index.
   *
   * @param o element to search for
   * @return the index of the last occurrence of the specified element in
   * this list, or -1 if this list does not contain the element
   * @throws ClassCastException   if the type of the specified element
   *                              is incompatible with this list (optional)
   * @throws NullPointerException if the specified element is null and this
   *                              list does not permit null elements (optional)
   */
  @Override
  public int lastIndexOf(Object o)
  {
    return wrapped.lastIndexOf(o);
  }
}
