// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================

package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.generics.util.Counter;

import java.util.Collection;
import java.util.Iterator;
import java.util.Map;
import java.util.function.Supplier;

/**
 * Basic implementation for counting sets.
 * <p>
 * Because it would be breaking the set semantics removal is not allowed.
 * All removal methods but {@link CountingHashSet#clear()}
 * therefore will throw an {@link UnsupportedOperationException}.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
class AbstractBasicCountingSet<T>
        implements CountingSet<T>
{
  @NotNull
  protected final Supplier<Counter> creator;
  @NotNull
  protected final Map<T, Counter> map;

  /**
   * Constructor.
   * @param map     counter map
   * @param creator creator for more counters
   */
  public AbstractBasicCountingSet(@NotNull Map<T, Counter> map,
                                  @NotNull Supplier<Counter> creator)
  {
    this.creator = creator;
    this.map = map;
  }

  /**
   * Returns the number of elements in this set (its cardinality).  If this
   * set contains more than <tt>Integer.MAX_VALUE</tt> elements, returns
   * <tt>Integer.MAX_VALUE</tt>.
   *
   * @return the number of elements in this set (its cardinality).
   */
  @Override
  public int size()
  {
    return map.size();
  }

  /**
   * Returns <tt>true</tt> if this set contains no elements.
   *
   * @return <tt>true</tt> if this set contains no elements.
   */
  @Override
  public boolean isEmpty()
  {
    return map.isEmpty();
  }

  /**
   * Returns <tt>true</tt> if this set contains the specified element.  More
   * formally, returns <tt>true</tt> if and only if this set contains an
   * element {@code e} such that {@code (o==null ? e==null : o.equals(e))}.
   *
   * @param o element whose presence in this set is to be tested.
   * @return <tt>true</tt> if this set contains the specified element.
   * @throws ClassCastException   if the type of the specified element
   *                              is incompatible with this set (optional).
   * @throws NullPointerException if the specified element is null and this
   *                              set does not support null elements (optional).
   */
  @Override
  public boolean contains(Object o)
  {
    return map.containsKey(o);
  }

  /**
   * Returns an iterator over the elements in this set.  The elements are
   * returned in no particular order (unless this set is an instance of some
   * class that provides a guarantee).
   *
   * @return an iterator over the elements in this set.
   */
  @Override
  @NotNull
  public Iterator<T> iterator()
  {
    return map.keySet().iterator();
  }

  /**
   * Returns an array containing all of the elements in this set.
   * Obeys the general contract of the <tt>Collection.toArray</tt> method.
   *
   * @return an array containing all of the elements in this set.
   */
  @Override
  @NotNull
  public Object[] toArray()
  {
    return map.keySet().toArray();
  }

  /**
   * Returns an array containing all of the elements in this set; the
   * runtime type of the returned array is that of the specified array.
   * Obeys the general contract of the
   * <tt>Collection.toArray(Object[])</tt> method.
   *
   * @param a the array into which the elements of this set are to
   *          be stored, if it is big enough; otherwise, a new array of the
   *          same runtime type is allocated for this purpose.
   * @return an array containing the elements of this set.
   * @throws ArrayStoreException  the runtime type of a is not a supertype
   *                              of the runtime type of every element in this set.
   * @throws NullPointerException if the specified array is <tt>null</tt>.
   */
  @Override
  @NotNull
  public <U> U[] toArray(@NotNull U[] a)
  {
    return map.keySet().toArray(a);
  }

  /**
   * Adds the specified element to this set if it is not already present
   * (optional operation).  More formally, adds the specified element,
   * {@code o}, to this set if this set contains no element
   * {@code e} such that {@code (o==null ? e==null : o.equals(e))}.
   * If this set already contains the specified
   * element, the call leaves this set unchanged and returns <tt>false</tt>.
   * In combination with the restriction on constructors, this ensures that
   * sets never contain duplicate elements.<p>
   *
   * The stipulation above does not imply that sets must accept all
   * elements; sets may refuse to add any particular element, including
   * <tt>null</tt>, and throwing an exception, as described in the
   * specification for <tt>Collection.add</tt>.  Individual set
   * implementations should clearly document any restrictions on the
   * elements that they may contain.
   *
   * @param o element to be added to this set.
   * @return <tt>true</tt>
   * @throws UnsupportedOperationException if the <tt>add</tt> method is not
   *                                       supported by this set.
   * @throws ClassCastException            if the class of the specified element
   *                                       prevents it from being added to this set.
   * @throws NullPointerException          if the specified element is null and this
   *                                       set does not support null elements.
   * @throws IllegalArgumentException      if some aspect of the specified element
   *                                       prevents it from being added to this set.
   */
  @Override
  public boolean add(T o)
  {
    Counter counter = map.get(o);
    boolean result = counter == null;
    if (result) {
      counter = creator.get();
      map.put(o, counter);
    }
    counter.add1();
    return result;
  }

  /**
   * Removes the specified element from this set if it is present (optional
   * operation).
   * <p>
   * This method will throw an {@link UnsupportedOperationException}!
   *
   * @param o object to be removed from this set, if present.
   * @return true if the set contained the specified element.
   * @throws ClassCastException            if the type of the specified element
   *                                       is incompatible with this set (optional).
   * @throws NullPointerException          if the specified element is null and this
   *                                       set does not support null elements (optional).
   * @throws UnsupportedOperationException if the <tt>remove</tt> method is
   *                                       not supported by this set.
   */
  @Override
  public boolean remove(Object o)
  {
    throw new UnsupportedOperationException("removal not allowed!");
  }

  /**
   * Returns <tt>true</tt> if this set contains all of the elements of the
   * specified collection.  If the specified collection is also a set, this
   * method returns <tt>true</tt> if it is a <i>subset</i> of this set.
   *
   * @param c collection to be checked for containment in this set.
   * @return <tt>true</tt> if this set contains all of the elements of the
   *         specified collection.
   * @throws ClassCastException   if the types of one or more elements
   *                              in the specified collection are incompatible with this
   *                              set (optional).
   * @throws NullPointerException if the specified collection contains one
   *                              or more null elements and this set does not support null
   *                              elements (optional);
   *                              or if the specified collection is
   *                              <tt>null</tt>.
   * @see #contains(Object)
   */
  @Override
  public boolean containsAll(@NotNull Collection<?> c)
  {
    return map.keySet().containsAll(c);
  }

  /**
   * Adds all of the elements in the specified collection to this set if
   * they're not already present (optional operation).  If the specified
   * collection is also a set, the <tt>addAll</tt> operation effectively
   * modifies this set so that its value is the <i>union</i> of the two
   * sets.  The behavior of this operation is unspecified if the specified
   * collection is modified while the operation is in progress.
   *
   * @param c collection whose elements are to be added to this set.
   * @return <tt>true</tt> if this set changed as a result of the call.
   * @throws UnsupportedOperationException if the <tt>addAll</tt> method is
   *                                       not supported by this set.
   * @throws ClassCastException            if the class of some element of the
   *                                       specified collection prevents it from being added to this
   *                                       set.
   * @throws NullPointerException          if the specified collection contains one
   *                                       or more null elements and this set does not support null
   *                                       elements, or if the specified collection is <tt>null</tt>.
   * @throws IllegalArgumentException      if some aspect of some element of the
   *                                       specified collection prevents it from being added to this
   *                                       set.
   * @see #add(Object)
   */
  @Override
  public boolean addAll(@NotNull Collection<? extends T> c)
  {
    boolean result = false;
    for (T elem : c) {
      if (add(elem)) {
        result = true;
      }
    }
    return result;
  }

  /**
   * Retains only the elements in this set that are contained in the
   * specified collection (optional operation).
   * <p>
   * This method will throw an {@link UnsupportedOperationException}!
   *
   * @param c collection that defines which elements this set will retain.
   * @return <tt>true</tt> if this collection changed as a result of the
   *         call.
   * @throws UnsupportedOperationException if the <tt>retainAll</tt> method
   *                                       is not supported by this Collection.
   * @throws ClassCastException            if the types of one or more elements in this
   *                                       set are incompatible with the specified collection
   *                                       (optional).
   * @throws NullPointerException          if this set contains a null element and
   *                                       the specified collection does not support null elements
   *                                       (optional);
   *                                       or if the specified collection is
   *                                       <tt>null</tt>.
   * @see #remove(Object)
   */
  @Override
  public boolean retainAll(@NotNull Collection<?> c)
  {
    throw new UnsupportedOperationException("removal not allowed!");
  }

  /**
   * Removes from this set all of its elements that are contained in the
   * specified collection (optional operation).
   * <p>
   * This method will throw an {@link UnsupportedOperationException}!
   *
   * @param c collection that defines which elements will be removed from
   *          this set.
   * @return <tt>true</tt> if this set changed as a result of the call.
   * @throws UnsupportedOperationException if the <tt>removeAll</tt>
   *                                       method is not supported by this Collection.
   * @throws ClassCastException            if the types of one or more elements in this
   *                                       set are incompatible with the specified collection
   *                                       (optional).
   * @throws NullPointerException          if this set contains a null element and
   *                                       the specified collection does not support null elements
   *                                       (optional);
   *                                       or if the specified collection is
   *                                       <tt>null</tt>.
   * @see #remove(Object)
   */
  @Override
  public boolean removeAll(@NotNull Collection<?> c)
  {
    throw new UnsupportedOperationException("removal not allowed!");
  }

  /**
   * Removes all of the elements from this set (optional operation).
   * This set will be empty after this call returns (unless it throws an
   * exception).
   *
   * @throws UnsupportedOperationException if the <tt>clear</tt> method
   *                                       is not supported by this set.
   */
  @Override
  public void clear()
  {
    map.clear();
  }

  /**
   * Get the count how often any item was added.
   * @param item item to look for
   * @return add count or {@code 0} if the item was never added
   *         (or the clear code has been used after its addition)
   */
  @Override
  public int getCount(T item)
  {
    Counter counter = map.get(item);
    return counter != null ? counter.getValue() : 0;
  }
}
