// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;

/**
 * Basic implementation of copyable object.
 * <p>
 * It uses cloning to create the copy.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class AbstractBasicCopyable<T extends Cloneable>
  implements Copyable<T>
{
  /**
   * Get a copy of this object.
   * Here the clone method is used to copy the object.
   * Cloneable objects are not expected to throw a {@link CloneNotSupportedException}.
   * If one is thrown nevertheless this method will an {@link InternalError} instead.
   *
   * @return independent object
   */
  @Override
  @NotNull
  @SuppressWarnings("unchecked")
  public T getCopy()
  {
    try {
      return (T)clone();
    } catch (CloneNotSupportedException e) {
      throw new InternalError("Cloneable can't clone: "+this.getClass());
    }
  }
}
