// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2015-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================

package de.caff;

import de.caff.version.ModuleVersionService;

import java.util.Comparator;

/**
 * Convenience class for accessing the release date of jars from de·caff.
 * <p>
 * The release date is contained in various central classes, and there is also
 * a packaging date in the jars. Running this classes main method
 * just forwards the action to {@link de.caff.util.Utility#main(String[])}
 * which prints out the jar build date and its own release date
 * as well as the release date of all classes found on the command line.
 * </p>
 * <p>
 * The sole purpose of this class is to make accessing the release date easier.
 * Now it is possible to extract the release data of {@code xyz.jar} by running
 * the following command:
 * </p>
 * <pre>
 * java -cp xyz.jar de.caff.Version
 * </pre>
 * instead of the more difficult to remember
 * <pre>
 * java -cp xyz.jar de.caff.util.Utility
 * </pre>
 * <p>
 * Obviously having a package-less class would make this even better,
 * but polluting the common name space seemed too much for a bit of
 * convenience.
 * </p>
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since October 25, 2018
 */
public final class Version
{
  /** Don't create. This is a static class. */
  private Version() {}

  /**
   * See {@link de.caff.version.ModuleVersionService} for more info.
   * @param args complete class names like {@code de.caff.util.Utility}
   */
  public static void main(String[] args)
  {
    final String format = "%40s: %s\n";
    ModuleVersionService.collectServices()
            .sorted(Comparator.comparing(ModuleVersionService::getModuleName))
            .forEach(mvs -> System.out.printf(format, mvs.getModuleName(), mvs.getModuleVersion()));
  }
}
