// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2016-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.xml;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.Types;
import de.caff.util.Base64;

import java.util.Locale;
import java.util.Stack;

/**
 * Simple XML writer.
 * <p>
 *   This writer makes creation of simple XML documents without advanced features simple.
 *   Use the following to create simple XML:
 * </p>
 * <ul>
 *   <li>
 *     Most constructors already create the XML declaration with version
 *     (always 1.0) and encoding. Other constructors allow finetuning, e.g.
 *     for HTML creation.
 *   </li>
 *   <li>
 *     The {@link #element(String, Object...)} method writes an empty element
 *     which is already closed with the given attributes' name-value pairs.
 *   </li>
 *   <li>
 *     The {@link #open(String, Object...)} method writes an opening element
 *     with the given  attributes' name-value pairs.
 *   </li>
 *   <li>
 *     The {@link #close()} method closes the element created by the latest
 *     {@link #open(String, Object...)} call.
 *   </li>
 *   <li>
 *     The {@link #finish()} method closes all open elements.
 *   </li>
 *   <li>
 *     The {@link #comment(String)} method inserts a comment with the given text.
 *   </li>
 *   <li>
 *     The {@link #text(Object)} method inserts the given text, taking care of
 *     escaping.
 *   </li>
 *   <li>
 *     The {@link #textf(String, Object...)} method does the same, but allows
 *     formatting like {@link String#format(String, Object...)}.
 *   </li>
 *   <li>
 *     The {@link #cdata(String)} method inserts the given text in raw
 *     form as a CDATA section.
 *   </li>
 *   <li>
 *     The {@link #nl()} method can be used to insert a newline,
 *     see also the comments on automatic wrapping below.
 *   </li>
 *   <li>
 *     The {@link #expand(byte[])} inserts binary data as 2-digit-per-byte
 *     hexadecimal numbers.
 *   </li>
 *   <li>
 *     The {@link #expandBase64(byte[])} inserts binary data in base64 encoding.
 *   </li>
 *   <li>
 *     The {@link #expand(byte[], int)} method can be used to format binary
 *     data as 2-digit-per-byte hexadecimal numbers, and can be used with
 *     either the {@link #text(Object)} method or as attribute value (in the
 *     latter case best with {@code columns} set to {@code 0}).
 *   </li>
 * </ul>
 * <p>
 *   Automatic wrapping and indentation of hierarchy levels can be switched on
 *   by using constructor which accept an {@code indentation} parameter with
 *   a non-negative indentation value.
 *   A value of {@code 0} switches auto-wrapping on, starting each opening tag
 *   on a new line. A value greater than zero will also indent each sub level
 *   by the given number of blanks. A negative value will switch both auto-wrapping
 *   and indentation off.
 * </p>
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @deprecated this class has an improved substitute, see {@link de.caff.io.xml.SimpleXmlWriter},
 *             where {@link #open(String, Object...)} is renamed to {@link de.caff.io.xml.SimpleXmlWriter#begin(String, Object...)}
 *             and {@link #close()} to {@link de.caff.io.xml.SimpleXmlWriter#end()}, which allows to make the class
 *             closable.
 */
@Deprecated
public class SimpleXmlWriter
{
  /** Default encoding in XML header. */
  public static final String DEFAULT_ENCODING = "utf-8";

  private static final String[] HEX = new String[256];

  static {
    for (int b = 255;  b >= 0;  --b) {
      HEX[b] = String.format("%02X", b);
    }
  }
  /** Stack of open element tags. */
  @NotNull
  private final Stack<String> openElements = new Stack<>();
  /** Collector used for output text collection. */
  @NotNull
  protected final StringCollector collector;
  /** Is automatic wrapping enabled? */
  protected final  boolean autoWrap;
  /** Use as indentation if non-{@code null}. */
  @Nullable
  protected final String indentation;
  /** Locale to be used when formatting. */
  @NotNull
  protected final Locale locale;

  /**
   * Constructor.
   * The created writer will use the {@link #getDefaultPreamble() default preamble},
   * default locale, no indentation, and no automatic wrapping,
   * @param collector string collector to which the XML output is written
   * @see OutputStreamStringCollector
   * @see TextStringCollector
   */
  public SimpleXmlWriter(@NotNull StringCollector collector)
  {
    this(collector, -1);
  }

  /**
   * Constructor.
   * The created writer will use the {@link #getDefaultPreamble() default preamble},
   * no indentation, and no automatic wrapping,
   * @param locale    locale used for text formatting in {@link #textf(String, Object...)}
   * @param collector string collector to which the XML output is written
   * @see OutputStreamStringCollector
   * @see TextStringCollector
   */
  public SimpleXmlWriter(@NotNull Locale locale, @NotNull StringCollector collector)
  {
    this(locale, collector, -1);
  }

  /**
   * Constructor.
   * The created writer will use the default locale, no indentation and no automatic wrapping.
   * @param collector string collector to which the XML output is written
   * @param preamble  preamble written at the start, if {@code null} no preamble is written
   * @see OutputStreamStringCollector
   * @see TextStringCollector
   */
  public SimpleXmlWriter(@NotNull StringCollector collector,
                         @Nullable String preamble)
  {
    this(collector, preamble, -1);
  }

  /**
   * Constructor.
   * The created writer will use no indentation and no automatic wrapping.
   * @param locale    locale used for text formatting in {@link #textf(String, Object...)}
   * @param collector string collector to which the XML output is written
   * @param preamble  preamble written at the start, if {@code null} no preamble is written
   * @see OutputStreamStringCollector
   * @see TextStringCollector
   */
  public SimpleXmlWriter(@NotNull Locale locale,
                         @NotNull StringCollector collector,
                         @Nullable String preamble)
  {
    this(locale, collector, preamble, -1);
  }

  /**
   * Constructor.
   * The created writer will use the default locale, and the {@link #getDefaultPreamble() default preamble}.
   * @param collector string collector to which the XML output is written
   * @param indentation if {@code 0} or more automatic newlines will be inserted and nested levels will be indented
   *                    by the given number of spaces
   */
  public SimpleXmlWriter(@NotNull StringCollector collector,
                         int indentation)
  {
    this(collector, getDefaultPreamble(), indentation);
  }

  /**
   * Constructor.
   * The created writer will use the {@link #getDefaultPreamble() default preamble}.
   * @param locale    locale used for text formatting in {@link #textf(String, Object...)}
   * @param collector string collector to which the XML output is written
   * @param indentation if {@code 0} or more automatic newlines will be inserted and nested levels will be indented
   *                    by the given number of spaces
   */
  public SimpleXmlWriter(@NotNull Locale locale,
                         @NotNull StringCollector collector,
                         int indentation)
  {
    this(locale, collector, getDefaultPreamble(), indentation);
  }

  /**
   * Constructor.
   * The created writer will us the default locale.
   * @param collector string collector to which the XML output is written
   * @param preamble preamble to be written at the beginning of the output, use {@code null} for no preamble
   * @param indentation if {@code 0} or more automatic newlines will be inserted and nested levels will be indented
   *                    by the given number of spaces
   * @see #getDefaultPreamble()
   * @see #getPreamble(String)
   */
  public SimpleXmlWriter(@NotNull StringCollector collector,
                         @Nullable String preamble,
                         int indentation)
  {
    this(Locale.getDefault(), collector, preamble, indentation);
  }

  /**
   * Constructor.
   * @param locale    locale used for text formatting in {@link #textf(String, Object...)}
   * @param collector string collector to which the XML output is written
   * @param preamble preamble to be written at the beginning of the output, use {@code null} for no preamble
   * @param indentation if {@code 0} or more automatic newlines will be inserted and nested levels will be indented
   *                    by the given number of spaces
   * @see #getDefaultPreamble()
   * @see #getPreamble(String)
   */
  public SimpleXmlWriter(@NotNull Locale locale,
                         @NotNull StringCollector collector,
                         @Nullable String preamble,
                         int indentation)
  {
    this.collector = collector;
    this.locale = locale;
    this.autoWrap = indentation >= 0;
    this.indentation = indentation > 0
            ? Types.spaces(indentation)
            : null;
    if (preamble != null) {
      add(preamble);
    }
  }

  /**
   * Is automatic wrapping enabled?
   * Automatic wrapping is enabled when a constructor with a non-negative indentation is used.
   * @return {@code true}: automatic wrapping is enabled<br>
   *         {@code false}: automatic wrapping is disabled
   */
  public boolean isAutoWrap()
  {
    return autoWrap;
  }

  /**
   * Get the indentation.
   * @return indentation (a number of spaces), possibly empty
   */
  @NotNull
  public String getIndentation()
  {
    return Types.notNull(indentation);
  }

  /**
   * Get the locale used for formatting text.
   * @return locale for formatting text
   * @see #textf(String, Object...)
   */
  @NotNull
  public Locale getLocale()
  {
    return locale;
  }

  /**
   * Get the default XML preamble with UTF-8 encoding.
   * @return default preamble
   */
  @NotNull
  public static String getDefaultPreamble()
  {
    return getPreamble(DEFAULT_ENCODING);
  }

  /**
   * Get an XML preamble with the given encoding.
   * @param encoding encoding
   * @return XML preamble with defining the given encoding
   */
  @NotNull
  public static String getPreamble(@NotNull String encoding)
  {
    return String.format("<?xml version=\"1.0\" encoding=\"%s\"?>", encoding);
  }

  /**
   * Insert an automatic wrap.
   * @param level indentation level
   */
  @SuppressWarnings("fallthrough")
  private void insertAutoWrap(final int level)
  {
    if (!autoWrap) {
      return;
    }
    switch (collector.getLastLetter()) {
    case '>':
      nl();
      // fallthrough
    case '\n':
      if (level > 0  &&  indentation != null) {
        for (int i = 0;  i < level;  ++i) {
          add(indentation);
        }
      }
    }
  }

  /**
   * Write the opening tag of an element.
   * @param tag         element tag
   * @param attributes  attributes, always in pairs of attribute name
   *                    and attribute value, values will be escaped
   * @return writer for chaining
   */
  @NotNull
  public SimpleXmlWriter open(@NotNull String tag,
                              Object ... attributes)
  {
    openElements.push(tag);
    return writeElement(true, tag, attributes);
  }

  /**
   * Close the latest open element.
   * This goes up one step up hierarchy.
   * @return writer for chaining
   * @throws java.util.EmptyStackException if closing although no element is open
   */
  @NotNull
  public SimpleXmlWriter close()
  {
    insertAutoWrap(openElements.size() - 1);
    add("</").quote(openElements.pop()).add(">");
    return this;
  }

  /**
   * Add a comment.
   * @param comment comment to add
   * @return writer for chaining
   */
  @NotNull
  public SimpleXmlWriter comment(@NotNull String comment)
  {
    insertAutoWrap(openElements.size());
    add("<!-- ").add(comment).add("-->");
    return this;
  }

  /**
   * Convert data into hex code and add it as text.
   * @param data data to add
   * @return writer for chaining
   */
  @NotNull
  public SimpleXmlWriter expand(@NotNull byte[] data)
  {
    for (byte b : data)  {
      add(HEX[b & 0xFF]);
    }
    return this;
  }

  /**
   * Expand an array of bytes into a hex string.
   * @param data     data to expand
   * @param columns  columns of hex digits of returned string before a line break is inserted.
   *                 If {@code 0} or less no break is inserted.
   * @return hex string
   */
  @NotNull
  public static String expand(@NotNull byte[] data, final int columns)
  {
    final StringBuilder sb;
    if (columns > 0) {
      sb = new StringBuilder(2*data.length + data.length / columns);
      for (int c = 0;  c < data.length;  ++c) {
        if (c % columns == 0 && c > 0) {
          sb.append("\n");
        }
        sb.append(HEX[data[c] & 0xFF]);
      }
    }
    else {
      sb = new StringBuilder(2*data.length);
      for (byte b : data) {
        sb.append(HEX[b & 0xFF]);
      }
    }
    return sb.toString();
  }

  /**
   * Convert data into hex code and add it as text.
   * @param data data to add
   * @return writer for chaining
   */
  @NotNull
  public SimpleXmlWriter expandBase64(@NotNull byte[] data)
  {
    add(Base64.encodeBytes(data));
    return this;
  }

  /**
   * Writes an empty element.
   * @param tag         element tag
   * @param attributes  attributes, always in pairs of attribute name
   *                    and attribute value, values will be escaped
   * @return writer for chaining
   */
  @NotNull
  public SimpleXmlWriter element(@NotNull String tag,
                                 Object ... attributes)
  {
    writeElement(false, tag, attributes);
    return this;
  }

  /**
   * Add text outside of tags.
   * @param text text to add
   * @return writer for chaining
   */
  @NotNull
  public SimpleXmlWriter text(@NotNull Object text)
  {
    return quote(text);
  }

  /**
   * Add formatted text outside of tags.
   * This will work the same as {@link String#format(String, Object...)}
   * using the {@link #getLocale() locale} set via the constructor.
   * @param format format string
   * @param args arguments for the format string
   * @return writer for chaining
   */
  @NotNull
  public SimpleXmlWriter textf(@NotNull String format, @NotNull Object... args)
  {
    return text(String.format(locale, format, args));
  }

  /**
   * Add a newline.
   * @return writer for chaining
   */
  @NotNull
  public SimpleXmlWriter nl()
  {
    return add("\n");
  }

  /**
   * Outputs the raw text as CDATA section.
   * @param text text to be output, not escaped in any form
   * @return writer for chaining
   */
  @NotNull
  public SimpleXmlWriter cdata(@NotNull String text)
  {
    // no autowrao here, which is questionable
    return add("<![CDATA[").add(text).add("]]>");
  }

  /**
   * Close all open elements.
   */
  public void finish()
  {
    while (!openElements.isEmpty()) {
      close();
    }
  }

  /**
   * Write an element, either open or closed.
   * @param open        if {@code true} further elements will become sub elements of this one,<br>
   *                    otherwise siblings
   * @param tag         element tag
   * @param attributes  attributes, attributes of element, have to come in name/value pairs
   * @return writer for chaining
   */
  @NotNull
  private SimpleXmlWriter writeElement(boolean open,
                                       @NotNull String tag,
                                       @NotNull Object[] attributes)
  {
    if (attributes.length % 2 == 1) {
      throw new IllegalArgumentException("Need an even number of attributes");
    }
    insertAutoWrap(open
                           ? openElements.size() - 1
                           : openElements.size());
    add("<").quote(tag);
    for (int a = 0;  a < attributes.length;  a += 2) {
      add(" ").quote(attributes[a]).add("=\"").quote(attributes[a + 1]).add("\"");
    }
    return add(open ? ">" : "/>");
  }

  /**
   * Add a raw text.
   * @param str text to add, callers have to ensure that it is correctly quoted
   * @return writer for chaining
   */
  @NotNull
  private SimpleXmlWriter add(@NotNull String str)
  {
    collector.add(str);
    return this;
  }

  /**
   * Add text which is quoted before it's added.
   * @param object object to add
   * @return writer for chaining
   */
  @NotNull
  private SimpleXmlWriter quote(@Nullable Object object)
  {
    final String s = object == null ? "null" : object.toString();
    final int len = s.length();
    for (int c = 0;  c < len;  ++c) {
      char ch = s.charAt(c);
      switch (ch) {
      case '<':
        collector.add("&lt;");
        break;
      case '>':
        collector.add("&gt;");
        break;
      case '"':
        collector.add("&quot;");
        break;
      case '&':
        collector.add("&amp;");
        break;
      default:
        collector.add(ch);
      }
    }
    return this;
  }
}
