// ============================================================================
// File:               Template
//
// Project:            CAFF
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   © 2022-2024  Rammi (rammi@caff.de)
//                     The usage of this source code in commercial or open 
//                     source projects is not allowed without explicit 
//                     permission.
//
// Created:            12/12/22 2:06 PM
//=============================================================================
package de.caff.util.templ;

import de.caff.annotation.NotNull;
import de.caff.generics.*;

import java.text.ParseException;
import java.util.*;
import java.util.function.Consumer;
import java.util.regex.Pattern;

/**
 * A simple class for string substitution in templates.
 * <p>
 * Templates are just defined by strings which contain placeholders.
 * Placeholders may contain defaults which are used when no user-defined
 * value is provided.
 * </p>
 * <ul>
 *   <li>General text is just text. I.e., anything that is not a placeholder.</li>
 *   <li>
 *     <b>Normal placeholders</b> have the form {@code %{name}} where name is at
 *     least one character long and consists of ASCII letters, decimal numbers,
 *     colons, double colons, underscores, or plus and minus signs.
 *   </li>
 *   <li>
 *       <b>Placeholders with defaults</b> have the form {@code %{name|@default@}}, where
 *       the rules for name are the same as for normal placeholders. The vertical bar
 *       indicates a default, which itself is enclosed in any letter of your choice as a marker
 *       (here for example {@code @}) is used. This allows the default text to contain any letters,
 *       you have to take care to enclose it in a letter which is not contained
 *       in combination with a directly following closing brace. So the above example could also
 *       contain email addresses without problems. The substitution test extent over
 *       multiple lines.
 *     <p>
 *       For nicer optics there are a few markers that expect a different end marker (here
 *       {@code ...} stands for the fallback text):
 *       <ul>
 *         <li>{@code %{name|<...>}}</li>
 *         <li>{@code %{name|(...)}}</li>
 *         <li>{@code %{name|[...]}}</li>
 *         <li>{@code %{name|{...}}}</li>
 *       </ul>
 *       and this also works the other way round, e.g. {@code %{name|>...<}}.
 *   </li>
 * </ul>
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since December 12, 2022
 */
public class Template
{
  private static final Indexable<CharPair> COMBI_DELIMITERS = Indexable.viewArray(
    new CharPair("<>"),
    new CharPair("()"),
    new CharPair("[]"),
    new CharPair("{}")
  );
  private static final Map<Character, Character> DELIMITER_MAPPING = new HashMap<>();
  static {
    COMBI_DELIMITERS.forEach(pair -> {
      DELIMITER_MAPPING.put(pair.first, pair.second);
      DELIMITER_MAPPING.put(pair.second, pair.first);
    });
  }
  /** Prefix of a text holder mark. */
  static final String PLACEHOLDER_PREFIX = "%{";
  /** Suffix of a text holder mark. */
  static final String PLACEHOLDER_SUFFIX = "}";
  /** The separator used to include a fallback. */
  static final String FALLBACK_SEPARATOR = "|";
  /** Pattern defining a valid name. */
  private static final Pattern VALID_NAME_PATTERN = Pattern.compile("[-0-9A-Za-z_+.!:]+");
  /** The text items. */
  @NotNull
  private final List<TextItem> textItems;
  /** Required placeholder names. */
  @NotNull
  private final Countable<String> varNames;

  /**
   * Constructor.
   * @param textItems  text items
   */
  private Template(@NotNull List<TextItem> textItems)
  {
    this.textItems = textItems;
    final Set<String> names = new HashSet<>();
    textItems.forEach(item -> item.registerName(names::add));
    varNames = Countable.viewCollection(names);
  }

  /**
   * Get the set of placeholder names which are used in this template.
   * @return required placeholder names, optional placeholders are not contained
   */
  @NotNull
  public Countable<String> getPlaceholders()
  {
    return varNames;
  }

  /**
   * Create a template.
   * @param templateText template text
   * @return template
   * @throws ParseException if text does contain invalid placeholder definitions
   */
  @NotNull
  public static Template from(@NotNull String templateText) throws ParseException
  {
    return new Template(parse(templateText));
  }

  /**
   * Apply the given variable mapping and return the result.
   * @param variableMapping variable mapping, has to contain a key for each name
   * @return text with placeholders replaced by the values given by {@code variableMapping}
   */
  @NotNull
  public String apply(@NotNull Dict<String, String> variableMapping)
  {
    varNames.forEach(name -> {
      if (!variableMapping.hasKey(name)) {
        throw new IllegalArgumentException(String.format("variableMapping does not provide a mapping for %s%s%s!",
                                                         PLACEHOLDER_PREFIX, name, PLACEHOLDER_SUFFIX));
      }
    });
    final StringBuilder sb = new StringBuilder();
    textItems.forEach(item -> sb.append(item.getText(variableMapping)));
    return sb.toString();
  }

  /**
   * Apply the given variable mapping and return the result.
   * @param variableMapping variable mapping, has to contain a key for each name
   * @return text with placeholders replaced by the values given by {@code variableMapping}
   */
  @NotNull
  public String apply(@NotNull Map<String, String> variableMapping)
  {
    return apply(Dict.viewMap(variableMapping));
  }

  /**
   * Apply the given name-variable mapping and return the result.
   * @param name1  name of first placeholder
   * @param value1 substitution value of first placeholder
   * @param moreNameValuePairs further name-value pairs, each even entry has to be a non-null string, and
   *                           each following odd entry has to be something non-null, for which its string
   *                           representation will be used as substitution value
   * @return text with placeholders replaced
   */
  @NotNull
  public String apply(@NotNull String name1, @NotNull Object value1, @NotNull Object ... moreNameValuePairs)
  {
    if (moreNameValuePairs.length % 2 == 1) {
      throw new IllegalArgumentException("moreNameValuePairs has to have an even size!");
    }
    final Map<String, String> map = new HashMap<>();
    map.put(name1, value1.toString());
    for (int idx = 0;  idx < moreNameValuePairs.length;  idx += 2) {
      if (!(moreNameValuePairs[idx] instanceof String)) {
        throw new IllegalArgumentException("The keys in moreNameValuePairs have to be Strings!");
      }
      map.put(moreNameValuePairs[idx].toString(),
              Objects.toString(Objects.requireNonNull(moreNameValuePairs[idx + 1])));
    }
    return apply(map);
  }

  /**
   * Item which generates text.
   */
  private interface TextItem
  {
    @NotNull
    String getText(@NotNull Dict<String, String> variableMapping);

    void registerName(@NotNull Consumer<String> nameCollector);
  }

  /**
   * Check a name that it does not contain invalid characters.
   * @param name name to check
   * @throws IllegalArgumentException if name contains invalid characters
   */
  public static void checkName(@NotNull String name)
  {
    if (!VALID_NAME_PATTERN.matcher(name).matches()) {
      throw new IllegalArgumentException(String.format("Not a valid name: \"%s\"!",  name));
    }
  }

  /**
   * Parse a template text definition into a list of text items.
   * @param templateText template text
   * @return text items
   * @throws ParseException on parse errors
   */
  @NotNull
  private static List<TextItem> parse(@NotNull String templateText)
          throws ParseException
  {
    final List<TextItem> result = new LinkedList<>();
    int pos = 0;
    while (pos < templateText.length()) {
      final int startSubst = templateText.indexOf(PLACEHOLDER_PREFIX, pos);
      if (startSubst < 0) {
        result.add(new Text(templateText.substring(pos)));
        break;
      }
      final int endSubst = templateText.indexOf(PLACEHOLDER_SUFFIX, startSubst + PLACEHOLDER_PREFIX.length());
      if (endSubst < 0) {
        throw new ParseException(String.format("Couldn't find substitution end marker for substitution starting at %s!", toPosition(templateText, startSubst)),
                                 startSubst);
      }
      if (pos != startSubst) {
        result.add(new Text(templateText.substring(pos, startSubst)));
      }
      final String phName = templateText.substring(startSubst + PLACEHOLDER_PREFIX.length(), endSubst);
      final int fbPos = phName.indexOf(FALLBACK_SEPARATOR);
      if (fbPos <= 0) {
        // no fallback
        result.add(new PlaceHolder(phName));
        pos = endSubst + PLACEHOLDER_SUFFIX.length();
      }
      else if (fbPos >= phName.length() - 2) {
        throw new ParseException(String.format("Invalid placeholder fallback at %s, need start and end marker!",
                                               toPosition(templateText, pos + 2 + fbPos)),
                                 pos + 2 + fbPos);
      }
      else {
        // w/ fallback
        final char marker = phName.charAt(fbPos + 1);
        final String endMarker = DELIMITER_MAPPING.getOrDefault(marker, marker) + PLACEHOLDER_SUFFIX;
        final int endOfPH = templateText.indexOf(endMarker, pos + PLACEHOLDER_SUFFIX.length() + fbPos + 2);
        if (endOfPH < 0) {
          throw new ParseException(String.format("Invalid placeholder fallback starting at %s, did not find end marker \"%s\"!",
                                                 toPosition(templateText, pos + PLACEHOLDER_SUFFIX.length() + fbPos + 2),
                                                 endMarker),
                                   pos + PLACEHOLDER_SUFFIX.length() + fbPos + 2);
        }
        result.add(new OptionalPlaceHolder(templateText.substring(startSubst + PLACEHOLDER_PREFIX.length(),
                                                                  startSubst + PLACEHOLDER_PREFIX.length() + fbPos),
                                           templateText.substring(startSubst + PLACEHOLDER_PREFIX.length() + fbPos + 2,
                                                                  endOfPH)));
        pos = endOfPH + PLACEHOLDER_PREFIX.length();
      }
    }
    return result;
  }

  /**
   * Calculate the position (line:char) from a text offset.
   * @param text   text, possibly multi-line
   * @param offset linear offset into text
   * @return offset as position line:char, e.g. {@code 2:1} for the 1st character of the second line
   */
  @NotNull
  private static String toPosition(@NotNull String text, int offset)
  {
    final String[] lines  = text.substring(0, offset).split("\n");
    return String.format("%d:%d", lines.length + 1, lines[lines.length - 1].length() + 1);
  }

  /** Raw text. */
  private static class Text
          implements TextItem
  {
    @NotNull
    private final String text;

    /**
     * Constructor.
     * @param text raw text
     */
    Text(@NotNull String text)
    {
      this.text = text;
    }

    @NotNull
    @Override
    public String getText(@NotNull Dict<String, String> variableMapping)
    {
      return text;
    }

    @Override
    public void registerName(@NotNull Consumer<String> nameCollector)
    {
      // no associated name
    }

    @Override
    @NotNull
    public String toString()
    {
      return text;
    }
  }

  /** Placeholder. */
  private static class PlaceHolder
          implements TextItem
  {
    @NotNull
    private final String name;

    /**
     * A substitution.
     * @param name variable name, must be at least 1 character long, and has to contain only
     *             ASCII letters, decimal digits, underscore, plus, minus, colon, or double colon
     */
    PlaceHolder(@NotNull String name)
    {
      checkName(name);
      this.name = name;
    }

    @NotNull
    @Override
    public String getText(@NotNull Dict<String, String> variableMapping)
    {
      return variableMapping.getNonNull(name);
    }

    @Override
    public void registerName(@NotNull Consumer<String> nameCollector)
    {
      nameCollector.accept(name);
    }

    @Override
    @NotNull
    public String toString()
    {
      return PLACEHOLDER_PREFIX + name + PLACEHOLDER_SUFFIX;
    }
  }

  /**
   * Placeholder with fallback.
   * Usually a name with a fallback value, but also allowing
   * an empty name to make this always provide the fallback.
   */
  private static class OptionalPlaceHolder
          implements TextItem
  {
    @NotNull
    private final String name;
    @NotNull
    private final Template fallback;

    OptionalPlaceHolder(@NotNull String name, @NotNull String fallback) throws ParseException
    {
      checkName(name);
      this.name = name;
      this.fallback = from(fallback);
    }

    @NotNull
    @Override
    public String getText(@NotNull Dict<String, String> variableMapping)
    {
      return name.isEmpty()
              ? fallback.apply(variableMapping)
              : variableMapping.getOr(name,
                                      nm -> fallback.apply(variableMapping));
    }

    @Override
    public void registerName(@NotNull Consumer<String> nameCollector)
    {
      // not registered as name is not required to be provided
    }

    @Override
    @NotNull
    public String toString()
    {
      return PLACEHOLDER_PREFIX + name + FALLBACK_SEPARATOR + fallback + PLACEHOLDER_SUFFIX;
    }
  }
}
