// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2005-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================

package de.caff.util.startup;

import com.sun.management.OperatingSystemMXBean;
import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.Types;
import de.caff.io.InputStreamPipe;
import de.caff.util.Utility;
import de.caff.util.debug.Debug;
import de.caff.util.debug.SimpleOutputtingDebugListener;

import java.io.*;
import java.lang.management.ManagementFactory;
import java.util.*;
import java.util.regex.Pattern;

import static de.caff.util.debug.Debug.warn;

/**
 * Restart the JVM with better settings.
 * This class circumvents the restrictions of the original JVM started by eg
 * double-clicking on a jar file, in which case the JVM starts with default
 * memory settings which are often restricted too much for complex programs.
 * <p>
 * You can influence the behavior by setting Java properties;
 * <ul>
 *   <li>
 *     {@code restart.max.32bit}: maximum size used for 32bit JVMs. When I first
 *     implemented the restarter, {@code 1700000000} (1.7GB) was a good value. But
 *     modern JVMs seem to use more memory for themselves, so nowadays the default
 *     used is only {@code 1400000000} (1.4GB). Switch to a 64bit JVM if possible.
 *   </li>
 *   <li>
 *     {@code debug.restart}: if set to {@code true}, additional debugging information is
 *     print to console.
 *   </li>
 *   <li>
 *     {@code no.restart}: if set to {@code true}, restarting is not done. This can be
 *     useful when you debug your application, because debuggers connect to the original
 *     app, not the restarted one.
 *   </li>
 *   <li>
 *     {@code restart.mem.part}: this defines the default part of the memory used for
 *     the application. This default is used when no method explicitly defining this
 *     part (parameter {@code memPart}) is used. Values are between {@code 0.0} and {@code 1.0}.
 *   </li>
 * </ul>
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public final class Restarter
{
  /** Property set by the restarter when starting a child process. */
  public static final String PROPERTY_RESTARTED = "de.caff.util.startup.Restarted";
  /** Property which can be set to {@code "true"} in order to don't restart under any circumstances. */
  public static final String PROPERTY_DONT_RESTART = "no.restart";
  /** The Java home property. */
  public static final String PROPERTY_JAVA_HOME = "java.home";
  /** The Java class path property. */
  public static final String PROPERTY_JAVA_CLASSPATH = "java.class.path";
  /** The Java virtual machine name property. */
  public static final String PROPERTY_JAVA_VM_NAME = "java.vm.name";
  /** Max size to use for 32bit JVMs ({@code restart.max.32bit}. */
  public static final long MAX_32BIT_SIZE = Utility.getLongParameter("restart.max.32bit", 1_400_000_000L);
  /**
   * Default part of memory to reserve for the application.
   * Can be set via Java property {@code restart.mem.part} to something between {@code 0.0} (no memory) and {@code 1.0} (all memory).
   * If property is not defined or has an insane value {@code 0.75} is used.
   */
  public static final double DEFAULT_MEMORY_PART;
  static {
    double value = Utility.getDoubleParameter("restart.mem.part", 0.75);
    if (value <= 0.0  ||  value > 1.0) {
      value = 0.75;
    }
    DEFAULT_MEMORY_PART = value;
  }
  /** Whether to debug the restart process to the console. */
  private static final boolean DEBUG = Utility.getBooleanParameter("debug.restart", false);

  /** Properties to exclude from forwarding. */
  private static final String[] SYSTEM_PREFIXES = {
          "file.",
          "java.",
          "line.separator",
          "os.",
          "path.separator",
          "sun.",
          "user."
  };
  /** Pattern for looking for 64bit JVM in virtual machine name. */
  private static final Pattern PATTERN_64BIT = Pattern.compile("64.?bit");
  /** Pattern for formatting properties. */
  static final String PROPERTY_FORMAT = "-D%s=%s";

  /**
   * Don't create this.
   */
  private Restarter() {}

  /**
   * Possibly restart the program using some 75% of available memory.
   * <p>
   * This method only returns if it did not restart the program,
   * either because the program was already restarted, or because
   * there were errors which kept the program from being restarted.
   * @param mainClass main class to start
   * @param args      program arguments
   * @return {@code true} if the program was already restarted, or there are no better memory settings available<br>
   *         {@code false} if there are problems keeping the program from being restarted
   */
  public static boolean possiblyRestart(@NotNull Class<?> mainClass, String ... args)
  {
    return possiblyRestart(mainClass.getName(), DEFAULT_MEMORY_PART, args);
  }

  /**
   * Possibly restart the program using improved memory settings.
   * <p>
   * This method only returns if it did not restart the program,
   * either because the program was already restarted, or because
   * there were errors which kept the program from being restarted.
   * @param mainClass main class to start
   * @param memoryPart part of physical memory to use
   * @param args      program arguments
   * @return {@code true} if the program was already restarted, or there are no better memory settings available<br>
   *         {@code false} if there are problems keeping the program from being restarted
   */
  public static boolean possiblyRestart(@NotNull Class<?> mainClass, double memoryPart, String ... args)
  {
    return possiblyRestart(mainClass.getName(), memoryPart, args);
  }

  /**
   * Possibly restart the program using some 75% of available memory..
   * <p>
   * This method only returns if it did not restart the program,
   * either because the program was already restarted, or because
   * there were errors which kept the program from being restarted.
   * @param mainClassName full name of main class to start
   * @param args      program arguments
   * @return {@code true} if the program was already restarted, or there are no better memory settings available<br>
   *         {@code false} if there are problems keeping the program from being restarted
   */
  public static boolean possiblyRestart(@NotNull String mainClassName, String ... args)
  {
    return possiblyRestart(mainClassName, DEFAULT_MEMORY_PART, args);
  }

  /**
   * Possibly restart the program using improved memory settings.
   * <p>
   * This method only returns if it did not restart the program,
   * either because the program was already restarted, or because
   * there were errors which kept the program from being restarted.
   * @param mainClassName full name of main class to start
   * @param memoryPart part of physical memory to use
   * @param args      program arguments
   * @return {@code true} if the program was already restarted, or there are no better memory settings available<br>
   *         {@code false} if there are problems keeping the program from being restarted
   */
  public static boolean possiblyRestart(@NotNull String mainClassName, double memoryPart, String ... args)
  {
    final SimpleOutputtingDebugListener debugListener = DEBUG ? new SimpleOutputtingDebugListener(System.out) : null;
    try {
      if (debugListener != null) {
        Debug.addCookedMessageDebugListener(debugListener);
        Debug.setMask(Debug.DEBUG_ALL_MASK);
      }
      if (Utility.getBooleanParameter(PROPERTY_DONT_RESTART, false)) {
        Debug.warn("Property %0 was set, not restarting.", PROPERTY_DONT_RESTART);
        return false;
      }
      try {
        if (Utility.getBooleanParameter(PROPERTY_RESTARTED, false)) {
          Debug.message("Property %0 was set, application was already restarted.", PROPERTY_RESTARTED);
          return true;
        }
        Runtime runtime = Runtime.getRuntime();
        long maxMem = runtime.maxMemory();
        long physMem = getPhysicalMemorySize();
        if (physMem <= 0L) {
          return false;
        }
        long requiredMem = (long)(memoryPart * physMem);
        if (!is64BitJVM()) {
          // we could run a small test using nested intervals here, but basically we need to keep the max memory low enough
          requiredMem = Math.min(requiredMem, MAX_32BIT_SIZE);
        }
        Debug.message("Trying to require "+requiredMem+" bytes...");
        if (requiredMem <= maxMem) {
          return true;
        }
        List<String> commandLine = new LinkedList<>();
        String javaCmd = getJavaCommand();
        if (javaCmd == null) {
          return false;
        }
        commandLine.add(javaCmd);
        commandLine.add(String.format("-mx%d", requiredMem));
        commandLine.add("-classpath");
        commandLine.add(System.getProperty(PROPERTY_JAVA_CLASSPATH));
        addSystemProperties(commandLine);
        commandLine.add(String.format(PROPERTY_FORMAT, PROPERTY_RESTARTED, "true"));
        commandLine.add(mainClassName);
        commandLine.addAll(Types.asList(args));
        Debug.message("Command line: %0", Types.join(" ", commandLine));
        ProcessBuilder processBuilder = new ProcessBuilder(commandLine);
        Process process = processBuilder.start();
        InputStreamPipe.startOn(process.getInputStream(), System.out);
        InputStreamPipe.startOn(process.getErrorStream(), System.err);
        final int status = process.waitFor();
        if (status == 0) {
          System.exit(status);
        }
        else {
          warn("Original process exited with status %0", status);
        }
      } catch (Throwable x) {
        // if anything goes wrong, debug that, but otherwise just do nothing
        Debug.error(x);
      }
      return false;
    } finally {
      if (debugListener != null) {
        Debug.removeCookedMessageDebugListener(debugListener);
      }
    }
  }

  /**
   * Add the system properties to the command line arguments.
   * @param commandLine command line arguments
   */
  public static void addSystemProperties(@NotNull Collection<String> commandLine)
  {
    outer:
    for (String propName : System.getProperties().stringPropertyNames()) {
      for (String prefix : SYSTEM_PREFIXES) {
        if (propName.startsWith(prefix)) {
          continue outer;
        }
      }
      commandLine.add(String.format(PROPERTY_FORMAT, propName, System.getProperty(propName)));
    }
  }

  /**
   * Get the amount of physical memory on this machine.
   * @return the amount of physical memory in bytes, or {@code 0L} if the amount is unknown
   */
  public static long getPhysicalMemorySize()
  {
    try {
      // this may result in a class cast exception, but currently there is no other simple way to get the physical memory size
      final OperatingSystemMXBean operatingSystemMXBean = (OperatingSystemMXBean)ManagementFactory.getOperatingSystemMXBean();
      return operatingSystemMXBean.getTotalPhysicalMemorySize();
    } catch (ClassCastException x) {
      Debug.error(x);
    }
    return 0;
  }

  /**
   * Get the java command with which the JVM was started.
   * @return java command or {@code null} if the command couldn't be determined
   */
  @Nullable
  public static String getJavaCommand()
  {
    String javaHome = System.getProperty(PROPERTY_JAVA_HOME);
    if (javaHome == null) {
      return null;
    }
    final String javaBinDir = javaHome + File.separatorChar + "bin";
    String javaBin = javaBinDir + File.separatorChar + "java";
    if (Utility.areWeOnWindows()) {
      javaBin += ".exe";
    }
    final File javaBinFile = new File(javaBin);
    if (javaBinFile.exists()  &&  javaBinFile.canExecute()) {
      return javaBin;
    }
    // try files which start with java.
    final File binDir = new File(javaBinDir);
    if (!binDir.exists() || !binDir.isDirectory()) {
      // no idea
      return null;
    }
    final FilenameFilter filter = (dir, name) -> name.startsWith("java.") && name.lastIndexOf('.') == 4;
    for (File file : Objects.requireNonNull(binDir.listFiles(filter))) {
      if (file.canExecute()) {
        return file.getPath();
      }
    }
    return null;
  }

  /**
   * Check whether the JVM running this class is 64bit.
   * This is using a heuristic check, so it is not completely reliable.
   * Basically it should err that it assumes that an actual 64bit machine is just 32bit.
   * @return {@code true} if this is a 64bit machine,<br>
   *         {@code false} if this is a 32bit machine
   */
  public static boolean is64BitJVM()
  {
    final String jvm = System.getProperty(PROPERTY_JAVA_VM_NAME);
    return jvm != null && PATTERN_64BIT.matcher(jvm.toLowerCase()).find();
  }

  /**
   * Test code.
   * @param args arguments (unused, but should be passed through)
   */
  public static void main(String[] args)
  {
    Debug.installCookedOutput();
    Debug.setMask(Debug.DEBUG_ALL_MASK - Debug.TRACE_FLAG);

    System.out.println("args: "+ Arrays.toString(args));

    Runtime runtime = Runtime.getRuntime();
    long maxMem = runtime.maxMemory();
    System.out.println("max mem: "+maxMem);
    Restarter.possiblyRestart(Restarter.class, 0.95, args);
    System.out.println("Running on ... and exit");
    System.exit(0);
  }
}
