// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2005-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;

import java.awt.*;
import java.util.prefs.Preferences;

/**
 *  A preference property which describes a dimension.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class SizePreferenceProperty
        extends AbstractBasicChangeableItem
        implements PreferenceProperty
{
  private static final String EXT_WIDTH  = "-WIDTH";
  private static final String EXT_HEIGHT = "-HEIGHT";
  private static final long serialVersionUID = -6294165270594094306L;

  private Dimension dimension;

  public SizePreferenceProperty(String basicName)
  {
    super(basicName);
  }

  /**
   * Read the property value from the preferences.
   *
   * @param preferences preferences from where to read the property value
   */
  @Override
  public void readFrom(@NotNull Preferences preferences)
  {
    final String prefix = getBasicName();
    int w = preferences.getInt(prefix+EXT_WIDTH,  -1);
    int h = preferences.getInt(prefix+EXT_HEIGHT, -1);

    if (w >= 0  &&  h >= 0) {
      setSize(new Dimension(w, h));
    }
    else {
      setSize((Dimension)null);
    }
  }

  /**
   * Store the current property value in the preferences.
   *
   * @param preferences preferences where to store the property value
   */
  @Override
  public void storeTo(@NotNull Preferences preferences)
  {
    final String prefix = getBasicName();
    if (dimension == null) {
      preferences.remove(prefix+EXT_WIDTH);
      preferences.remove(prefix+EXT_HEIGHT);
    }
    else {
      preferences.putInt(prefix+EXT_WIDTH,  dimension.width);
      preferences.putInt(prefix+EXT_HEIGHT, dimension.height);
    }
  }

  /**
   *  Set the size of the given component.
   *  @param comp component which size should be set
   *  @return {@code true} if the size of the component is set,
   *          otherwise {@code false}
   */
  public boolean setComponentSize(@NotNull Component comp)
  {
    if (dimension != null) {
      comp.setSize(dimension.width, dimension.height);
      return true;
    }
    return false;
  }

  /**
   *  Get the size of the component and store them in the preferences.
   *  @param preferences preferences to store to
   *  @param comp component which size is stored
   */
  public void storeComponentSize(@NotNull Preferences preferences, @NotNull Component comp)
  {
    dimension = comp.getSize();
    storeTo(preferences);
  }

  /**
   *  Set the dimesnion directly.
   *  @param dim new dimension
   */
  public void setSize(@Nullable Dimension dim)
  {
    if (dim == null) {
      if (dimension != null) {
        Dimension oldDim = dimension;
        dimension = null;
        fireValueChange(getBasicName(), oldDim, dim);
      }
    }
    else {
      if (!dim.equals(dimension)) {
        Dimension oldDim = dimension;
        dimension = new Dimension(dim);
        fireValueChange(getBasicName(), oldDim, dim);
      }
    }
  }

  /**
   *  Set the dimension directly.
   *  @param comp component from which to set the dimension
   */
  public void setSize(@NotNull Component comp)
  {
    setSize(comp.getSize());
  }

}
