// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2016-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.UniformMatcher;

import java.util.prefs.Preferences;

/**
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 27, 2016
 */
public class SimpleStringPreferenceProperty
        extends SimpleNullableProperty<String>
        implements StringPreferenceProperty
{
  private static final long serialVersionUID = 5950816329017671141L;

  /**
   * Constructor.
   * This will use a matcher using the {@link #equals(Object)} method
   * and an initial value of {@code null}.
   *
   * @param basicName property name
   */
  public SimpleStringPreferenceProperty(@NotNull String basicName)
  {
    super(basicName);
  }

  /**
   * Constructor.
   * This will use an matcher using the {@link #equals(Object)} method.
   *
   * @param basicName property name
   * @param value     initial value
   */
  public SimpleStringPreferenceProperty(@NotNull String basicName, @Nullable String value)
  {
    super(basicName, value);
  }

  /**
   * Constructor.
   * This will use an initial value of {@code null}.
   *
   * @param basicName property name
   * @param matcher   matcher which defines whether a newly set value differs from the current value
   */
  public SimpleStringPreferenceProperty(@NotNull String basicName,
                                        @NotNull UniformMatcher<? super String> matcher)
  {
    super(basicName, matcher);
  }

  /**
   * Constructor.
   *
   * @param basicName property name
   * @param value     initial value
   * @param matcher   matcher which defines whether a newly set value differs from the current value
   */
  public SimpleStringPreferenceProperty(@NotNull String basicName, @Nullable String value,
                                        @NotNull UniformMatcher<? super String> matcher)
  {
    super(basicName, value, matcher);
  }

  /**
   * Read the property value from the preferences.
   *
   * @param preferences preferences from where to read the property value
   */
  @Override
  public void readFrom(@NotNull Preferences preferences)
  {
    setValue(preferences.get(getBasicName(), getValue()));
  }

  /**
   * Store the current property value in the preferences.
   *
   * @param preferences preferences where to store the property value
   */
  @Override
  public void storeTo(@NotNull Preferences preferences)
  {
    final String value = getValue();
    if (value == null) {
      preferences.remove(getBasicName());
    }
    else {
      preferences.put(getBasicName(), value);
    }
  }
}
