// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2016-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings;

import de.caff.annotation.NotNull;
import de.caff.generics.Types;

import java.beans.PropertyChangeListener;
import java.util.ArrayList;
import java.util.List;
import java.util.prefs.Preferences;

/**
 * A simple grouping of preferences properties.
 * It will forward everything to the collected preference properties.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 27, 2016
 */
public class SimplePreferenceGroup
        implements PreferenceProperty
{
  @NotNull
  private final List<PreferenceProperty> properties;

  /**
   * Constructor.
   * @param properties preference properties in this group
   */
  public SimplePreferenceGroup(@NotNull List<PreferenceProperty> properties)
  {
    this.properties = new ArrayList<>(properties);
  }

  /**
   * Constructor.
   * @param properties prefe
   */
  public SimplePreferenceGroup(PreferenceProperty ... properties)
  {
    this.properties = Types.asList(properties);
  }

  /**
   * Add a listener which will be called for all value changes.
   *
   * @param listener value change listener to add
   */
  @Override
  public void addValueChangeListener(@NotNull PropertyChangeListener listener)
  {
    for (PreferenceProperty property : properties) {
      property.addValueChangeListener(listener);
    }
  }

  /**
   * Remove a listener which will be called for all value changes.
   *
   * @param listener value change listener to remove
   */
  @Override
  public void removeValueChangeListener(@NotNull PropertyChangeListener listener)
  {
    for (PreferenceProperty property : properties) {
      property.removeValueChangeListener(listener);
    }
  }

  /**
   * Read the property value from the preferences.
   *
   * @param preferences preferences from where to read the property value
   */
  @Override
  public void readFrom(@NotNull Preferences preferences)
  {
    for (PreferenceProperty property : properties) {
      property.readFrom(preferences);
    }
  }

  /**
   * Store the current property value in the preferences.
   *
   * @param preferences preferences where to store the property value
   */
  @Override
  public void storeTo(@NotNull Preferences preferences)
  {
    for (PreferenceProperty property : properties) {
      property.storeTo(preferences);
    }
  }

  /**
   * Add a listener which will be called for all value changes.
   * The listener is registered with a weak reference, allowing it to be freed if
   * this is the only reference.
   * <p>
   * If you want to remove the listener please use the corresponding
   * {@link #removeValueChangeListenerWeakly(PropertyChangeListener)}
   * method.
   *
   * @param listener value change listener to add
   * @see #addValueChangeListener(PropertyChangeListener)
   */
  @Override
  public void addValueChangeListenerWeakly(@NotNull PropertyChangeListener listener)
  {
    for (PreferenceProperty property : properties) {
      property.addValueChangeListenerWeakly(listener);
    }
  }

  /**
   * Remove a weakly referenced listener which will be called for all value changes.
   *
   * @param listener value change listener to remove
   * @see #removeValueChangeListener(PropertyChangeListener)
   */
  @Override
  public void removeValueChangeListenerWeakly(@NotNull PropertyChangeListener listener)
  {
    for (PreferenceProperty property : properties) {
      property.removeValueChangeListenerWeakly(listener);
    }
  }
}
