// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2016-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.UniformMatcher;
import de.caff.generics.matcher.Match;

/**
 * A simple property which cannot be {@code null}.
 * <p>
 * You can define a specific matcher which allows to define when a value will be changed
 * in the {@link #setValue(Object)} method.
 * The default is to use the {@link #equals(Object)} method.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 27, 2016
 */
public class SimpleNullableProperty<T>
        extends AbstractBasicChangeableItem
        implements Property<T>
{
  private static final long serialVersionUID = -6245653244996622537L;

  @Nullable
  private T value;

  @NotNull
  private UniformMatcher<? super T> matcher;

  /**
   * Constructor.
   * This will use a matcher using the {@link #equals(Object)} method
   * and an initial value of {@code null}.
   *
   * @param basicName property name
   */
  public SimpleNullableProperty(@NotNull String basicName)
  {
    this(basicName, null, Match.DEFAULT_MATCHER);
  }

  /**
   * Constructor.
   * This will use an matcher using the {@link #equals(Object)} method.
   * @param basicName property name
   * @param value     initial value
   */
  public SimpleNullableProperty(@NotNull String basicName, @Nullable T value)
  {
    this(basicName, value, Match.DEFAULT_MATCHER);
  }

  /**
   * Constructor.
   * This will use an initial value of {@code null}.
   *
   * @param basicName property name
   * @param matcher   matcher which defines whether a newly set value differs from the current value
   */
  public SimpleNullableProperty(@NotNull String basicName, @NotNull UniformMatcher<? super T> matcher)
  {
    this(basicName, null, matcher);
  }

  /**
   * Constructor.
   * @param basicName property name
   * @param value     initial value
   * @param matcher   matcher which defines whether a newly set value differs from the current value
   */
  public SimpleNullableProperty(@NotNull String basicName, @Nullable T value, @NotNull UniformMatcher<? super T> matcher)
  {
    super(basicName);
    this.value = value;
    this.matcher = matcher;
  }

  /**
   * Get the value of the property.
   *
   * @return property value
   */
  @Override
  @Nullable
  public T getValue()
  {
    return value;
  }

  /**
   * Set the value.
   *
   * @param value new value
   */
  @Override
  public void setValue(@Nullable T value)
  {
    if (!matcher.areEqual(value, this.value)) {
      T oldValue = this.value;
      this.value = value;
      fireValueChange(getPropertyName(), oldValue, value);
    }
  }
}
