// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2016-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;

import java.io.File;
import java.util.prefs.Preferences;

/**
 * Simple property holding a file.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 27, 2016
 */
public class SimpleDirectoryPreferenceProperty
        extends AbstractBasicChangeableItem
        implements DirectoryPreferenceProperty
{
  private static final long serialVersionUID = 4626333074555535769L;

  @Nullable
  private File value;

  /**
   * Constructor.
   * Construct a property with the given name or name prefix.
   *
   * @param basicName name or name prefix
   */
  public SimpleDirectoryPreferenceProperty(@NotNull String basicName)
  {
    super(basicName);
  }

  /**
   * Constructor.
   * @param basicName name or name prefix
   * @param directory initial directory
   */
  public SimpleDirectoryPreferenceProperty(@NotNull String basicName,
                                           @Nullable File directory)
  {
    super(basicName);
    value = directory;
  }


  /**
   * Get the directory.
   *
   * @return the directory
   */
  @Override
  public File getDirectory()
  {
    return value;
  }

  /**
   * Set the directory.
   *
   * @param directory new directory
   */
  @Override
  public void setDirectory(File directory)
  {
    if (directory != null ? !directory.equals(value) : value != null) {
      File oldDirectory = value;
      value = directory;
      fireValueChange(getBasicName(), oldDirectory, directory);
    }
  }

  /**
   * Read the property value from the preferences.
   *
   * @param preferences preferences from where to read the property value
   */
  @Override
  public void readFrom(@NotNull Preferences preferences)
  {
    String value = preferences.get(getBasicName(), null);
    if (value != null) {
      setDirectory(new File(value));
    }
  }

  /**
   * Store the current property value in the preferences.
   *
   * @param preferences preferences where to store the property value
   */
  @Override
  public void storeTo(@NotNull Preferences preferences)
  {
    if (value == null) {
      preferences.remove(getBasicName());
    }
    else {
      preferences.put(getBasicName(), value.getPath());
    }
  }
}
