// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2016-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings;

import de.caff.annotation.NotNull;

import java.util.prefs.Preferences;

/**
 * Simple preference property holding a short value.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 27, 2016
 */
public class SimpleCharPreferenceProperty
        extends SimpleCharProperty
        implements CharPreferenceProperty
{
  private static final long serialVersionUID = 9139416270591156961L;

  /**
   * Constructor.
   * @param basicName basic property name or prefix
   */
  public SimpleCharPreferenceProperty(@NotNull String basicName)
  {
    super(basicName);
  }

  /**
   * Convenience constructor.
   * @param basicName basic property name or prefix
   * @param initialValue initialValue, only the lower 8 bits are used
   *
   */
  public SimpleCharPreferenceProperty(@NotNull String basicName,
                                      int initialValue)
  {
    super(basicName, initialValue);
  }

  /**
   * Constructor.
   * Construct a property with the given name or name prefix.
   *
   * @param basicName name or name prefix
   * @param value     initial value
   */
  public SimpleCharPreferenceProperty(@NotNull String basicName, char value)
  {
    super(basicName, value);
  }

  /**
   * Read the property value from the preferences.
   *
   * @param preferences preferences from where to read the property value
   */
  @Override
  public void readFrom(@NotNull Preferences preferences)
  {
    setValue((char)preferences.getInt(getBasicName(), getValue()));
  }

  /**
   * Store the current property value in the preferences.
   *
   * @param preferences preferences where to store the property value
   */
  @Override
  public void storeTo(@NotNull Preferences preferences)
  {
    preferences.putInt(getBasicName(), getValue());
  }
}

