// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2016-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings;

import de.caff.annotation.NotNull;

/**
 * A simple property, holding a byte value.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 28, 2016
 */
public class SimpleByteProperty
        extends AbstractBasicChangeableItem
        implements Property<Byte>
{
  private static final long serialVersionUID = -1678345547647882885L;

  private byte value;

  /**
   * Constructor.
   * Construct a property with the given name or name prefix.
   *
   * @param basicName name or name prefix
   */
  public SimpleByteProperty(@NotNull String basicName)
  {
    this(basicName, 0);
  }

  /**
   * Constructor.
   * Construct a property with the given name or name prefix.
   *
   * @param basicName name or name prefix
   * @param value     initial value, only the lower 8 bits are used
   */
  public SimpleByteProperty(@NotNull String basicName,
                            int value)
  {
    this(basicName, (byte)value);
  }

  /**
   * Constructor.
   * Construct a property with the given name or name prefix.
   *
   * @param basicName name or name prefix
   * @param value     initial value, only the lower 8 bits are used
   */
  public SimpleByteProperty(@NotNull String basicName,
                            byte value)
  {
    super(basicName);
    this.value = value;
  }

  /**
   * Get the value of the property.
   *
   * @return property value
   */
  @Override
  @NotNull
  public Byte getValue()
  {
    return value;
  }

  /**
   * Set the value.
   *
   * @param value new value (not {@code null}
   */
  @Override
  public void setValue(Byte value)
  {
    if (value == null) {
      throw new IllegalArgumentException("value must not be null!");
    }
    setValue(value.byteValue());
  }


  /**
   * Set the value.
   *
   * @param value new value
   */
  public void setValue(byte value)
  {
    if (value != this.value) {
      final byte oldValue = this.value;
      this.value = value;
      fireValueChange(getPropertyName(), oldValue, value);
    }
  }

  /**
   * Set the value.
   * <p>
   * This is a convenience method.
   *
   * @param value new value, only the lower 8 bits are used
   */
  public void setValue(int value)
  {
    setValue((byte)value);
  }
}
