// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2016-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings;

import de.caff.annotation.NotNull;

/**
 * Simple property holding a boolean value.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 27, 2016
 */
public class SimpleBooleanProperty
        extends AbstractBasicChangeableItem
        implements Property<Boolean>
{
  private static final long serialVersionUID = -1387146652421540976L;
  private boolean value;

  /**
   * Constructor.
   * @param basicName property name
   * @param value     initial value
   */
  public SimpleBooleanProperty(@NotNull String basicName, boolean value)
  {
    super(basicName);
    this.value = value;
  }

  /**
   * Get the value of the property.
   *
   * @return property value
   */
  @Override
  @NotNull
  public Boolean getValue()
  {
    return value;
  }

  /**
   * Set the value.
   *
   * @param value new value (not {@code null})
   */
  @Override
  public void setValue(Boolean value)
  {
    if (value == null) {
      throw new IllegalArgumentException("value must not be null!");
    }
    setValue(value.booleanValue());
  }

  /**
   * Set the value.
   * @param value new value
   */
  public void setValue(boolean value)
  {
    if (this.value != value) {
      this.value = value;
      fireValueChange(getPropertyName(), !this.value, value);
    }
  }
}
