// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2005-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings;

import de.caff.i18n.I18n;

import java.util.Locale;
import java.util.MissingResourceException;

/**
 *  Helpful methods.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class SettingsTools
{
  private static final double[] HUMAN_STEPS = {
          0.1, 0.2, 0.25, 0.5, 1
  };

  /**
   *  Get an optional I18n string.
   *  @param tag tag
   *  @param l   locale
   *  @return i18n string or {@code null}
   */
  public static String getOptionalI18nString(String tag, Locale l)
  {
    try {
      return I18n.getString(tag, l);
    } catch (MissingResourceException e) {
      return null;
    }
  }

  /**
   * Get a step for double setters which is pleasing humans.
   * @param value calculated stepper value
   * @return stepper value which is more pleasing for humans
   */
  public static double getHumanStepNear(double value)
  {
    final double log = Math.ceil(Math.log10(value));
    final double valueNearOne = value * Math.pow(10, -log);
    double nearestStep = 0;
    double nearestDist = Double.MAX_VALUE;
    for (double step : HUMAN_STEPS) {
      double dist = Math.abs(valueNearOne - step);
      if (dist < nearestDist) {
        nearestStep = step;
        nearestDist = dist;
      }
    }
    return nearestStep * Math.pow(10, log);
  }
}
