//=============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2005-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.i18n.I18n;

import javax.swing.*;
import java.util.Locale;
import java.util.prefs.Preferences;

/**
 *  Enum value representing a locale.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class LocaleEnumValue
        implements EnumValue<Locale>
{
  /** The locale. */
  private final Locale locale;

  /**
   *  Create a locale enum value.
   *  @param locale locale to set
   */
  public LocaleEnumValue(Locale locale)
  {
    this.locale = locale;
  }

  /**
   * Get a human readable name for this enum value.
   *
   * @param l locale to use
   * @return human readable short name
   */
  @NotNull
  @Override
  public String getShortName(@Nullable Locale l)
  {
    return locale == null ?  I18n.getString("localeDefault", l)  :  locale.getDisplayName(l);
  }

  /**
   * Get a longer description which describes this value.
   *
   * @param l locale to use
   * @return human readable description or {@code null} if no description is available
   */
  @Override
  public String getLongName(@Nullable Locale l)
  {
    return null;
  }

  /**
   * Get a possible icon to display for this value.
   *
   * @param l locale to use
   * @return icon or {@code null} if no icon is available
   */
  @Override
  public Icon getIcon(@Nullable Locale l)
  {
    return null;
  }

  /**
   * Store this value to the preferences under the given key.
   *
   * @param preferences preferences to store to
   * @param key         key under which to store
   */
  @Override
  public void storeTo(@NotNull Preferences preferences, @NotNull String key)
  {
    if (locale == null) {
      preferences.remove(key);
    }
    else {
      preferences.put(key, locale.toString());
    }
  }

  /**
   * Is this value equal to the value which is stored in the preferences under the given key?
   *
   * @param preferences preferences where a enum value is stored
   * @param key         key under which the enum value is stored
   * @return {@code true} if the enum value is equal to the one stored in the preferences,
   *         {@code false} otherwise
   */
  @Override
  public boolean isEqualTo(@NotNull Preferences preferences, @NotNull String key)
  {
    String value = preferences.get(key, null);
    return (locale == null  &&  value == null)  ||  (locale != null  &&  locale.toString().equals(value));
  }

  /**
   * Get the real value which is used in this enum.
   *
   * @return the real value
   */
  @Override
  public Locale getRealValue()
  {
    return locale;
  }

  /**
   *  Get the locale.
   *  @return locale ({@code null} for default locale
   */
  public Locale getLocale()
  {
    return locale;
  }
}
