//=============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2005-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings;

import de.caff.annotation.NotNull;

import java.util.prefs.Preferences;

/**
 *  Enum value which contains an integer.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class IntegerEnumValue
        extends AbstractBasicEnumValue<Integer>
{
  private static final long serialVersionUID = -279832491184013834L;
  /** The integer value. */
  private final int value;
  /** The value as an object. */
  private final Integer realValue;

  /**
   *  Constructor.
   *  @param baseTag  basic i18n tag
   *  @param value    the real value
   */
  public IntegerEnumValue(String baseTag, int value)
  {
    super(baseTag);
    this.value = value;
    realValue = value;
  }

  /**
   * Store this value to the preferences under the given key.
   *
   * @param preferences preferences to store to
   * @param key        key under which to store
   */
  @Override
  public void storeTo(@NotNull Preferences preferences, @NotNull String key)
  {
    preferences.putInt(key, value);
  }

  /**
   * Is this value equal to the value which is stored in the preferences under the given key?
   *
   * @param preferences preferences where a enum value is stored
   * @param key        key under which the enum value is stored
   * @return {@code true} if the enum value is equal to the one stored in the preferences,
   *         {@code false} otherwise
   */
  @Override
  public boolean isEqualTo(@NotNull Preferences preferences, @NotNull String key)
  {
    return preferences.getInt(key, value+1) == value;
  }

  /**
   * Get the real value which is used in this enum.
   *
   * @return the real value
   */
  @Override
  public Integer getRealValue()
  {
    return realValue;
  }

  /**
   *  Get the integer value.
   *  @return value
   */
  public int getValue()
  {
    return value;
  }

  /**
   * Returns the value of the specified number as a {@code double}.
   * This may involve rounding.
   *
   * @return the numeric value represented by this object after conversion
   *         to type {@code double}.
   */
  @Override
  public double doubleValue()
  {
    return value;
  }

  /**
   * Returns the value of the specified number as a {@code float}.
   * This may involve rounding.
   *
   * @return the numeric value represented by this object after conversion
   *         to type {@code float}.
   */
  @Override
  public float floatValue()
  {
    return value;
  }

  /**
   * Returns the value of the specified number as an {@code int}.
   * This may involve rounding or truncation.
   *
   * @return the numeric value represented by this object after conversion
   *         to type {@code int}.
   */
  @Override
  public int intValue()
  {
    return value;
  }

  /**
   * Returns the value of the specified number as a {@code long}.
   * This may involve rounding or truncation.
   *
   * @return the numeric value represented by this object after conversion
   *         to type {@code long}.
   */
  @Override
  public long longValue()
  {
    return value;
  }
}
