//=============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2005-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;

import javax.swing.*;
import java.util.Locale;
import java.util.prefs.Preferences;

/**
 *  Description of an enum value.
 * <p>
 *  A enum value is described by a distinguished 
 *  human-readable name, maybe further description,
 *  a possibility to read it from and write it to
 *  a Preferences object and the real value which
 *  it represents.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public interface EnumValue<E>
{
  /**
   *  Get a human readable name for this enum value.
   *  @param l locale to use
   *  @return  human readable short name
   */
  @NotNull
  public String getShortName(@Nullable Locale l);

  /**
   *  Get a longer description which describes this value.
   *  @param l locale to use
   *  @return human readable description or {@code null} if no description is available
   */
  @Nullable
  public String getLongName(@Nullable Locale l);

  /**
   * Get a possible icon to display for this value.
   * @param l locale to use
   * @return icon or {@code null} if no icon is available
   */
  public Icon getIcon(@Nullable Locale l);

  /**
   *  Store this value to the preferences under the given key.
   *  @param preferences preferences to store to
   *  @param key key under which to store
   */
  public void storeTo(@NotNull Preferences preferences,
                      @NotNull String key);

  /**
   *  Is this value equal to the value which is stored in the preferences under the given key?
   *  @param preferences preferences where a enum value is stored
   *  @param key key under which the enum value is stored
   *  @return {@code true} if the enum value is equal to the one stored in the preferences,
   *          {@code false} otherwise
   */
  public boolean isEqualTo(@NotNull Preferences preferences,
                           @NotNull String key);

  /**
   *  Get the real value which is used in this enum.
   *  @return the real value
   */
  public E getRealValue();
}
