//=============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2014-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings;

import de.caff.annotation.NotNull;

import java.beans.PropertyChangeListener;

/**
 * Interface for items which might change.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public interface ChangeableItem
{
  /**
   *  Add a listener which will be called for all value changes.
   *
   *  @param listener value change listener to add
   */
  void addValueChangeListener(@NotNull PropertyChangeListener listener);

  /**
   *  Remove a listener which will be called for all value changes.
   *
   *  @param listener value change listener to remove
   */
  void removeValueChangeListener(@NotNull PropertyChangeListener listener);

  /**
   *  Add a listener which will be called for all value changes.
   *  The listener is registered with a weak reference, allowing it to be freed if
   *  this is the only reference.
   *  <p>
   *  If you want to remove the listener please use the corresponding
   *  {@link #removeValueChangeListenerWeakly(java.beans.PropertyChangeListener)}
   *  method.
   *
   *  @param listener value change listener to add
   *  @see #addValueChangeListener(java.beans.PropertyChangeListener)
   */
  void addValueChangeListenerWeakly(@NotNull PropertyChangeListener listener);

  /**
   *  Remove a weakly referenced listener which will be called for all value changes.
   *
   *  @param listener value change listener to remove
   *  @see #removeValueChangeListener(java.beans.PropertyChangeListener)
   */
  void removeValueChangeListenerWeakly(@NotNull PropertyChangeListener listener);
}
