//=============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2016-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.util.debug.Debug;
import de.caff.util.measure.IllegalPhysicalLengthFormatException;
import de.caff.util.measure.PhysicalLength;

import java.util.Locale;
import java.util.prefs.Preferences;

/**
 * Basic implementation of a {@link de.caff.util.measure.PhysicalLength} preference property.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since Mai.25, 2016
 */
public class BasicPhysicalLengthPreferenceProperty
        extends AbstractBasicChangeableItem
        implements PhysicalLengthPreferenceProperty
{

  private static final long serialVersionUID = 8587773466603751354L;

  /** The physical length  */
  private PhysicalLength physicalLength;

  /**
   * Constructor.
   * Construct a property with the given name.
   *
   * @param basicName name or name prefix
   */
  public BasicPhysicalLengthPreferenceProperty(@NotNull String basicName)
  {
    super(basicName);
  }

  /**
   * Constructor.
   * Construct a property with the given name and value.
   *
   * @param basicName name or name prefix
   * @param length    start length, use {@code null} for undefined
   */
  public BasicPhysicalLengthPreferenceProperty(@NotNull String basicName,
                                               @Nullable PhysicalLength length)
  {
    super(basicName);
    physicalLength = length;
  }

  /**
   * Read the property value from the preferences.
   *
   * @param preferences preferences from where to read the property value
   */
  @Override
  public void readFrom(@NotNull Preferences preferences)
  {
    final String length = preferences.get(getBasicName(), null);
    if (length == null) {
      physicalLength = null;
    }
    else {
      try {
        physicalLength = PhysicalLength.fromString(length,
                                                   Locale.US);
      } catch (IllegalPhysicalLengthFormatException e) {
        Debug.error(e);
        physicalLength = null;
      }
    }
  }

  /**
   * Store the current property value in the preferences.
   *
   * @param preferences preferences where to store the property value
   */
  @Override
  public void storeTo(@NotNull Preferences preferences)
  {
    if (physicalLength == null) {
      preferences.remove(getBasicName());
    }
    else {
      preferences.put(getBasicName(),
                      physicalLength.toString(Locale.US));
    }
  }

  /**
   * Get the value of the property.
   *
   * @return property value
   */
  @Override
  public PhysicalLength getValue()
  {
    return physicalLength;
  }

  /**
   * Set the value.
   *
   * @param value new value
   */
  @Override
  public void setValue(PhysicalLength value)
  {
    if (value == null) {
      if (physicalLength != null) {
        final PhysicalLength oldLength = physicalLength;
        physicalLength = null;
        fireValueChange(getBasicName(), oldLength, null);
      }
    }
    else {
      if (!value.equals(physicalLength)) {
        final PhysicalLength oldLength = physicalLength;
        physicalLength = value;
        fireValueChange(getBasicName(), oldLength, physicalLength);
      }
    }
  }
}
