//=============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2014-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings;

import de.caff.annotation.NotNull;
import de.caff.generics.DoublePair;

import java.awt.geom.Rectangle2D;

/**
 * Two-dimensional alignment as a combination of
 * vertical and horizontal alignment.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public enum Alignment2D
{
  SouthWest(VerticalAlignment.South, HorizontalAlignment.West),
  South(VerticalAlignment.South, HorizontalAlignment.Center),
  SouthEast(VerticalAlignment.South, HorizontalAlignment.East),
  West(VerticalAlignment.Center, HorizontalAlignment.West),
  Center(VerticalAlignment.Center, HorizontalAlignment.Center),
  East(VerticalAlignment.Center, HorizontalAlignment.East),
  NorthWest(VerticalAlignment.North, HorizontalAlignment.West),
  North(VerticalAlignment.North, HorizontalAlignment.Center),
  NorthEast(VerticalAlignment.North, HorizontalAlignment.East);

  @NotNull
  private final VerticalAlignment verticalAlignment;
  @NotNull
  private final HorizontalAlignment horizontalAlignment;

  /**
   * Constructor.
   * @param verticalAlignment   vertical alignment
   * @param horizontalAlignment horizontal alignment
   */
  Alignment2D(@NotNull VerticalAlignment verticalAlignment,
              @NotNull HorizontalAlignment horizontalAlignment)
  {
    this.verticalAlignment = verticalAlignment;
    this.horizontalAlignment = horizontalAlignment;
  }

  /**
   * Get the underlying vertical alignment.
   * @return vertical alignment
   */
  @NotNull
  public VerticalAlignment getVerticalAlignment()
  {
    return verticalAlignment;
  }

  /**
   * Get the underlying horizontal alignment.
   * @return horizontal alignment
   */
  @NotNull
  public HorizontalAlignment getHorizontalAlignment()
  {
    return horizontalAlignment;
  }

  /**
   * Get the offset for a text with the given bounds.
   * @param textBounds text bounds
   * @param yPointsDown {@code true}: Y axis points down, rectangle's minimal horizontal edge is North (typical screen coordinates),<br>
   *                    {@code false}: Y axis points up, rectangle's minimal horizontal edge is South (often used elsewhere)
   * @return a pair of translations offsets in X ({@linkplain DoublePair#first}) and Y ({@linkplain DoublePair#second}) direction
   *         to make the origin the correct text attachment point
   */
  @NotNull
  public DoublePair getTextOffset(@NotNull Rectangle2D textBounds,
                                  boolean yPointsDown)
  {
    final double w = textBounds.getWidth();
    final double h = textBounds.getHeight();
    final double dx = -textBounds.getMinX() - w * horizontalAlignment.getFactor();
    final double dy = yPointsDown
            ? -textBounds.getMaxY() + h * verticalAlignment.getFactor()
            : -textBounds.getMinY() - h * verticalAlignment.getFactor();
    return new DoublePair(dx, dy);
  }
}
