//=============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2005-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.i18n.I18n;

import java.util.Locale;
import java.util.MissingResourceException;

/**
 *  Basic implementation of a localizable preference property.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public abstract class AbstractBasicLocalizableChangeableItem
        extends AbstractBasicChangeableItem
{
  private static final long serialVersionUID = -6801428945301497210L;
  /** The basic i18n tag used for the label. */
  @NotNull
  private final String baseTag;

  /**
   *  Constructor.
   *  @param basicName  basic name (prefix)
   *  @param baseTag    basic i18n tag
   */
  protected AbstractBasicLocalizableChangeableItem(@NotNull String basicName,
                                                   @NotNull String baseTag)
  {
    super(basicName);
    this.baseTag = baseTag;
  }

  /**
   * Get a name for the editor component.
   *
   * @param l locale used for i18n
   * @return common title or {@code null}
   */
  @Nullable
  public String getName(@Nullable Locale l)
  {
    return getLabelString(l);
  }

  /**
   *  Return a localized string to be used as a label.
   *  @param l  locale
   *  @return label string
   */
  public String getLabelString(@Nullable Locale l)
  {
    return getString(l, I18n.SUFFIX_TEXT);
  }

  /**
   *  Return a localized string to be used as a menu entry.
   *  @param l  locale
   *  @return menu entry string
   */
  public String getMenuString(@Nullable Locale l)
  {
    String str = getString(l, I18n.SUFFIX_TEXT);
    return str == null  ?  getLabelString(l)  :  str;
  }

  /**
   *  Get the basic i18n label tag.
   *  @return label tag
   */
  @NotNull
  public String getBaseTag()
  {
    return baseTag;
  }

  /**
   *  Get a localized text for a tag which is the result of concatenating the basic label tag with
   *  the tag extension.
   *  @param locale         locale
   *  @param tagExtension   tag extension
   *  @return localized string or {@code null} if the resulting tag is undefined
   */
  @Nullable
  protected String getString(@Nullable Locale locale, @NotNull String tagExtension)
  {
    try {
      return I18n.getString(baseTag+tagExtension, locale);
    } catch (MissingResourceException e) {
      return null;
    }
  }
}
