//=============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2005-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.i18n.I18n;
import de.caff.util.Utility;

import javax.swing.*;
import java.util.Locale;

/**
 *  Base class used for enum values.
 * 
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
abstract class AbstractBasicEnumValue<E>
        extends Number
        implements EnumValue<E>
{
  private static final long serialVersionUID = -8099011124534435314L;
  /** The basic I18n tag. */
  private final String baseTag;

  /**
   *  Create an abstract basic enum value.
   *  @param baseTag basic i18n tag
   */
  protected AbstractBasicEnumValue(String baseTag)
  {
    this.baseTag = baseTag;
  }

  /**
   * Get a human readable name for this enum value.
   *
   * @param l locale to use
   * @return human readable short name
   */
  @NotNull
  @Override
  public String getShortName(@Nullable Locale l)
  {
    return I18n.getString(baseTag+I18n.SUFFIX_TEXT, l);
  }

  /**
   * Get a longer description which describes this value.
   *
   * @param l locale to use
   * @return human readable description or {@code null} if no description is available
   */
  @Override
  public String getLongName(@Nullable Locale l)
  {
    String name = SettingsTools.getOptionalI18nString(baseTag+I18n.SUFFIX_DESCRIPTION, l);
    return name == null  ?  getShortName(l)  :  name;
  }

  /**
   * Get a possible icon to display for this value.
   *
   * @param l locale to use
   * @return icon or {@code null} if no icon is available
   */
  @Override
  public Icon getIcon(@Nullable Locale l)
  {
    String path = SettingsTools.getOptionalI18nString(baseTag + I18n.SUFFIX_ICON, l);
    return path == null
            ? null
            : new ImageIcon(Utility.loadImage(path));
  }

  /**
   * Get the basic enum value.
   * This is needed when wrappers are used.
   *
   * @return basic value
   */
  public EnumValue<E> getBasicValue()
  {
    return this;
  }
}
