//=============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2002-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================

package de.caff.util.print;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.util.measure.LengthUnit;
import de.caff.util.measure.PhysicalLength;

import javax.print.attribute.standard.MediaPrintableArea;
import javax.print.attribute.standard.MediaSize;
import java.awt.print.PageFormat;
import java.awt.print.Paper;

/**
 *  This is the basic representation of a paper format 
 *  with size and margin information.
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class BasicPaperFormat
  implements PaperFormat,
             java.io.Serializable
{
  private static final long serialVersionUID = -6120534107108996084L;

  /**
   *  Helper class for enum describing the orientation of a paper.
   */
  public enum Orientation
  {
    Portrait(0, "portrait"),
    Landscape(90, "landscape"),
    UpsideDown(180, "upside-down"),
    Seascape(270, "seascape");

    /** Rotation angle in degrees. */
    private final int degrees;
    /** Name of this rotation if used on unrotated paper. */
    @NotNull
    private final String name;

    /**
     *  Constructor.
     *  @param degrees rotation degrees (CCW)
     *  @param name    name if used on unrotated paper in portrait format
     */
    Orientation(int degrees, @NotNull String name)
    {
      this.degrees = degrees;
      this.name = name;
    }

    /**
     *  Get the rotation in degrees, measured CCW.
     *  @return rotation
     */
    public int getDegrees()
    {
      return degrees;
    }

    /**
     * Get the name of this orientation.
     * @return orientation name
     */
    @Override
    @NotNull
    public String toString()
    {
      return name;
    }
  }

  /** The portrait orientation. */
  public static final Orientation ORIENTATION_PORTRAIT    = Orientation.Portrait;
  /** The landscape orientation. */
  public static final Orientation ORIENTATION_LANDSCAPE   = Orientation.Landscape;
  /** The upside-down orientation. */
  public static final Orientation ORIENTATION_UPSIDE_DOWN = Orientation.UpsideDown;
  /** The seascape orientation. */
  public static final Orientation ORIENTATION_SEASCAPE    = Orientation.Seascape;

  /** The AWT point unit as used in the java.awt.print.Paper class. */
  public static final LengthUnit AWT_POINTS  = new LengthUnit(2.54e-2/72,
                                                              "AWT point",
                                                              "AWT pt");
  /** The default margin size (1 cm). */
  public static final PhysicalLength DEFAULT_MARGIN =
    new PhysicalLength(10, LengthUnit.MILLIMETER);

  /** A5 paper. */
  public static final BasicPaperFormat A5_PAPER  =
    new BasicPaperFormat("A5", "iso-a5", 148, 210);
  /** A4 paper. */
  public static final BasicPaperFormat A4_PAPER  =
    new BasicPaperFormat("A4", "iso-a4", 210, 297);
  /** A3 paper. */
  public static final BasicPaperFormat A3_PAPER  =
    new BasicPaperFormat("A3", "iso-a3", 297, 420);
  /** A2 paper. */
  public static final BasicPaperFormat A2_PAPER  =
    new BasicPaperFormat("A2", "iso-a2", 420, 595);
  /** A1 paper. */
  public static final BasicPaperFormat A1_PAPER  =
    new BasicPaperFormat("A1", "iso-a1", 595, 841);
  /** A0 paper. */
  public static final BasicPaperFormat A0_PAPER  =
    new BasicPaperFormat("A0", "iso-a0", 841, 1189);
  /** B5 paper. */
  public static final BasicPaperFormat B5_PAPER  =
    new BasicPaperFormat("B5", "iso-b5", 176, 250);
  /** B4 paper. */
  public static final BasicPaperFormat B4_PAPER  =
    new BasicPaperFormat("B4", "iso-b4", 250, 353);
  /** B3 paper. */
  public static final BasicPaperFormat B3_PAPER  =
    new BasicPaperFormat("B3", "iso-b3", 353, 500);
  /** B2 paper. */
  public static final BasicPaperFormat B2_PAPER  =
    new BasicPaperFormat("B2", "iso-b2", 500, 707);
  /** B1 paper. */
  public static final BasicPaperFormat B1_PAPER  =
    new BasicPaperFormat("B1", "iso-b1", 707, 1000);
  /** B0 paper. */
  public static final BasicPaperFormat B0_PAPER  =
    new BasicPaperFormat("B0", "iso-b0", 1000, 1414);
  /** Letter format paper. */
  public static final BasicPaperFormat LETTER_PAPER  =
    new BasicPaperFormat("Letter", "na-letter", 216, 279);
  /** Legal format paper. */
  public static final BasicPaperFormat LEGAL_PAPER  =
    new BasicPaperFormat("Legal", "na-legal", 216, 356);
  /** Executive format paper. */
  public static final BasicPaperFormat EXECUTIVE_PAPER  =
    new BasicPaperFormat("Executive", "executive", 190, 254);
  /** Tabloid format paper. */
  public static final BasicPaperFormat TABLOID_PAPER  =
    new BasicPaperFormat("Tabloid", null, 279, 432);

  /** Supported paper formats. */
  public static final BasicPaperFormat[] PAPER_FORMATS = {
    A5_PAPER,
    A4_PAPER,
    A3_PAPER,
    A2_PAPER,
    A1_PAPER,
    A0_PAPER,
    B5_PAPER,
    B4_PAPER,
    B3_PAPER,
    B2_PAPER,
    B1_PAPER,
    B0_PAPER,
    LETTER_PAPER,
    LEGAL_PAPER,
    EXECUTIVE_PAPER,
    TABLOID_PAPER
  };

  /** Paper width. */
  @NotNull
  private final PhysicalLength width;
  /** Paper height. */
  @NotNull
  private final PhysicalLength height;
  /** Left margin. */
  @NotNull
  private final PhysicalLength leftMargin;
  /** Right margin. */
  @NotNull
  private final PhysicalLength rightMargin;
  /** Top margin. */
  @NotNull
  private final PhysicalLength topMargin;
  /** Bottom margin. */
  @NotNull
  private final PhysicalLength bottomMargin;
  /** The name of the paper (if any). */
  @Nullable
  private String         name;
  /** The RFC 2911 ISO name. */
  @Nullable
  private String   isoName;

  /**
   *  Internal constructor assuming mm units and default margin 
   *  used for construction of basic papers.
   *  @param name   paper format name
   *  @param isoName ISO name as in RFC 2911
   *  @param width  width in mm
   *  @param height height in mm
   */
  protected BasicPaperFormat(@Nullable String name,
                             @Nullable String isoName,
                             double width,
                             double height)
  {
    this(new PhysicalLength(width, LengthUnit.MILLIMETER),
         new PhysicalLength(height, LengthUnit.MILLIMETER),
         DEFAULT_MARGIN);
    this.name = name;
    this.isoName = isoName;
  }

  /**
   * Create a paper from another with a possible rotation.
   * This always assumes that the basic paper is in portrait format!
   * @param name       name for the newly created paper
   * @param basicPaper paper from which to construct the new paper
   * @param rotation   orientation to use, assuming the {@code basicPaper} is in portrait format
   */
  public BasicPaperFormat(@Nullable String name, @NotNull PaperFormat basicPaper, @NotNull Orientation rotation)
  {
     this(name, null, basicPaper, rotation);
  }
  /**
   * Create a paper from another with a possible rotation.
   * This always assumes that the basic paper is in portrait format!
   * @param name       name for the newly created paper
   * @param isoName ISO name as in RFC 2911
   * @param basicPaper paper from which to construct the new paper
   * @param rotation   orientation to use, assuming the {@code basicPaper} is in portrait format
   */
  public BasicPaperFormat(@Nullable String name,
                          @Nullable String isoName,
                          @NotNull PaperFormat basicPaper,
                          @NotNull Orientation rotation)
  {
    switch (rotation) {
    case Portrait:
      width = basicPaper.getWidth();
      height = basicPaper.getHeight();
      leftMargin = basicPaper.getLeftMargin();
      rightMargin = basicPaper.getRightMargin();
      topMargin   = basicPaper.getTopMargin();
      bottomMargin = basicPaper.getBottomMargin();
      break;

    case Landscape:
      width = basicPaper.getHeight();
      height = basicPaper.getWidth();
      leftMargin = basicPaper.getTopMargin();
      rightMargin = basicPaper.getBottomMargin();
      topMargin   = basicPaper.getRightMargin();
      bottomMargin = basicPaper.getLeftMargin();
      break;

    case Seascape:
      width = basicPaper.getHeight();
      height = basicPaper.getWidth();
      leftMargin = basicPaper.getBottomMargin();
      rightMargin = basicPaper.getTopMargin();
      topMargin   = basicPaper.getLeftMargin();
      bottomMargin = basicPaper.getRightMargin();
      break;

    case UpsideDown:
      width = basicPaper.getWidth();
      height = basicPaper.getHeight();
      leftMargin = basicPaper.getRightMargin();
      rightMargin = basicPaper.getLeftMargin();
      topMargin   = basicPaper.getBottomMargin();
      bottomMargin = basicPaper.getTopMargin();
      break;

    default:
      throw new IllegalArgumentException("Unknown orientation: "+rotation);
    }
    this.name = name;
    this.isoName = isoName;
  }

  /**
   *  Create a paper format with a uniform margin width.
   *  @param width  paper width
   *  @param height paper height
   *  @param margin uniform margin size
   */
  public BasicPaperFormat(@NotNull PhysicalLength width,
                          @NotNull PhysicalLength height,
                          @NotNull PhysicalLength margin)
  {
    this(width, height, margin, margin, margin, margin);
  }

  /**
   *  Create a paper format with a the given margins.
   *  @param width       paper width
   *  @param height      paper height
   *  @param leftMargin  left margin width
   *  @param rightMargin right margin width
   *  @param topMargin top margin width
   *  @param bottomMargin bottom margin width
   */
  public BasicPaperFormat(@NotNull PhysicalLength width,
                          @NotNull PhysicalLength height,
                          @NotNull PhysicalLength leftMargin,
                          @NotNull PhysicalLength rightMargin,
                          @NotNull PhysicalLength topMargin,
                          @NotNull PhysicalLength bottomMargin)
  {
    this.width       = width;
    this.height      = height;
    this.leftMargin  = leftMargin;
    this.rightMargin = rightMargin;
    this.topMargin = topMargin;
    this.bottomMargin = bottomMargin;
    this.isoName = null;
  }

  /**
   *  Copy constructor. This copies especially the name, but allows for other
   *  margins.
   *  @param pf  basic paper format to copy
   *  @param leftMargin  left margin width
   *  @param rightMargin right margin width
   *  @param topMargin top margin width
   *  @param bottomMargin bottom margin width
   */
  public BasicPaperFormat(@NotNull PaperFormat    pf,
                          @NotNull PhysicalLength leftMargin,
                          @NotNull PhysicalLength rightMargin,
                          @NotNull PhysicalLength topMargin,
                          @NotNull PhysicalLength bottomMargin)
  {
    this(pf.getWidth(),
         pf.getHeight(),
         leftMargin,
         rightMargin,
         topMargin,
         bottomMargin);
    if (pf instanceof BasicPaperFormat) {
      name = ((BasicPaperFormat)pf).name;
      isoName = ((BasicPaperFormat)pf).isoName;
    }
  }

  /**
   *  Get the paper's width.
   *  @return width
   */
  @Override
  @NotNull
  public PhysicalLength getWidth()
  {
    return width;
  }

  /**
   *  Get the paper's height.
   *  @return height
   */
  @Override
  @NotNull
  public PhysicalLength getHeight()
  {
    return height;
  }

  /**
   *  Get the left margin size.
   *  @return left margin size
   */
  @Override
  @NotNull
  public PhysicalLength getLeftMargin()
  {
    return leftMargin;
  }

  /**
   *  Get the right margin size.
   *  @return right margin size
   */
  @Override
  @NotNull
  public PhysicalLength getRightMargin()
  {
    return rightMargin;
  }

  /**
   *  Get the top margin size.
   *  @return top margin size
   */
  @Override
  @NotNull
  public PhysicalLength getTopMargin()
  {
    return topMargin;
  }

  /**
   *  Get the bottom margin size.
   *  @return bottom margin size
   */
  @Override
  @NotNull
  public PhysicalLength getBottomMargin()
  {
    return bottomMargin;
  }

  /**
   *  Get the name of the paper format.
   *  The name is only set for standard size papers.
   *  @return  paper format name or {@code null}
   */
  @Override
  @Nullable
  public String getName()
  {
    return name;
  }

  /**
   *  Get the width of the printable area.
   *  @param format paper format
   *  @return inner width
   */
  @NotNull
  public static PhysicalLength getInnerWidth(PaperFormat format)
  {
    return format.getWidth().minus(format.getLeftMargin()).minus(format.getRightMargin());
  }

  /**
   *  Get the height of the printable area.
   *  @param format paper format
   *  @return inner height
   */
  @NotNull
  public static PhysicalLength getInnerHeight(PaperFormat format)
  {
    return format.getHeight().minus(format.getTopMargin()).minus(format.getBottomMargin());
  }

  /**
   * Get this paper format in landscape orientation.
   * This assumes that this paper format is in portrait orientation, and returns a rotated version of this paper.
   * @return 90\u00b0 counterclockwise rotated version of this paper
   */
  @NotNull
  public BasicPaperFormat toLandScape()
  {
    return new BasicPaperFormat(name+" ("+ORIENTATION_LANDSCAPE+")",
                                this, ORIENTATION_LANDSCAPE);
  }

  /**
   * Get this paper format in seascape orientation.
   * This assumes that this paper format is in portrait orientation, and returns a rotated version of this paper.
   * @return 90\u00b0 clockwise rotated version of this paper
   */
  @NotNull
  public BasicPaperFormat toSeaScape()
  {
    return new BasicPaperFormat(name+" ("+ORIENTATION_SEASCAPE+")",
                                this, ORIENTATION_SEASCAPE);
  }

  /**
   * Get this paper format in upside down orientation.
   * This assumes that this paper format is in portrait orientation, and returns a rotated version of this paper.
   * @return 180\u00b0 rotated version of this paper
   */
  @NotNull
  public BasicPaperFormat toUpsideDown()
  {
    return new BasicPaperFormat(name+" ("+ORIENTATION_UPSIDE_DOWN+")",
                                this, ORIENTATION_UPSIDE_DOWN);
  }

  /**
   *  Is this paper format equal to another format.
   *  @param o  object to compare to
   */
  @Override
  public boolean equals(Object o)
  {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }

    BasicPaperFormat that = (BasicPaperFormat)o;

    if (!bottomMargin.equals(that.bottomMargin)) {
      return false;
    }
    if (!height.equals(that.height)) {
      return false;
    }
    if (!leftMargin.equals(that.leftMargin)) {
      return false;
    }
    if (name != null ? !name.equals(that.name) : that.name != null) {
      return false;
    }
    if (!rightMargin.equals(that.rightMargin)) {
      return false;
    }
    if (!topMargin.equals(that.topMargin)) {
      return false;
    }
    //noinspection RedundantIfStatement
    if (!width.equals(that.width)) {
      return false;
    }

    return true;
  }

  /**
   * Get a hashcode.
   * @return hash code
   */
  @Override
  public int hashCode()
  {
    int result;
    result = width.hashCode();
    result = 31 * result + height.hashCode();
    result = 31 * result + leftMargin.hashCode();
    result = 31 * result + rightMargin.hashCode();
    result = 31 * result + topMargin.hashCode();
    result = 31 * result + bottomMargin.hashCode();
    result = 31 * result + (name != null ? name.hashCode() : 0);
    return result;
  }

  /**
   *  Does the other paper have the same size and margins?
   *  @param other other paper
   *  @return {@code true} if the other paper and this paper have equal size and margins
   */
  public boolean equalSized(@NotNull PaperFormat other)
  {
    return width.equals(other.getWidth())    &&
           height.equals(other.getHeight())  &&
           leftMargin.equals(other.getLeftMargin())    &&
           rightMargin.equals(other.getRightMargin())  &&
           topMargin.equals(other.getTopMargin())      &&
           bottomMargin.equals(other.getBottomMargin());
  }

  /**
   * Get the media size.
   * @return media size
   */
  @NotNull
  public MediaSize getMediaSize()
  {
    return new MediaSize((float)width.getLength(LengthUnit.MILLIMETER),
                         (float)height.getLength(LengthUnit.MILLIMETER),
                         MediaSize.MM);
  }

  /**
   * Get the printable area of the paper.
   * @return printable area
   */
  @NotNull
  public MediaPrintableArea getMediaPrintableArea()
  {
    return new MediaPrintableArea((float)leftMargin.getLength(LengthUnit.MILLIMETER),
                                  (float)topMargin.getLength(LengthUnit.MILLIMETER),
                                  (float)width.minus(leftMargin).minus(rightMargin).getLength(LengthUnit.MILLIMETER),
                                  (float)height.minus(topMargin).minus(bottomMargin).getLength(LengthUnit.MILLIMETER),
                                  MediaPrintableArea.MM);
  }

  /**
   *  Get the predefined paper format for a given name.
   *  @param paperName  paper format name
   *  @return matching format or {@code null}
   */
  @Nullable
  public static BasicPaperFormat getPaperFormat(@Nullable String paperName)
  {
    if (paperName != null) {
      String lowerName = paperName.toLowerCase();
      if (lowerName.startsWith("din")) {
        // strip DIN
        paperName = paperName.substring(3).trim();
      }
      else if (lowerName.startsWith("iso-")) {
        paperName = paperName.substring(4);
      }
      else if (lowerName.startsWith("na-")) {
        paperName = paperName.substring(3);
      }
      for (BasicPaperFormat pf : PAPER_FORMATS) {
        if (paperName.equalsIgnoreCase(pf.getName())) {
          return pf;
        }
      }
    }
    return null;
  }

  /**
   *  Create a Java AWT paper from an internal paper format.
   *  @param paperFormat paper format
   *  @return corresponding Java AWT paper
   */
  @NotNull
  public static Paper toPaper(@NotNull PaperFormat paperFormat)
  {
    Paper paper = new Paper();
    paper.setSize(paperFormat.getHeight().getLength(AWT_POINTS),
                  paperFormat.getWidth().getLength(AWT_POINTS));
    paper.setImageableArea(paperFormat.getLeftMargin().getLength(AWT_POINTS),
                           paperFormat.getTopMargin().getLength(AWT_POINTS),
                           getInnerWidth(paperFormat).getLength(AWT_POINTS),
                           getInnerHeight(paperFormat).getLength(AWT_POINTS));
    return paper;
  }

  /**
   *  Create an internal paper format from a Java AWT paper.
   *  @param paper  Java AWT paper
   *  @return corresponding paper format
   */
  @NotNull
  public static BasicPaperFormat fromPaper(@NotNull Paper paper)
  {
    return new BasicPaperFormat(new PhysicalLength(paper.getWidth(), AWT_POINTS),
                                new PhysicalLength(paper.getHeight(), AWT_POINTS),
                                new PhysicalLength(paper.getImageableX(), AWT_POINTS),
                                new PhysicalLength(paper.getWidth()-
                                                   paper.getImageableX()-
                                                   paper.getImageableWidth(), AWT_POINTS),
                                new PhysicalLength(paper.getImageableY(), AWT_POINTS),
                                new PhysicalLength(paper.getHeight()-
                                                   paper.getImageableY()-
                                                   paper.getImageableHeight(), AWT_POINTS));
  }

  /**
   * Creat an internal paper format from a Java AWT page format.
   * @param pageFormat page format
   * @return corresponding paper format
   */
  @NotNull
  public static BasicPaperFormat fromPageFormat(@NotNull PageFormat pageFormat)
  {
    return new BasicPaperFormat(new PhysicalLength(pageFormat.getWidth(), AWT_POINTS),
                                new PhysicalLength(pageFormat.getHeight(), AWT_POINTS),
                                new PhysicalLength(pageFormat.getImageableX(), AWT_POINTS),
                                new PhysicalLength(pageFormat.getWidth()-
                                                   pageFormat.getImageableX()-
                                                   pageFormat.getImageableWidth(), AWT_POINTS),
                                new PhysicalLength(pageFormat.getImageableY(), AWT_POINTS),
                                new PhysicalLength(pageFormat.getHeight()-
                                                   pageFormat.getImageableY()-
                                                   pageFormat.getImageableHeight(), AWT_POINTS));
  }

  /**
   * Returns a string representation of the object. 
   *
   * @return a string representation of the object.
   */
  @Override
  public String toString()
  {
    return getName();
  }
}
