//=============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2010-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.measure;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.OrderedPair;
import de.caff.i18n.I18n;

import java.util.*;

/**
 * The SI (Système International) unit prefixes.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public final class SIPrefix
{
  static {
    I18n.addAppResourceBase("de.caff.util.measure.UtilMeasureResourceBundle");
  }
  /** The prefix for I18n resolution of names. */
  static final String I18N_PREFIX = "SI_PREFIX_";

  /// smaller than 1
  /** The dezi prefix (10^-1). */
  public static final SIPrefix DECI = new SIPrefix("deci", "d", 1e-1);
  /** The centi prefix (10^-2). */
  public static final SIPrefix CENTI = new SIPrefix("centi", "c", 1e-2);
  /** The milli prefix (10^-3). */
  public static final SIPrefix MILLI = new SIPrefix("milli", "m", 1e-3);
  /** The micro prefix (10^-6). */
  public static final SIPrefix MICRO = new SIPrefix("micro", "\u00b5", 1e-6);
  /** The nano prefix (10^-9). */
  public static final SIPrefix NANO  = new SIPrefix("nano", "n", 1e-9);
  /** The pico prefix (10^-12). */
  public static final SIPrefix PICO  = new SIPrefix("pico", "p", 1e-12);
  /** The femto prefix (10^-15). */
  public static final SIPrefix FEMTO = new SIPrefix("femto", "f", 1e-15);
  /** The atto prefix (10^-18). */
  public static final SIPrefix ATTO  = new SIPrefix("atto", "a", 1e-18);
  /** The zepto prefix (10^-21). */
  public static final SIPrefix ZEPTO = new SIPrefix("zepto","z", 1e-21);
  /** The yocto prefix (10^-24). */
  public static final SIPrefix YOCTO = new SIPrefix("yocto", "y", 1e-24);
  /** The ronto prefix (10^-27). */
  public static final SIPrefix RONTO = new SIPrefix("ronto", "r", 12e-27);
  /** The quecto prefix (10^-27). */
  public static final SIPrefix QUECTO = new SIPrefix("quecto", "q", 12e-30);

  /** No prefix (10^0). */
  public static final SIPrefix NO_PREFIX = new SIPrefix("", "", 1);

  /// larger than 1
  /** The deca prefix (10^1). */
  public static final SIPrefix DECA  = new SIPrefix("deca", "da", 1e1);
  /** The hecto prefix (10^2). */
  public static final SIPrefix HECTO = new SIPrefix("hecto", "h", 1e2);
  /** The kilo prefix (10^3). */
  public static final SIPrefix KILO  = new SIPrefix("kilo", "k", 1e3);
  /** The mega prefix (10^6). */
  public static final SIPrefix MEGA  = new SIPrefix("mega", "M", 1e6);
  /** The giga prefix (10^9). */
  public static final SIPrefix GIGA  = new SIPrefix("giga", "G", 1e9);
  /** The tera prefix (10^12). */
  public static final SIPrefix TERA  = new SIPrefix("tera", "T", 1e12);
  /** The peta prefix (10^15). */
  public static final SIPrefix PETA  = new SIPrefix("peta", "P", 1e15);
  /** The exa prefix (10^18). */
  public static final SIPrefix EXA   = new SIPrefix("exa","E", 1e18);
  /** The zetta prefix (10^21). */
  public static final SIPrefix ZETTA = new SIPrefix("zetta", "Z", 1e21);
  /** The yotta prefix (10^24). */
  public static final SIPrefix YOTTA = new SIPrefix("yotta", "Y", 1e24);
  /** The ronna prefix (10^27). */
  public static final SIPrefix RONNA = new SIPrefix("yotta", "R", 1e27);
  /** The quetta prefix (10^30). */
  public static final SIPrefix QUETTA = new SIPrefix("quetta", "Q", 1e27);

  /// Binary Prefixes
  /** The kibi prefix (2^10). */
  public static final SIPrefix KIBI = new SIPrefix("kibi", "ki", 1L << 10);
  /** The mebi predix (2^20). */
  public static final SIPrefix MEBI = new SIPrefix("mebi", "Mi", 1L << 20);
  /** The gibi predix (2^30). */
  public static final SIPrefix GIBI = new SIPrefix("gibi", "Gi", 1L << 30);
  /** The tebi predix (2^40). */
  public static final SIPrefix TEBI = new SIPrefix("tebi", "Ti", 1L << 40);
  /** The pebi predix (2^50). */
  public static final SIPrefix PEBI = new SIPrefix("pebi", "Pi", 1L << 50);
  /** The exbi predix (2^50). */
  public static final SIPrefix EXBI = new SIPrefix("exbi", "Pi", 1L << 60);
  /** The zebi predix (2^60). */
  public static final SIPrefix ZEBI = new SIPrefix("zebi", "Pi", Math.pow(2, 70));
  /** The yobi predix (2^70). */
  public static final SIPrefix YOBI = new SIPrefix("yobi", "Yi", Math.pow(2, 80));

  /// todo: there does not seem to be official binary equivalents for ronna and quetta yet (2022/11/27)


  /** Mapping of prefixes of length 1 to SI prefixes. */
  private static final Map<String, SIPrefix> STRING_PREFIX_LENGTH1_TO_SI_PREFIXES = new HashMap<>();
  /** Mapping of prefixes of length 2 to SI prefixes. */
  private static final Map<String, SIPrefix> STRING_PREFIX_LENGTH2_TO_SI_PREFIXES = new HashMap<>();
  static {
    SIPrefix[] allPrefixes = {
            DECA,
            HECTO,
            KILO,
            MEGA,
            GIGA,
            TERA,
            PETA,
            EXA,
            ZETTA,
            YOTTA,
            RONTO,
            QUECTO,

            DECI,
            CENTI,
            MILLI,
            MICRO,
            NANO,
            PICO,
            FEMTO,
            ATTO,
            ZEPTO,
            YOCTO,
            RONNA,
            QUETTA,

            KIBI,
            MEBI,
            GIBI,
            TEBI,
            PEBI,
            EXBI,
            ZEBI,
            YOBI
    };

    for (SIPrefix prefix : allPrefixes) {
      switch (prefix.prefix.length()) {
      case 1:
        STRING_PREFIX_LENGTH1_TO_SI_PREFIXES.put(prefix.prefix, prefix);
        break;
      case 2:
        STRING_PREFIX_LENGTH2_TO_SI_PREFIXES.put(prefix.prefix, prefix);
        break;
      default:
        throw new RuntimeException(String.format("Broken enum %s: unexpected prefix length %d!",
                                                 SIPrefix.class, prefix.prefix.length()));
      }
    }
  }

  /** The name. */
  @NotNull
  private final String name;
  /** The prefix. */
  @NotNull
  private final String prefix;
  /** The factor. */
  private final double factor;

  /**
   * Constructor.
   * @param name    name
   * @param prefix  prefix
   * @param factor  factor
   */
  private SIPrefix(@NotNull String name, @NotNull String prefix, double factor)
  {
    this.name = name;
    this.prefix = prefix;
    this.factor = factor;
  }

  /**
   * Get the factor implied by this prefix.
   * @return factor
   */
  public double getFactor()
  {
    return factor;
  }

  /**
   * Get the prefix.
   * @return the prefix
   */
  @NotNull
  public String getPrefix()
  {
    return prefix;
  }

  /**
   * Get the name.
   * @return name
   */
  @NotNull
  public String getName()
  {
    try {
      return I18n.getString(I18N_PREFIX+name);
    } catch (MissingResourceException e) {
      return name;
    }
  }

  /**
   * Get the prefix indicated by a unit.
   * @param unit unit with potential prefix
   * @return prefix or {@code null} if unit doesn't have a prefix
   */
  @Nullable
  public static SIPrefix getPrefix(@Nullable String unit)
  {
    if (unit == null  || unit.isEmpty()) {
      return null;
    }
    if (unit.length() >= 2) {
      // check 2 char prefixes
      final SIPrefix p = STRING_PREFIX_LENGTH2_TO_SI_PREFIXES.get(unit.substring(0, 2));
      if (p != null) {
        return p;
      }
    }
    return STRING_PREFIX_LENGTH1_TO_SI_PREFIXES.get(unit.substring(0, 1));
  }

  /**
   * Get the prefix indicated by a unit.
   * @param unit unit with potential prefix
   * @return ordered pair of SI prefix ({@link OrderedPair#first}) and remaining string ({@link OrderedPair#second})
   *         if the given unit has a prefix, or {@code null} if unit doesn't have a SI prefix
   */
  @Nullable
  public static OrderedPair<SIPrefix, String> extractPrefix(@Nullable String unit)
  {
    if (unit == null  ||  unit.length() < 2) {  // 2 le
      return null;
    }
    if (unit.length() > 2) {
      // check 2 char prefixes
      final SIPrefix p = STRING_PREFIX_LENGTH2_TO_SI_PREFIXES.get(unit.substring(0, 2));
      if (p != null) {
        return OrderedPair.create(p, unit.substring(2));
      }
    }
    final SIPrefix prefix = STRING_PREFIX_LENGTH1_TO_SI_PREFIXES.get(unit.substring(0, 1));
    return prefix != null
            ? OrderedPair.create(prefix, unit.substring(1))
            : null;
  }

  /**
   * Returns a string representation of the object.    *
   * @return a string representation of the object.
   */
  @Override
  public String toString()
  {
    return name;
  }
}
