//=============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2001-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================

package de.caff.util.measure;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;

import java.text.NumberFormat;
import java.text.ParsePosition;
import java.util.Locale;

/**
 *  A physical length.
 * <p>
 *  This is an immutable class.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public final class PhysicalLength
  implements java.io.Serializable,
             Comparable<PhysicalLength>
{
  /** The format for printing units. */
  @NotNull
  public static final String UNIT_FORMAT = "%f %s";

  private static final long serialVersionUID = -2448161374948652225L;

  /** The length. */
  private final double length;
  /** The unit. */
  private final LengthUnit unit;
  /** Precalculate for faster access. */
  private final double lengthInMeter;


  /** Null length. */
  @NotNull
  public static final PhysicalLength NULL = new PhysicalLength(0, LengthUnit.METER);

  /**
   *  Create a new physical length.
   *  @param value length value
   *  @param unit  length unit 
   */
  public PhysicalLength(double value,
                        @NotNull LengthUnit unit)
  {
    length = value;
    this.unit = unit;
    lengthInMeter = unit.unitToMeter(value);
  }

  /**
   * Get the length unit.
   * @return length unit
   */
  public LengthUnit getUnit()
  {
    return unit;
  }

  /**
   *  Get the length in meter. 
   *  @return length im meter
   */
  public double getLengthInMeter()
  {
    return lengthInMeter;
  }

  /**
   * Get the length.
   * @return length in {@link #getUnit() unit}
   */
  public double getLength()
  {
    return length;
  }

  /**
   *  Get the length in an arbitrary unit.
   *  @param  targetUnit  length unit for return value
   *  @return  the length in the given value
   */
  public double getLength(@NotNull LengthUnit targetUnit)
  {
    return targetUnit.otherToUnit(length, unit);
  }

  /**
   *  Return a string representation of this length.
   *  This uses the default locale.
   *  @return length in internal unit as string
   */
  @Override
  public String toString()
  {
    return String.format(UNIT_FORMAT, length, unit);
  }

  /**
   *  Return a string representation of this length using the given unit.
   *  @param  targetUnit  length unit
   *  @return length in unit as string
   */
  public String toString(@NotNull LengthUnit targetUnit)
  {
    return String.format(UNIT_FORMAT, getLength(targetUnit), targetUnit);
  }

  /**
   *  Return a string representation of this length.
   *  @param locale locale to use for number output
   *  @return length in internal unit as string
   */
  public String toString(@NotNull Locale locale)
  {
    return String.format(locale, UNIT_FORMAT, length, unit);
  }

  /**
   *  Return a string representation of this length using the given unit.
   *  @param  targetUnit  length unit
   *  @param locale locale to use for number output
   *  @return length in unit as string
   */
  public String toString(@NotNull LengthUnit targetUnit,
                         @NotNull Locale locale)
  {
    return String.format(locale, UNIT_FORMAT, getLength(targetUnit), targetUnit);
  }

  /**
   *  Is this length nearly equal to another one?
   *  @param  other   length unit to compare to
   *  @param  epsilon the maximal error allowed
   *  @return {@code true} if the lengths are nearly equal, otherwise {@code false}
   */
  public boolean equals(@NotNull PhysicalLength other, @NotNull PhysicalLength epsilon)
  {
    return Math.abs(other.lengthInMeter - this.lengthInMeter) <= epsilon.lengthInMeter;
  }

  /**
   *  Is this length equal to another one?
   *  @param  o  object to compare to
   *  @return {@code true} if the lengths are equals, otherwise {@code false}
   */
  @Override
  public boolean equals(Object o)
  {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }

    PhysicalLength that = (PhysicalLength)o;

    return equals(that, NULL);
  }

  /**
   * Getr a useful hashcode.
   * @return hash code
   */
  @Override
  public int hashCode()
  {
    long temp = lengthInMeter != +0.0d ? Double.doubleToLongBits(lengthInMeter) : 0L;
    return (int)(temp ^ (temp >>> 32));
  }

  /**
   *  Get a scaled physical length.
   *  @param scale scaling factor
   *  @return a physical length which is scale times this physical length
   */
  @NotNull
  public PhysicalLength times(double scale)
  {
    return new PhysicalLength(length*scale, unit);
  }

  /**
   *  Get a divided part of this physical length.
   *  Because a physical length is internally just a double all rules
   *  which apply for doubles apply here (like division by zero etc).
   *  @param divider divider
   *  @return this physical length divided by divider
   */
  @NotNull
  public PhysicalLength dividedBy(double divider)
  {
    return new PhysicalLength(length/divider, unit);
  }

  /**
   *  Get the scaling between two physical lengths.
   *  Because a physical length is internally just a double all rules
   *  which apply for doubles apply here (like division by zero etc).
   *  @param physicalLength other length
   *  @return scale: this/physicalLength
   */
  public double dividedBy(@NotNull PhysicalLength physicalLength)
  {
    return lengthInMeter / physicalLength.lengthInMeter;
  }

  /**
   *  Get the sum of this physical length and another.
   *  @param another another physical length
   *  @return this + another
   */
  @NotNull
  public PhysicalLength plus(@NotNull PhysicalLength another)
  {
    return new PhysicalLength(length+another.getLength(unit), unit);
  }

  /**
   *  Get the difference of this physical length and another.
   *  @param another another physical length
   *  @return this - another
   */
  @NotNull
  public PhysicalLength minus(@NotNull PhysicalLength another)
  {
    return new PhysicalLength(length-another.getLength(unit), unit);
  }

  /**
   * Get this length in another unit.
   * @param targetUnit other unit
   * @return physical length in the given unit
   */
  @NotNull
  public PhysicalLength in(@NotNull LengthUnit targetUnit)
  {
    return new PhysicalLength(targetUnit.otherToUnit(length, unit), targetUnit);
  }

  /**
   * Compares this object with the specified object for order.  Returns a
   * negative integer, zero, or a positive integer as this object is less
   * than, equal to, or greater than the specified object.
   *
   * <p>The implementor must ensure <tt>sgn(x.compareTo(y)) ==
   * -sgn(y.compareTo(x))</tt> for all <tt>x</tt> and <tt>y</tt>.  (This
   * implies that <tt>x.compareTo(y)</tt> must throw an exception iff
   * <tt>y.compareTo(x)</tt> throws an exception.)
   *
   * <p>The implementor must also ensure that the relation is transitive:
   * <tt>(x.compareTo(y)&gt;0 &amp;&amp; y.compareTo(z)&gt;0)</tt> implies
   * <tt>x.compareTo(z)&gt;0</tt>.
   *
   * <p>Finally, the implementor must ensure that <tt>x.compareTo(y)==0</tt>
   * implies that <tt>sgn(x.compareTo(z)) == sgn(y.compareTo(z))</tt>, for
   * all <tt>z</tt>.
   *
   * <p>It is strongly recommended, but <i>not</i> strictly required that
   * <tt>(x.compareTo(y)==0) == (x.equals(y))</tt>.  Generally speaking, any
   * class that implements the <tt>Comparable</tt> interface and violates
   * this condition should clearly indicate this fact.  The recommended
   * language is "Note: this class has a natural ordering that is
   * inconsistent with equals."
   *
   * <p>In the foregoing description, the notation
   * <tt>sgn(</tt><i>expression</i><tt>)</tt> designates the mathematical
   * <i>signum</i> function, which is defined to return one of <tt>-1</tt>,
   * <tt>0</tt>, or <tt>1</tt> according to whether the value of
   * <i>expression</i> is negative, zero or positive.
   *
   * @param o the object to be compared.
   * @return a negative integer, zero, or a positive integer as this object
   * is less than, equal to, or greater than the specified object.
   * @throws NullPointerException if the specified object is null
   * @throws ClassCastException   if the specified object's type prevents it
   *                              from being compared to this object.
   */
  @Override
  public int compareTo(@NotNull PhysicalLength o)
  {
    return Double.compare(lengthInMeter,
                          o.lengthInMeter);
  }

  /**
   *  Decodes a physical length description in text form.
   *  E.g. &quot;12 km&quot;, &quot;-7.3in&quot; and similar.
   *  @param descr text description containing a physical length
   *  @param locale locale to use for number decoding
   *  @return a physical length
   *  @throws IllegalPhysicalLengthFormatException if the string does not contain a valid physical length
   *  @throws NullPointerException if descr is {@code null}
   */
  @NotNull
  public static PhysicalLength fromString(@NotNull String descr, @Nullable Locale locale)
          throws IllegalPhysicalLengthFormatException
  {
    descr = descr.trim();
    if (descr.length() < 2) {
      throw new IllegalPhysicalLengthFormatException("String too short to contain valid physical length: "+descr);
    }
    NumberFormat format = NumberFormat.getNumberInstance(locale);
    ParsePosition parsePosition = new ParsePosition(0);
    Number number = format.parse(descr, parsePosition);
    if (parsePosition.getIndex() == 0) {
      throw new IllegalPhysicalLengthFormatException("String does not start with a valid floating point number for locale "+locale+": "+descr);
    }
    String unitName = descr.substring(parsePosition.getIndex()).trim();
    LengthUnit unit = LengthUnit.getLengthUnit(unitName);
    if (unit == null) {
      throw new IllegalPhysicalLengthFormatException("Unknown length unit: "+unitName);
    }
    return new PhysicalLength(number.doubleValue(), unit);
  }

  /**
   *  Decodes a physical length description in text form,using the default locale.
   *  E.g. &quot;12 km&quot;, &quot;-7.3in&quot; and similar.
   *  @param descr text description containing a physical length
   *  @return a physical length
   *  @throws IllegalPhysicalLengthFormatException if the string does not contain a valid physical length
   *  @throws NullPointerException if descr is {@code null}
   */
  @NotNull
  public static PhysicalLength fromString(@NotNull String descr)
          throws IllegalPhysicalLengthFormatException
  {
    return fromString(descr, Locale.getDefault());
  }

  /**
   * Factory method for lengths in inches.
   * @param inches number of inches
   * @return length with the given number of inches
   */
  @NotNull
  public static PhysicalLength in(double inches)
  {
    return new PhysicalLength(inches, LengthUnit.INCH);
  }

  /**
   * Factory method for lengths in foot.
   * @param feet number of feet
   * @return length with the given number of feet
   */
  @NotNull
  public static PhysicalLength ft(double feet)
  {
    return new PhysicalLength(feet, LengthUnit.FOOT);
  }

  /**
   * Factory method for lengths in meter.
   * @param meter number of meters
   * @return length with the given number of meters
   */
  @NotNull
  public static PhysicalLength m(double meter)
  {
    return new PhysicalLength(meter, LengthUnit.METER);
  }

  /**
   * Factory method for lengths in millimeter.
   * @param millimeter number of millimeters
   * @return length with the given number of millimeters
   */
  @NotNull
  public static PhysicalLength mm(double millimeter)
  {
    return new PhysicalLength(millimeter, LengthUnit.MILLIMETER);
  }

  public static void main(String[] args)
  {
    for (String arg : args) {
      try {
        PhysicalLength length = PhysicalLength.fromString(arg);
        System.out.println("Read: " + length);
      } catch (IllegalPhysicalLengthFormatException e) {
        e.printStackTrace();
      }
    }
  }
}
