//=============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2001-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================

package de.caff.util.measure;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;

import java.util.Objects;

import static de.caff.generics.Primitives.areEqual;
import static de.caff.generics.Primitives.boxed;

/**
 *  A physical length unit.
 *  
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public final class LengthUnit
  implements java.io.Serializable
{
  /** The METER unit. */
  public static final LengthUnit METER      = new LengthUnit(1.0,
							    "meter", 
							    "m");
  /** The KILOMETER unit. */
  public static final LengthUnit KILOMETER  = new LengthUnit(SIPrefix.KILO, METER);
  /** The DECIMETER unit. */
  public static final LengthUnit DECIMETER  = new LengthUnit(SIPrefix.DECI, METER);
  /** The CENTIMETER unit. */
  public static final LengthUnit CENTIMETER = new LengthUnit(SIPrefix.CENTI, METER);
  /** The MILLIMETER unit. */
  public static final LengthUnit MILLIMETER = new LengthUnit(SIPrefix.MILLI, METER);
  /** The MICROMETER unit. */
  public static final LengthUnit MICROMETER = new LengthUnit(SIPrefix.MICRO, METER);
  /** The NANOMETER unit. */
  public static final LengthUnit NANOMETER  = new LengthUnit(SIPrefix.NANO, METER);
  /** The PICOMETER unit. */
  public static final LengthUnit PICOMETER  = new LengthUnit(SIPrefix.PICO, METER);
  /** The Angstrom unit. */
  public static final LengthUnit ANGSTROM   = new LengthUnit(1.0e-10,
							     "\u00c5ngstr\u00f8m",
							     "\u00c5");
  /** The Astronomical Unit. */
  public static final LengthUnit ASTRONOMICAL_UNIT = new LengthUnit(149597870700.0,
                                                                    "Astronomical Unit",
                                                                    "au");
  /** The light year. */
  public static final LengthUnit LIGHTYEAR = new LengthUnit(9460730472580800.0,
                                                             "Lightyear",
                                                             "Lj");
  /** The parsec. */
  public static final LengthUnit PARSEC     = new LengthUnit(30856775777948584.0,
                                                             "Parsec",
                                                             "pc");
  /** The INCH unit. */
  public static final LengthUnit INCH       = new LengthUnit(2.54e-2,
							     "inch",
							     "in");
  /** The FOOT unit. */
  public static final LengthUnit FOOT       = new LengthUnit(12*INCH.ratio,
							     "foot",
							     "ft");
  /** The YARD unit. */
  public static final LengthUnit YARD       = new LengthUnit(36*INCH.ratio,
                                                             "yard",
                                                             "yd");
  /** The statute MILE unit. */
  public static final LengthUnit MILE       = new LengthUnit(1760*YARD.ratio,
                                                             "mile",
                                                             "stat.mil.");
  /** The POINT unit. */
  public static final LengthUnit POINT      = new LengthUnit(INCH.ratio / 72,
                                                             "point",
                                                             "pt");
  /** The printers point, as in England and USA. */
  public static final LengthUnit PICA_POINT = new LengthUnit(INCH.ratio / 72.27000072,
                                                             "Anglo-Saxon pica point",
                                                             "pp");
  // more to come...

  private static final LengthUnit[] UNITS = {
          METER,
          KILOMETER,
          DECIMETER,
          CENTIMETER,
          MILLIMETER,
          MICROMETER,
          NANOMETER,
          PICOMETER,
          ANGSTROM,
          ASTRONOMICAL_UNIT,
          LIGHTYEAR,
          PARSEC,
          INCH,
          FOOT,
          YARD,
          MILE,
          POINT,
          PICA_POINT
  };
  private static final long serialVersionUID = -8998868857886200850L;

  /** The unit/meter ratio. */
  private final double ratio;
  /** The unit's name. */
  @NotNull
  private final String name;
  /** The unit's shortcut. */
  @NotNull
  private final String shortCut;

  /**
   *  Construct a length unit with the given unit/meter ratio, name and
   *  shortcut.
   *  @param  metersPerUnit    the ratio <tt>unit/m</tt>
   *  @param  unitName         the name of the unit
   *  @param  unitShortCut     the standard shortcut for the unit
   */
  public LengthUnit(double metersPerUnit,
		    @NotNull String unitName,
		    @NotNull String unitShortCut)
  {
    ratio    = metersPerUnit;
    name     = unitName;
    shortCut = unitShortCut;
  }

  /**
   * Construct a length from a base unit with an SI prefix.
   * @param prefix   SI prefix
   * @param baseUnit base unit
   */
  private LengthUnit(@NotNull SIPrefix prefix, @NotNull LengthUnit baseUnit)
  {
    ratio = baseUnit.ratio * prefix.getFactor();
    name = prefix.getName() + baseUnit.name;
    shortCut = prefix.getPrefix() + baseUnit.shortCut;
  }

  /**
   *  Return the shortcut.
   *  @return the unit's shortcut
   */
  @NotNull
  public String getShortCut()
  {
    return shortCut;
  }

  /**
   *  Return the name.
   *  @return the unit's name
   */
  @NotNull
  public String getName() 
  {
    return name;
  }

  /**
   *  Return a string representation.
   *  @return the shortcut
   */
  @Override
  public String toString()
  {
    return getShortCut();
  }

  /**
   *  Transform the given length in meters to this unit.
   *  @param  meter length (in meter)
   *  @return length (in units)
   */
  public double meterToUnit(double meter)
  {
    return meter/ratio;
  }

  /**
   *  Transform the given length in units to meter.
   *  @param  units length (in units)
   *  @return length (in meter)
   */
  public double unitToMeter(double units)
  {
    return units*ratio;
  }

  /**
   *  Transform the given length from arbitrary units to this units.
   *  @param  length  length in given units
   *  @param  other   (other) unit of length
   *  @return length (in this unit)
   */
  public double otherToUnit(double length, @NotNull LengthUnit other)
  {
    return other.unitToMeter(length)/ratio;
  }

  /**
   *  Compare to another unit.
   *  @param  obj  other unit
   *  @return {@code true} if both units are equal.
   */
  @Override
  public boolean equals(Object obj)
  {
    if (this == obj) {
      return true;
    }
    if (obj == null || getClass() != obj.getClass()) {
      return false;
    }

    final LengthUnit that = (LengthUnit)obj;

    return areEqual(that.ratio, ratio) &&
           name.equals(that.name) &&
           shortCut.equals(that.shortCut);
  }

  /**
   * Provide a useful hash code.
   * @return hash code
   */
  @Override
  public int hashCode()
  {
    return Objects.hash(boxed(ratio), name, shortCut);
  }

  /**
   * Get a length using this unit.
   * @param length length
   * @return physical length using this unit
   */
  @NotNull
  public PhysicalLength length(double length)
  {
    return new PhysicalLength(length, this);
  }

  /**
   *  Get the length unit for a given unit shortcut.
   *  @param unitShortcut unit shortcut (eg <tt>mm</tt> or <tt>in</tt>)
   *  @return matching unit or {@code null}
   */
  @Nullable
  public static LengthUnit getLengthUnit(@Nullable String unitShortcut)
  {
    for (LengthUnit u : UNITS) {
      if (u.getShortCut().equalsIgnoreCase(unitShortcut)) {
        return u;
      }
    }
    return null;
  }
}
