// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 1999-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================

package de.caff.util.debug;

import de.caff.annotation.NotNull;

import java.io.OutputStream;
import java.io.PrintStream;

/**
 *  A debug listener which copies all messages to a stream.
 *  It's useful for all categories of messages.
 *
 *  @see Debug
 *
 *  @author Rammi
 */
public class SimpleOutputtingDebugListener 
  implements CookedMessageDebugListener,
	     AnyMessageDebugListener
{
  private final PrintStream _out;

  /**
   *  Construct a new listener for the given stream.
   *  @param  stream  output stream
   */
  public SimpleOutputtingDebugListener(OutputStream stream) {
    _out = new PrintStream(stream);
  }

  /**
   *  Write the message to the output stream.
   *  @param msg  message
   *  @param pos  postion
   */
  private void print(String msg, String pos) {
    _out.println(msg+"["+pos+"]");
  }

  /**
   *  Receive a raw trace debug message.
   *  @param msg  the message
   *  @param pos  postion
   */
  @Override
  public void receiveTraceMessage(@NotNull String msg, @NotNull String pos) {
    print(msg, pos);
  }

  /**
   *  Receive a raw standard debug message.
   *  @param msg  die Meldung
   *  @param pos  postion
   */
  @Override
  public void receiveStandardMessage(@NotNull String msg, @NotNull String pos) {
    print(msg, pos);
  }

  /**
   *  Receive a raw warning debug message.
   *  @param msg  the message
   *  @param pos  postion
   */
  @Override
  public void receiveWarningMessage(@NotNull String msg, @NotNull String pos) {
    print(msg, pos);
  }

  /**
   *  Receive a raw error message.
   *  @param msg  the message
   *  @param pos  postion
   */
  @Override
  public void receiveErrorMessage(@NotNull String msg, @NotNull String pos) {
    print(msg, pos);
  }

  /**
   *  Receive a raw logging debug message.
   *  @param msg  the message
   *  @param pos  postion
   */
  @Override
  public void receiveLogMessage(@NotNull String msg, @NotNull String pos) {
    print(msg, pos);
  }

  /**
   *  Receive a raw fatal error message and return whether the program shall be
   *  exited.
   *  @param   msg  the message
   *  @param pos  postion
   *  @return  {@code 1} -- exit the program
   */
  @Override
  public int receiveFatalMessage(@NotNull String msg, @NotNull String pos) {
    print(msg, pos);
    return 1;
  }

  /**
   *  Receive a raw message about a failed assertion and return, whether an exception
   *  shall be thrown.
   *  @param   msg  the message
   *  @param pos  postion
   *  @return  {@code true} -- create exception
   */
  @Override
  public boolean receiveFailedAssertionMessage(@NotNull String msg, @NotNull String pos) {
    print(msg, pos);
    return true;
  }

  
  /**
   *  Receive a cooked message. 
   *  @param   msgType message type (ignored)
   *  @param   msg     the message
   *  @param pos  postion
   */
  @Override
  public void receiveCookedMessage(int msgType, String msg, String pos) {
    print(msg, pos);
  }

}
