// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 1999-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================

package de.caff.util.debug;

import de.caff.annotation.NotNull;

import java.lang.reflect.*;

/**
 *  Interface for in-deep inspection of an object using reflection.
 *  Is not working well at the moment.
 *
 *  @author Rammi
 */
public class ObjectInspector
{
  /**
   * Return a detailed description of an object.
   * @param obj  object to inspect
   * @param withGetters include getters in the object's properties?
   * @return detailed description
   */
  static String getInfo(@NotNull Object obj, boolean withGetters) {
    try {
      Class<?>  c = obj.getClass();
      StringBuilder sb = new StringBuilder();
      sb.append("Object: ").append(c.getName()).append('@').append(Integer.toHexString(obj.hashCode())).append('\n');
      while (c != null) {
        sb.append(getFields(c, obj, withGetters));
        c = c.getSuperclass();
      }
      return sb.toString();
    } catch (Exception x) {
      // SecurityException?
      return x.toString();
    }
  }

  private static String getFields(Class<?> c, Object obj, boolean withGetters) {
    StringBuilder ret = new StringBuilder("{ // ");
    ret.append(c.getName()).append('\n');
    Field[] fields = c.getDeclaredFields();
    for (Field field : fields) {
      ret.append('\t').append(field.toGenericString()).append(" = ").append(getValue(field, obj)).append(";\n");
    }
    if (withGetters) {
      for (Method method : c.getDeclaredMethods()) {
        if (method.getReturnType() == null) {
          continue;
        }
        if (!Modifier.isPublic(method.getModifiers())) {
          continue;
        }
        if (method.getParameterTypes().length > 0) {
          continue;
        }
        if (!method.getName().startsWith("get")) {
          continue;
        }
        try {
          final Object result = method.invoke(obj);
          ret.append('\t').append(method.toGenericString()).append("()=").append(result).append(";\n");
        } catch (IllegalAccessException e) {
        } catch (InvocationTargetException e) {
        }
      }
    }
    ret.append("}");
    return ret.toString();
  }
  
  private static String getValue(Field f, Object obj) {
    try {
      //      ReflectPermission perm = new ReflectPermission("suppressAccessChecks");
      f.setAccessible(true);

      StringBuilder ret = new StringBuilder();
      Class<?> type = f.getType();
      if (type.isPrimitive()) {
	if (type == Integer.TYPE) {
	  // int
	  ret.append(f.getInt(obj));
	}
	else if (type == Boolean.TYPE) {
	  // boolean
	  ret.append(f.getBoolean(obj));
	}
	else if (type == Byte.TYPE) {
	  // byte
	  ret.append(f.getByte(obj));
	}
	else if (type == Short.TYPE) {
	  // short
	  ret.append(f.getShort(obj));
	}
	else if (type == Long.TYPE) {
	  // long
	  ret.append(f.getLong(obj));
	}
	else if (type == Float.TYPE) {
	  // float
	  ret.append(f.getFloat(obj));
	}
	else if (type == Double.TYPE) {
	  // double
	  ret.append(f.getDouble(obj));
	}
      }
      else {
	ret.append(f.get(obj));
      }
      
      return ret.toString();
    } catch (IllegalAccessException x) {
      return "?";
    } catch (SecurityException x) {
      return "?";
    }
  }
}
