// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 1999-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================

package de.caff.util.debug;

import de.caff.util.MemFormat;

import javax.swing.*;
import java.awt.*;

/**
 *  A panel displaying automatically updated info about memory usage of the JVM.
 *
 *  @author Rammi
 */
public class MemoryUsagePanel
        extends JPanel
{
  private static final int MEMORY_UPDATE_TIME = 2*1000;
  private static final long serialVersionUID = 4485568376466427703L;
  private final JLabel         _usedMemLabel;
  private final JLabel         _freeMemLabel;
  private final JLabel         _totalMemLabel;
  private final JLabel         _maxUsedMemLabel;
  private long           _maxUsedMem;

  public MemoryUsagePanel() {
    this(MEMORY_UPDATE_TIME);
  }

  public MemoryUsagePanel(int updateTime) {
    super(new BorderLayout());
    JPanel sub      = new JPanel(new GridLayout(0, 2, 1, 1));
    sub.add(new JLabel("used:  ", JLabel.RIGHT));
    sub.add(_usedMemLabel = new JLabel());
    
    sub.add(new JLabel("free:  ", JLabel.RIGHT));
    sub.add(_freeMemLabel = new JLabel());
    
    sub.add(new JLabel("TOTAL:  ", JLabel.RIGHT));
    sub.add(_totalMemLabel = new JLabel());
    
    sub.add(new JLabel("max. used:  ", JLabel.RIGHT));
    sub.add(_maxUsedMemLabel = new JLabel());
    
    add("North", sub);
      
    JButton col = new JButton("Collect!");
    col.addActionListener(e -> {
      System.gc();
      _maxUsedMem = 0;
      updateMemoryDisplay();
    });
    col.setToolTipText("Invokes the garbage collector and resets max.used");
    add("South", col);
    
    if (updateTime > 0) {
      new Timer(MEMORY_UPDATE_TIME, e -> updateMemoryDisplay()).start();
    }
  }

  /**
   *  Get a string from a memory size value.
   *  @param  size  size value
   *  @return string representation
   */
  private static String getString(long size) {
    return MemFormat.Base10.toString(size);
  }

  /**
   *  Update the memory display.
   */
  private void updateMemoryDisplay() {
    Runtime rt = Runtime.getRuntime();
    long freeMem  = rt.freeMemory();
    long totalMem = rt.totalMemory();
    long usedMem  = totalMem - freeMem;

    _usedMemLabel.setText(getString(usedMem));
    _freeMemLabel.setText(getString(freeMem));
    _totalMemLabel.setText(getString(totalMem));
    if (usedMem > _maxUsedMem) {
      _maxUsedMem = usedMem;
      _maxUsedMemLabel.setText(getString(_maxUsedMem));
    }
  }
}
