// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 1999-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================

package de.caff.util.debug;

import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.File;
import java.io.FileWriter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;

/**
 *  A text window frame in which debugging messages are outputted in cooked format.
 *  It is possible to show an extended interface with buttons allowing global tuning
 *  of the debugging interface.
 *  
 *  @see Debug
 *
 *  @author Rammi
 */
public class DebugMessageWindow
        extends JFrame
        implements CookedMessageDebugListener
{
  private static final long serialVersionUID = 2154406130530409995L;
  private final JTextArea      _msgArea;
  private int            _endPos;

  
  /**
   *  Constructor.
   *  @param  extended  use extended interface with tuning buttons?
   */
  public DebugMessageWindow(boolean extended) {
    super("Debug Messages");

    setDefaultCloseOperation(DO_NOTHING_ON_CLOSE);
    _msgArea = new JTextArea(25, 100);
    _msgArea.setFont(new Font("Monospaced", Font.PLAIN, 12));
    _msgArea.setEditable(false);
    getContentPane().add(new JScrollPane(_msgArea));

    JPanel panel = new JPanel(new BorderLayout());
    getContentPane().add("East", panel);
      
    if (extended) {
      JPanel memPanel = new MemoryUsagePanel();
      memPanel.setBorder(BorderFactory.createTitledBorder(BorderFactory.createEtchedBorder(),
							  "Memory"));
      panel.add("North", memPanel);

      JPanel switchPanel = new DebugLevelSwitchBoard();
      panel.add("Center", switchPanel);
      switchPanel.setBorder(BorderFactory.createTitledBorder(BorderFactory.createEtchedBorder(),
							  "Global Settings"));
      panel.add(switchPanel);
    }

    JPanel allButtons  = new JPanel(new BorderLayout());
    allButtons.setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5));
    panel.add("South", allButtons);

    JButton b = new JButton("Print System Properties");
    allButtons.add("North", b);
    b.addActionListener(e -> printProperties());
    b.setToolTipText("Print the system properties in the text window");
    
    JPanel buttonPanel = new JPanel(new GridLayout(0, 1));
    allButtons.add("South", buttonPanel);

    b = new JButton("Clear");
    buttonPanel.add(b);
    b.addActionListener(e -> clear());
    b.setToolTipText("Clears the text window");

    b = new JButton("Save...");
    buttonPanel.add(b);
    b.addActionListener(e -> save());
    b.setToolTipText("Allows to save the content of the text window");

    addWindowListener(new WindowAdapter() {
      @Override public void windowClosing(WindowEvent e) {
	Debug.removeCookedMessageDebugListener(DebugMessageWindow.this);
	dispose();
      }
    });
    Debug.addCookedMessageDebugListener(this);

    UIManager.addPropertyChangeListener(evt -> {
      if ("lookAndFeel".equals(evt.getPropertyName())) {
        SwingUtilities.updateComponentTreeUI(DebugMessageWindow.this);
      }
    });
    pack();
    setVisible(true);
  }

  /**
   *  Receive a message and append it to the text window.
   *  @param  msgType  ignored
   *  @param  message  new message
   *  @param  pos      position
   */
  @Override
  public void receiveCookedMessage(int msgType, final String message, final String pos) {
    SwingUtilities.invokeLater(() -> append(message + DebugMessageCook.cookedPosition(pos)));
  }

  /**
   *  Append a text to the message area.
   *  @param  text  text to append
   */
  private void append(String text) {
    _msgArea.append(text);
    _endPos += text.length();
    _msgArea.setCaretPosition(_endPos);
  }

  /**
   *  Clear text window.
   */
  private void clear() {
    SwingUtilities.invokeLater(() -> {
      _msgArea.setText("");
      _endPos = 0;
    });
  }

  /**
   *  Save text window content.
   */
  private void save() {
    JFileChooser chooser = new JFileChooser();

    if (chooser.showDialog(this, "Sichern") == JFileChooser.APPROVE_OPTION) {
      File file = chooser.getSelectedFile();
      try {
	FileWriter out = new FileWriter(file);
	out.write(_msgArea.getText());
	out.close();
	Debug.message("Debugmeldungen gesichert in \"%0\"", file);
      } catch (Exception x) {
	Debug.error("Sichern von \"%0\" gescheitert.\nGrund:\n%1",
		    file, x.getMessage());
      }
    }
  }

  /**
   *  Add the system properties to the debug window.
   */
  private void printProperties() {
    ArrayList<String>   array = new ArrayList<>();
    for (Enumeration<?> p = System.getProperties().propertyNames();  p.hasMoreElements();  ) {
      String prop = (String)p.nextElement();
      try {
	array.add("\t"+prop+" = \""+System.getProperty(prop)+"\"\n");
      } catch (SecurityException x) {
	array.add("\t"+prop+" not available due to security restrictions\n");
      }
    }
    Collections.sort(array);
    
    append("SYSTEM PROPERTIES:\n");
    for (Object anArray : array) {
      append(anArray.toString());
    }
  }
}

