// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 1999-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================

package de.caff.util.debug;

import de.caff.annotation.NotNull;
import de.caff.generics.Types;

import java.text.DateFormat;
import java.util.*;

/**
 *  This class &quot;cooks&quot; debug messages. I.e. the messages are prepended
 *  with date and type information and extended with a line showing where
 *  the message was created. The message itself is inserted by a tab.
 *
 *  @see Debug
 *
 *  @author Rammi
 */
class DebugMessageCook
        implements AnyMessageDebugListener,
                   DebugConstants
{
  private static final boolean THROW_ASSERTION_EXCEPTION = true;
  private static final boolean STOP_ON_FATAL_ERRORS      = true;
  private static final boolean EMPTY_THROW_ASSERTION     = false;
  private static final int     EMPTY_FATAL_RETURN        = 0;

  private static final String  HEAD_TRACE     = "TRACE";
  private static final String  HEAD_MESSAGE   = "MESSAGE";
  private static final String  HEAD_WARNING   = "WARNING";
  private static final String  HEAD_ERROR     = "ERROR";
  private static final String  HEAD_FATAL     = "FATAL ERROR";
  private static final String  HEAD_LOG       = "LOGGING";
  private static final String  HEAD_ASSERTION = "ASSERTION FAILED";

  private Collection<ListenerData> _list = Types.synchronizedCollection(new LinkedList<ListenerData>());

  /**
   *  Data collection for listener.
   */
  static class ListenerData {
    final CookedMessageDebugListener listener;
    final boolean                    stopOnFatalErrors;
    final boolean                    throwAssertionException;

    /**
     *  Construtor.
     *  Allows to set what happens to fatal errors and failed assertions.
     *  @param  listener           the listener waiting to be served by this cook
     *  @param  stopOnFatalErrors  stop program on fatal errors?
     *  @param  throwAssertionException throw exception on failed assertions?
     *  @exception NullPointerException if the listener is not set
     */
    ListenerData(@NotNull CookedMessageDebugListener listener,
		 boolean stopOnFatalErrors,
		 boolean throwAssertionException) 
    {
      this.listener                = listener;
      this.stopOnFatalErrors       = stopOnFatalErrors;
      this.throwAssertionException = throwAssertionException;
    }  
  }

  /**
   *  Add a listener. This one stops on fatal errors and throws an exception
   *  on failed assertions.
   *  @param  listener           the listener waiting to be served by this cook
   */
  public void addListener(@NotNull CookedMessageDebugListener listener) {
    addListener(listener,
                STOP_ON_FATAL_ERRORS,
                THROW_ASSERTION_EXCEPTION);
  }
  
  /**
   *  Add a listener.
   *  @param  listener           the listener waiting to be served by this cook
   *  @param  stopOnFatalErrors  stop program on fatal errors?
   *  @param  throwAssertionException throw exception on failed assertions?
   */
  public synchronized void addListener(@NotNull CookedMessageDebugListener listener,
                                       boolean stopOnFatalErrors,
                                       boolean throwAssertionException) {
    synchronized (this) {
      List<ListenerData> copy = new LinkedList<>(_list);
      copy.add(new ListenerData(listener, stopOnFatalErrors, throwAssertionException));
      _list = copy;
    }
  }

  /**
   *  Remove a listener.
   *  @param  listener   listener to be removed
   */
  public synchronized void removeListener(@NotNull CookedMessageDebugListener listener) {
    synchronized (this) {
      List<ListenerData> copy;
      copy = new LinkedList<>(_list);
      for (Iterator<ListenerData> it = copy.iterator();  it.hasNext();  ) {
        ListenerData data = it.next();
        if (data.listener == listener) {
          it.remove();
          break;
        }
      }
      _list = copy;
    }
  }
  
  /**
   *  Create formatted output. Prepend with date/category and append stack position.
   *  @param head    header
   *  @param msg     message
   *  @return formatted version of message
   */
  private static String getFormatted(String head, String msg) {
    StringBuilder compile = new StringBuilder();
    compile.append(DateFormat.getDateTimeInstance().format(new Date())).append('\t').append(head).append(":\n");
    if (msg != null) {
      int lastIndex = 0;
      int index     = msg.indexOf('\n');
      while (index != -1) {
        compile.append('\t').append(msg.substring(lastIndex, index+1));
        lastIndex = index+1;
        index     =  msg.indexOf('\n', lastIndex);
      }
      compile.append('\t').append(msg.substring(lastIndex)).append('\n');
    }
    else {
      compile.append("\r<null>\n");
    }
    return compile.toString();
  }

  private synchronized Collection<ListenerData> getListeners()
  {
    return _list;
  }

  /**
   *  Sends the cooked message to all listeners.
   *  @param msgType type of message
   *  @param cooked  cooked message
   *  @param pos     position
   */
  private void distribute(int msgType, String cooked, String pos) {
    for (ListenerData data: getListeners()) {
      data.listener.receiveCookedMessage(msgType, cooked, pos);
    }
  }

  /**
   *  Sends the cooked fatal message to all listeners.
   *  If at least one wants to exit the program this is returned.
   *  @param cooked  cooked message
   *  @param pos     postion
   *  @return {@code 1}, if at least one listener wants to exit<br>
   *          {@code 0} else
   */
  private int distributeFatal(String cooked, String pos) {
    boolean exit = false;
    for (ListenerData data: getListeners()) {
      data.listener.receiveCookedMessage(FATAL, cooked, pos);
      if (!exit  &&  data.stopOnFatalErrors) {
	exit = true;
      }
    }
    return exit ? 1 : 0;
  }

  /**
   *  Sends the assertion message to all listeners.
   *  If at least one wants to throw an exception the program this is returned.
   *  @param cooked  cooked message
   *  @param pos     postion
   *  @return {@code true}, if at least one listener wants to throw<br>
   *          {@code false} else
   */
  private boolean distributeFailedAssertion(String cooked, String pos) {
    boolean dothrow = false;
    for (ListenerData data: getListeners()) {
      data.listener.receiveCookedMessage(ASSERT, cooked, pos);
      if (!dothrow  &&  data.stopOnFatalErrors) {
	dothrow = true;
      }
    }
    return dothrow;
  }

  /**
   *  Receive a trace debug message, cook it and serve it to the listeners.
   *  @param msg  the raw message
   *  @param pos  postion
   */
  @Override
  public void receiveTraceMessage(@NotNull String msg, @NotNull String pos) {
    if (!getListeners().isEmpty()) {
      distribute(TRACE, getFormatted(HEAD_TRACE, msg), pos);
    }
  }

  /**
   *  Receive a standard debug message, cook it and serve it to the listeners.
   *  @param msg  the raw message
   *  @param pos  postion
   */
  @Override
  public void receiveStandardMessage(@NotNull String msg, @NotNull String pos) {
    if (!getListeners().isEmpty()) {
      distribute(MESSAGE, getFormatted(HEAD_MESSAGE, msg), pos);
    }
  }

  /**
   *  Receive a warning debug message, cook it and serve it to the listeners.
   *  @param msg  the raw message
   *  @param pos  postion
   */
  @Override
  public void receiveWarningMessage(@NotNull String msg, @NotNull String pos) {
    if (!getListeners().isEmpty()) {
      distribute(WARNING, getFormatted(HEAD_WARNING, msg), pos);
    }
  }

  /**
   *  Receive an error debug message, cook it and serve it to the listeners.
   *  @param msg  the raw message
   *  @param pos  postion
   */
  @Override
  public void receiveErrorMessage(@NotNull String msg, @NotNull String pos) {
    if (!getListeners().isEmpty()) {
      distribute(ERROR, getFormatted(HEAD_ERROR, msg), pos);
    }
  }

  /**
   *  Receive a logging message, cook it and serve it to the listeners.
   *  @param msg  the raw message
   *  @param pos  postion
   */
  @Override
  public void receiveLogMessage(@NotNull String msg, @NotNull String pos) {
    if (!getListeners().isEmpty()) {
      distribute(LOG, getFormatted(HEAD_LOG, msg), pos);
    }
  }

  /**
   *  Receive a fatal error  message, cook it and serve it to the listeners.
   *  If the listener list is empty, {@code 0} is returned.
   *  @param msg  the raw message
   *  @param pos  postion
   *  @return not equal to {@code 0} for exit<br>
   *          else keep program running
   */
  @Override
  public int receiveFatalMessage(@NotNull String msg, @NotNull String pos) {
    if (!getListeners().isEmpty()) {
      return distributeFatal(getFormatted(HEAD_FATAL, msg), pos);
    }
    else {
      return EMPTY_FATAL_RETURN;
    }
  }

  /**
   *  Receive a failed assertion message, cook it and serve it to the listeners.
   *  If the listener list is empty, {@code false} is returned.
   *  @param msg  the raw message
   *  @param pos  postion
   *  @return throw an assertion?
   */
  @Override
  public boolean receiveFailedAssertionMessage(@NotNull String msg, @NotNull String pos) {
    if (!getListeners().isEmpty()) {
      return distributeFailedAssertion(getFormatted(HEAD_ASSERTION, msg), pos);
    }
    else {
      return EMPTY_THROW_ASSERTION;
    }
  }

  /**
   *  Get a position in default cooked format.
   *  @param  pos position string
   *  @return cooked version to be appended to a message
   */
  public static String cookedPosition(String pos) {
    return "\t[at "+pos+"]\n";
  }
  

}

