// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 1999-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================

package de.caff.util.debug;

import javax.swing.*;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;

/**
 *  A panel with Checkboxes allowing to switch on/off global debug levels
 *
 *  @author Rammi
 */
public class DebugLevelSwitchBoard
        extends JPanel
        implements Debug.ModeChangeListener, DebugConstants
{
  private static final long serialVersionUID = 6426942752115020936L;
  private final JCheckBox      _traceCB;
  private final JCheckBox      _messageCB;
  private final JCheckBox      _warningCB;
  private final JCheckBox      _errorCB;
  private final JCheckBox      _fatalCB;
  private final JCheckBox      _logCB;
  private final JCheckBox      _assertCB;
  private final JCheckBox      _stdoutCB;
  private final JCheckBox      _stderrCB;

  public DebugLevelSwitchBoard() {
    setLayout(new BoxLayout(this, BoxLayout.Y_AXIS));

    _traceCB = new JCheckBox("Capture trace messages", Debug.getTraceMode());
    _traceCB.addItemListener(e -> Debug.setTraceMode(e.getStateChange() == ItemEvent.SELECTED));
    _traceCB.setToolTipText("Toggles the global setting for trace debug messages.");
    add(_traceCB);
      
    _messageCB = new JCheckBox("Capture standard messages", Debug.getStandardMode());
    _messageCB.addItemListener(e -> Debug.setStandardMode(e.getStateChange() == ItemEvent.SELECTED));
    _messageCB.setToolTipText("Toggles the global setting for standard debug messages.");
    add(_messageCB);

    _warningCB = new JCheckBox("Capture warnings", Debug.getWarningMode());
    _warningCB.addItemListener(e -> Debug.setWarningMode(e.getStateChange() == ItemEvent.SELECTED));
    _warningCB.setToolTipText("Toggles the global setting for warning debug messages.");
    add(_warningCB);

    _errorCB = new JCheckBox("Capture error messages", Debug.getErrorMode());
    _errorCB.addItemListener(e -> Debug.setErrorMode(e.getStateChange() == ItemEvent.SELECTED));
    _errorCB.setToolTipText("Toggles the global setting for error debug messages.");
    add(_errorCB);

    _fatalCB = new JCheckBox("Capture fatal error messages", Debug.getFatalMode());
    _fatalCB.addItemListener(e -> Debug.setFatalMode(e.getStateChange() == ItemEvent.SELECTED));
    _fatalCB.setToolTipText("Toggles the global setting for fatal error debug messages.");
    add(_fatalCB);

    _logCB = new JCheckBox("Capture logging messages", Debug.getLogMode());
    _logCB.addItemListener(e -> Debug.setLogMode(e.getStateChange() == ItemEvent.SELECTED));
    _logCB.setToolTipText("Toggles the global setting for logging messages.");
    add(_logCB);

    _assertCB = new JCheckBox("Capture failed assertions", Debug.getAssertionMode());
    _assertCB.addItemListener(e -> Debug.setAssertionMode(e.getStateChange() == ItemEvent.SELECTED));
    _assertCB.setToolTipText("Toggles the global setting for failed assertion messages.");
    add(_assertCB);
      
    _stdoutCB = new JCheckBox("Capture stdout", Debug.getStdoutCaptureMode());
    _stdoutCB.addItemListener(e -> Debug.setStdoutCaptureMode(e.getStateChange() == ItemEvent.SELECTED));
    _stdoutCB.setToolTipText("If set, messages to stdout are shown as standard messages");
    add(_stdoutCB);
      
    _stderrCB = new JCheckBox("Capture stderr", Debug.getStderrCaptureMode());
    _stderrCB.addItemListener(e -> Debug.setStderrCaptureMode(e.getStateChange() == ItemEvent.SELECTED));
    _stderrCB.setToolTipText("If set, messages to stderr are shown as error messages");
    add(_stderrCB);

    Debug.addModeChangeListener(this);
  }

  /**
   *  The state of a debug mode is changed.
   *  @param  modeID  mode number
   *  @param  state   new state
   */
  @Override
  public void modeChanged(int modeID, boolean state) {
    final int     id = modeID;
    final boolean st = state;

    SwingUtilities.invokeLater(() -> {
      switch (id) {
      case TRACE:
        _traceCB.setSelected(st);
        break;

      case MESSAGE:
        _messageCB.setSelected(st);
        break;

      case WARNING:
        _warningCB.setSelected(st);
        break;

      case DebugConstants.ERROR:
        _errorCB.setSelected(st);
        break;

      case FATAL:
        _fatalCB.setSelected(st);
        break;

      case LOG:
        _logCB.setSelected(st);
        break;

      case ASSERT:
        _assertCB.setSelected(st);
        break;
      }
    });
  }

  /**
   *  The state of the capturing of stdout is changed.
   *  @param state  new state
   */
  @Override
  public void stdoutCaptureChanged(boolean state) {
    final boolean st = state;
    
    SwingUtilities.invokeLater(() -> _stdoutCB.setSelected(st));
  }

  /**
   *  The state of the capturing of stderr is changed.
   *  @param state  new state
   */
  @Override
  public void stderrCaptureChanged(boolean state) {
    final boolean st = state;
    
    SwingUtilities.invokeLater(() -> _stderrCB.setSelected(st));
  }

}
