// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 1999-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================

package de.caff.util.debug;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.Types;
import de.caff.util.Utility;

import javax.swing.*;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.*;
import java.text.StringCharacterIterator;
import java.util.Collection;
import java.util.LinkedList;
import java.util.Locale;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 *  General tool for debugging purposes.
 * <p>
 *  This class defines static methods allowing output of debugging messages and 
 *  assertion checks.
 *  <p>
 *  Debugging messages fall into 6 categories handled by appropriately named methods.
 *  <ul>
 *  <li>{@code trace()} -- trace messages (ultra low level priority)</li>
 *  <li>{@code message()} -- standard messages (low level priority)</li>
 *  <li>{@code warn()} -- warning messages (normal priority)</li>
 *  <li>{@code error()} -- error messages (high priority)</li>
 *  <li>{@code fatal()} -- fatal error messages (ultra high priority),
 *                              allowing to interrupt program</li>
 *  <li>{@code log()} -- logging messages</li>
 *  </ul>
 *  Each of these methods is overloaded to accept 
 *  <ul>
 *  <li>a String parameter</li>
 *  <li>an Exception parameter</li>
 *  <li>a format parameter and possible argument using {@link Format#format(Object, Object...) internal format} escapes.</li>
 *  </ul>
 *  There are also variants where a {@code f} character is appended to the name (eg. {@code tracef()} or {@code logf()})
 *  which use standard {@code String.format()} format. It is possible to set the locale for this formatting
 *  via Java property {@code debug.locale} (evaluated when this class is initialized).
 *  <p>
 *  There is an {@code insist()} (formerly called {@code assert} until it became a Java keyword)
 *  method allowing to define assertions, too.
 *  <p>
 *  Each of the 6 categories and the handling of assertions can be switched on or off
 *  globally. Switching a category off means that debug messages in that category are not captured but
 *  discarded. Discarding is done as early as possible so there is only a small overhead with discarded messages
 *  (a method call, possibly parameter boxing, one if).
 *  Capturing <b>a lot</b> of messages may slow down a program considerably (here a lot depends on the
 *  registered listeners).
 *  <p>
 *  For every category there's a dedicated listener interface allowing listeners to
 *  register for that kind of messages (including an {@code AnyMessageDebugListener}
 *  interface for registration on all categories). These listeners will receive <em>raw</em>
 *  messages, i.e. get directly the messages as they are sent in the code. There's another
 *  {@code CookedMessageDebugListener} interface where listeners receive <em>cooked</em>
 *  messages. Cooked messages include the time and the sender's file position as well as an indicator
 *  of the category of the message.
 *  <p>
 *  There are no listeners registered by default but there are convenience methods allowing
 *  to create a simple listener allowing output on stdout (see {@link #installCookedOutput()})
 *  or a stream (see {@link #installCookedOutput(java.io.OutputStream)}) and a method which
 *  creates a JFrame to display the messages and allowing switching of the categories
 *  (see {@link #createDebugWindow()} or {@link #createDebugWindow(boolean)}).
 *  <p>
 *  It's possible to capture the standard output streams of the program. When this is done
 *  text printed to <tt>stdout</tt> is send via the {@code message()} interface and text
 *  printed to <tt>stderr</tt> is send via the {@code error()} interface. When captured these
 *  streams are buffered until another kind of message arrives or capturing is switched off.
 *  <p>
 *  You should prefer the following syntax when writing debug messages
 *  <pre>
 *    Debug.trace("foo=%1", foo);
 *  </pre>
 *  instead of
 *  <pre>
 *    Debug.trace("foo="+foo);
 *  </pre>
 *
 *  @see Format
 *  @see AnyMessageDebugListener
 *  @see AssertionMessageDebugListener
 *  @see CookedMessageDebugListener
 *  @see ErrorMessageDebugListener
 *  @see FatalMessageDebugListener
 *  @see LogMessageListener
 *  @see StandardMessageDebugListener
 *  @see WarningMessageDebugListener
 *
 *  @author Rammi
 */
public class Debug
        implements DebugConstants
{
  private static long             _mask            = 0;

  private static final boolean    _bufferStreams   = false;

  @NotNull
  private static final DebugMessageCook _cook            = new DebugMessageCook();
  @NotNull
  private static final Collection<TraceMessageDebugListener>     _traceList       =
          Types.synchronizedCollection(new LinkedList<>());
  @NotNull
  private static final Collection<StandardMessageDebugListener>  _stdList         =
          Types.synchronizedCollection(new LinkedList<>());
  @NotNull
  private static final Collection<WarningMessageDebugListener>   _warnList        =
          Types.synchronizedCollection(new LinkedList<>());
  @NotNull
  private static final Collection<ErrorMessageDebugListener>     _errorList       =
          Types.synchronizedCollection(new LinkedList<>());
  @NotNull
  private static final Collection<FatalMessageDebugListener>     _fatalList       =
          Types.synchronizedCollection(new LinkedList<>());
  @NotNull
  private static final Collection<AssertionMessageDebugListener> _assertList      =
          Types.synchronizedCollection(new LinkedList<>());
  @NotNull
  private static final Collection<LogMessageListener>            _logList         =
          Types.synchronizedCollection(new LinkedList<>());
  @NotNull
  private static final Collection<ModeChangeListener>            _modeList        =
          Types.synchronizedCollection(new LinkedList<>());

  private static PrintStream      _oldStdout       = null;
  private static StringBuffer     _collectedStdout = null;  // note: no StringBuilder, does not sync
  private static PrintStream      _oldStderr       = null;
  private static StringBuffer     _collectedStderr = null;  // note: no StringBuilder, does not sync

  private static boolean          _haveInitializedFromProperties = false;
  private static JFrame           _debugWindow     = null;
  @NotNull
  private static final Object    _loggerConnectorSyncHelp = new Object();
  private static LoggerConnector _loggerConnector  = null;
  @NotNull
  private static final Locale stringFormatLocale;

  static {
    // _cook receives all messages
    addAnyMessageDebugListener(_cook);

    // allow to set locale via Java property
    final String loc = System.getProperty("debug.locale");
    stringFormatLocale = loc != null
            ? Locale.forLanguageTag(loc)
            : Locale.getDefault();
  }

  /**
   *  A Java interface for listeners which wants to be informed about the internal
   *  mode of the debug interface.
   */
  public interface ModeChangeListener {
    /**
     *  The debugging mode for an id is changed.
     *  @see DebugConstants
     *  @param  modeID  one of {@code TRACE}, {@code MESSAGE}, {@code WARNING},
     *                  {@code ERROR}, {@code FATAL}, {@code LOG},
     *                  or {@code ASSERT}
     *  @param  state   new state of the given category
     */
    void modeChanged(int modeID, boolean state);

    /**
     *  The capturing mode for stdout is changed.
     *  @param  state   new capturing mode
     */
    void stdoutCaptureChanged(boolean state);

    /**
     *  The capturing mode for stderr is changed.
     *  @param  state   new capturing mode
     */
    void stderrCaptureChanged(boolean state);
  }

  /**
   * Logger connector for connection to newer Log4j system.
   */
  private static class LoggerConnector
          implements AnyMessageDebugListener
  {
    private final Logger logger;

    /**
     * Constructor.
     */
    private LoggerConnector()
    {
      logger = Logger.getLogger("de.caff.util.debug");
    }

    /**
     * Get the logger.
     * @return logger
     */
    public Logger getLogger()
    {
      return logger;
    }

    /**
     * Standard format.
     * @param msg  message text
     * @param pos  position of message
     * @return formatted message
     */
    private static String format(@NotNull String msg,
                                 @NotNull String pos)
    {
      return msg + " [" + pos + "]";
    }

    /**
     * Receive a raw message about a failed assertion and return, whether an exception
     * shall be thrown.
     *
     * @param msg the message
     * @param pos position
     * @return {@code true} -- create exception<br>
     *         {@code false} -- don't create exception
     */
    @Override
    public boolean receiveFailedAssertionMessage(@NotNull String msg,
                                                 @NotNull String pos)
    {
      logger.severe("Failed assertion: " + format(msg, pos));
      return true;
    }

    /**
     * Receive a raw error message.
     *
     * @param msg the message
     * @param pos position
     */
    @Override
    public void receiveErrorMessage(@NotNull String msg,
                                    @NotNull String pos)
    {
      logger.warning(format(msg, pos));
    }

    /**
     * Receive a raw fatal error message and return whether the program shall be
     * exited.
     *
     * @param msg the message
     * @param pos position
     * @return unequal {@code 0} -- exit the program<br>
     *         {@code 0} -- go on
     */
    @Override
    public int receiveFatalMessage(@NotNull String msg,
                                   @NotNull String pos)
    {
      logger.severe(format(msg, pos));
      return 1;
    }

    /**
     * Receive a raw logging debug message.
     *
     * @param msg the message
     * @param pos position
     */
    @Override
    public void receiveLogMessage(@NotNull String msg,
                                  @NotNull String pos)
    {
      logger.log(Level.INFO, format(msg, pos));
    }

    /**
     * Receive a raw standard debug message.
     *
     * @param msg the message
     * @param pos position
     */
    @Override
    public void receiveStandardMessage(@NotNull String msg,
                                       @NotNull String pos)
    {
      logger.fine(format(msg, pos));
    }

    /**
     * Receive a raw trace debug message.
     *
     * @param msg the message
     * @param pos position
     */
    @Override
    public void receiveTraceMessage(@NotNull String msg,
                                    @NotNull String pos)
    {
      logger.finest(format(msg, pos));
    }

    /**
     * Receive a raw warning debug message.
     *
     * @param msg the message
     * @param pos position
     */
    @Override
    public void receiveWarningMessage(@NotNull String msg,
                                      @NotNull String pos)
    {
      logger.warning(format(msg, pos));
    }
  }

  /**
   *  Add a ModeChangeListener.
   *  @param  listener   the listener to add
   */
  public static void addModeChangeListener(@NotNull ModeChangeListener listener)
  {
    _modeList.add(listener);
  }

  /**
   *  Remove a mode change listener
   *  @param  listener   the listener to remove
   */
  public static void removeModeChangeListener(@NotNull ModeChangeListener listener)
  {
    _modeList.remove(listener);
  }

  /**
   *  Informs the registered listeners of a mode change.
   *  @param  modeID   mode number
   *  @param  state    new state
   */
  private static void informModeChangeListeners(int modeID, boolean state)
  {
    for (ModeChangeListener listener: _modeList) {
      listener.modeChanged(modeID, state);
    }
  }

  /**
   *  Informs the registered listeners of a mode change concerning stdout capturing.
   *  @param  state    new state
   */
  private static void informStdoutCaptureModeListeners(boolean state)
  {
    for (ModeChangeListener listener: _modeList) {
      listener.stdoutCaptureChanged(state);
    }
  }

  /**
   *  Informs the registered listeners of a mode change concerning stdout capturing.
   *  @param  state    new state
   */
  private static void informStderrCaptureModeListeners(boolean state)
  {
    for (ModeChangeListener listener: _modeList) {
      listener.stderrCaptureChanged(state);
    }
  }

  /**
   *  Append a trace message listener. This listener will receive debug messages
   *  sent with {@code trace}.
   *  @param  listener  new listener
   */
  public static void addTraceMessageDebugListener(@NotNull TraceMessageDebugListener listener)
  {
    _traceList.add(listener);
  }

  /**
   *  Remove a trace message listener
   *  @param  listener  listener to remove
   */
  public static void removeTraceMessageDebugListener(@NotNull TraceMessageDebugListener listener)
  {
    _traceList.remove(listener);
  }

  /**
   *  Append a standard debug message listener. This listener will receive debug
   *  messages sent with {@code message}.
   *  @param  listener  new listener
   */
  public static void addStandardMessageDebugListener(@NotNull StandardMessageDebugListener listener)
  {
    _stdList.add(listener);
  }

  /**
   *  Remove a standard debug message listener.
   *  @param  listener  listener to remove
   */
  public static void removeStandardMessageDebugListener(@NotNull StandardMessageDebugListener listener)
  {
    _stdList.remove(listener);
  }

  /**
   *  Append a warning message debug listener. This listener will receive debug
   *  messages sent with {@code warn}.
   *  @param  listener  new listener
   */
  public static void addWarningMessageDebugListener(@NotNull WarningMessageDebugListener listener)
  {
    _warnList.add(listener);
  }

  /**
   *  Remove a warning message debug listener.
   *  @param  listener  listener to remove
   */
  public static void removeWarningMessageDebugListener(@NotNull WarningMessageDebugListener listener)
  {
    _warnList.remove(listener);
  }

  /**
   *  Append an error message debug listener. This listener will receive messages
   *  sent with {@code error}.
   *  @param  listener  new listener
   */
  public static void addErrorMessageDebugListener(@NotNull ErrorMessageDebugListener listener)
  {
    _errorList.add(listener);
  }

  /**
   *  Remove an error message debug listener.
   *  @param  listener  listener to remove
   */
  public static void removeErrorMessageDebugListener(@NotNull ErrorMessageDebugListener listener)
  {
    _errorList.remove(listener);
  }

  /**
   *  Append a fatal error message debug listener. This listener will receive messages
   *  sent with {@code fatal}.
   *  @param  listener  new listener
   */
  public static void addFatalMessageDebugListener(@NotNull FatalMessageDebugListener listener)
  {
    _fatalList.add(listener);
  }

  /**
   *  Remove a fatal error message debug listener.
   *  @param  listener  listener to remove
   */
  public static void removeFatalMessageDebugListener(@NotNull FatalMessageDebugListener listener)
  {
    _fatalList.remove(listener);
  }

  /**
   *  Add an assert message debug listener. This listener will receive assert messages on
   *  failed assertions and can initiate a exception.
   *  @param  listener  new listener
   */
  public static void addAssertionMessageDebugListener(@NotNull AssertionMessageDebugListener listener)
  {
    _assertList.add(listener);
  }

  /**
   *  Remove an assert message debug listener.
   *  @param  listener  listener to remove
   */
  public static void removeAssertionMessageDebugListener(@NotNull AssertionMessageDebugListener listener)
  {
    _assertList.remove(listener);
  }

  /**
   *  Append a log message listener. This listener will receive messages sent with 
   *  {@code log}.
   *  @param  listener  new listener
   */
  public static void addLogMessageListener(@NotNull LogMessageListener listener)
  {
    _logList.add(listener);
  }

  /**
   *  Remove a log message listener.
   *  @param  listener  listener to remove
   */
  public static void removeLogMessageListener(@NotNull LogMessageListener listener)
  {
    _logList.remove(listener);
  }

  /**
   *  Append a listener for cooked messages. This listener will receive messages
   *  of all allowed categories in an enhanced format.
   *  @param  listener  new listener
   */
  public static void addCookedMessageDebugListener(@NotNull CookedMessageDebugListener listener)
  {
    _cook.addListener(listener);
  }

  /**
   *  Remove a listener for cooked messages.
   *  @param  listener  listener to remove
   */
  public static void removeCookedMessageDebugListener(@NotNull CookedMessageDebugListener listener)
  {
    _cook.removeListener(listener);
  }

  /**
   *  Append a listener for all messages. This listener will receive messages of all
   *  allowed categories in raw format.
   *  @param  listener  new listener
   */
  public static void addAnyMessageDebugListener(@NotNull AnyMessageDebugListener listener)
  {
    addTraceMessageDebugListener(listener);
    addStandardMessageDebugListener(listener);
    addWarningMessageDebugListener(listener);
    addErrorMessageDebugListener(listener);
    addFatalMessageDebugListener(listener);
    addAssertionMessageDebugListener(listener);
    addLogMessageListener(listener);
  }

  /**
   *  Remove a listener for all messages.
   *  @param  listener  listener to remove
   */
  public static void removeAnyMessageDebugListener(@NotNull AnyMessageDebugListener listener)
  {
    removeTraceMessageDebugListener(listener);
    removeStandardMessageDebugListener(listener);
    removeWarningMessageDebugListener(listener);
    removeErrorMessageDebugListener(listener);
    removeFatalMessageDebugListener(listener);
    removeAssertionMessageDebugListener(listener);
    removeLogMessageListener(listener);
  }

  /**
   *  Set the debugging mask. This allows one step setting of which messages are
   *  handled or discarded.
   *  @param mask  debug mask. The possible values are defined in DebugConstants.
   */
  public static void setMask(long mask)
  {
    if (mask != _mask) {
      long diff = _mask ^ mask;
      for (int i = 0;   i < 64;   ++i) {
        if ((0x01L << i  &  diff) != 0) {
          informModeChangeListeners(i, (mask & 0x01L << i) != 0);
        }
      }
      _mask = mask;
    }
  }

  /**
   *  Get the active debugging mask.
   *  @return  debugging mask. The possible values are defined in DebugConstant.
   */
  public static long getMask()
  {
    return _mask;
  }

  /**
   *  Set the trace mode.
   *  @param  mode  {@code true} = trace messages are handled<br>
   *                {@code false} = trace messages are discarded
   */
  public static void setTraceMode(boolean mode)
  {
    if (getTraceMode() != mode) {
      if (mode) {
        _mask |= TRACE_FLAG;
      }
      else {
        _mask &= ~TRACE_FLAG;
      }
      informModeChangeListeners(TRACE, mode);
    }
  }

  /**
   *  Get the trace mode.
   *  @return  {@code true} = trace messages are handled<br>
   *           {@code false} = trace messages are discarded
   */
  public static boolean getTraceMode()
  {
    return (_mask & TRACE_FLAG) != 0;
  }

  /**
   *  Set the standard message mode.
   *  @param  mode  {@code true} = standard messages are handled<br>
   *                {@code false} = standard messages are discarded
   */
  public static void setStandardMode(boolean mode)
  {
    if (getStandardMode() != mode) {
      if (mode) {
        _mask |= MESSAGE_FLAG;
      }
      else {
        _mask &= ~MESSAGE_FLAG;
      }
      informModeChangeListeners(MESSAGE, mode);
    }
  }

  /**
   *  Get the standard message mode.
   *  @return  {@code true} = standard messages are handled<br>
   *           {@code false} = standard messages are discarded
   */
  public static boolean getStandardMode()
  {
    return (_mask & MESSAGE_FLAG) != 0;
  }

  /**
   *  Set the warning message mode.
   *  @param  mode  {@code true} = warning messages are handled<br>
   *                {@code false} = warning messages are discarded
   */
  public static void setWarningMode(boolean mode)
  {
    if (getWarningMode() != mode) {
      if (mode) {
        _mask |= WARNING_FLAG;
      }
      else {
        _mask &= ~WARNING_FLAG;
      }
      informModeChangeListeners(WARNING, mode);
    }
  }

  /**
   *  Get the warning message mode.
   *  @return  {@code true} = warning messages are handled<br>
   *           {@code false} = warning messages are discarded
   */
  public static boolean getWarningMode()
  {
    return (_mask & WARNING_FLAG) != 0;
  }

  /**
   *  Set the error message mode.
   *  @param  mode  {@code true} = error messages are handled<br>
   *                {@code false} = error messages are discarded
   */
  public static void setErrorMode(boolean mode)
  {
    if (getErrorMode() != mode) {
      if (mode) {
        _mask |= ERROR_FLAG;
      }
      else {
        _mask &= ~ERROR_FLAG;
      }
      informModeChangeListeners(ERROR, mode);
    }
  }

  /**
   *  Set the error message mode.
   *  @return  {@code true} = error messages are handled<br>
   *           {@code false} = error messages are discarded
   */
  public static boolean getErrorMode()
  {
    return (_mask & ERROR_FLAG) != 0;
  }

  /**
   *  Set the logging message mode.
   *  @param  mode  {@code true} = logging messages are handled<br>
   *                {@code false} = logging messages are discarded
   */
  public static void setLogMode(boolean mode)
  {
    if (getLogMode() != mode) {
      if (mode) {
        _mask |= LOG_FLAG;
      }
      else {
        _mask &= ~LOG_FLAG;
      }
      informModeChangeListeners(LOG, mode);
    }
  }

  /**
   *  Get the logging message mode.
   *  @return  {@code true} = logging messages are handled<br>
   *           {@code false} = logging messages are discarded
   */
  public static boolean getLogMode()
  {
    return (_mask & LOG_FLAG) != 0;
  }

  /**
   *  Set the fatal error message mode.
   *  @param  mode  {@code true} = fatal error messages are handled<br>
   *                {@code false} = fatal error messages are discarded (i.e. no exit)
   */
  public static void setFatalMode(boolean mode)
  {
    if (getFatalMode() != mode) {
      if (mode) {
        _mask |= FATAL_FLAG;
      }
      else {
        _mask &= ~FATAL_FLAG;
      }
      informModeChangeListeners(FATAL, mode);
    }
  }

  /**
   *  Get the fatal error message mode.
   *  @return  {@code true} = fatal error messages are handled<br>
   *           {@code false} = fatal error messages are discarded (i.e. no exit)
   */
  public static boolean getFatalMode()
  {
    return (_mask & FATAL_FLAG) != 0;
  }

  /**
   *  Set the assertion handling mode.
   *  @param  mode  {@code true} = assertions are handled<br>
   *                {@code false} = assertions are discarded (i.e. no Exception)
   */
  public static void setAssertionMode(boolean mode)
  {
    if (getAssertionMode() != mode) {
      if (mode) {
        _mask |= ASSERT_FLAG;
      }
      else {
        _mask &= ~ASSERT_FLAG;
      }
      informModeChangeListeners(ASSERT, mode);
    }
  }

  /**
   *  Get the assertion handling mode.
   *  @return  {@code true} = assertions are handled<br>
   *           {@code false} = assertions are discarded (i.e. no Exception)
   */
  public static boolean getAssertionMode()
  {
    return (_mask & ASSERT_FLAG) != 0;
  }

  /**
   *  Send a trace message to all listeners.
   *  @param  msg  the message
   */
  private static void distributeTraceMessage(@NotNull String msg)
  {
    sendCollectedStreams();
    final String callerPosition = getCallerPosition();
    for (TraceMessageDebugListener listener : _traceList) {
      listener.receiveTraceMessage(msg, callerPosition);
    }
  }

  /**
   *  Send a standard message to all listeners.
   *  @param  msg  the message
   */
  private static void distributeStandardMessage(@NotNull String msg)
  {
    sendCollectedStreams();
    final String callerPosition = getCallerPosition();
    for (StandardMessageDebugListener listener : _stdList) {
      listener.receiveStandardMessage(msg, callerPosition);
    }
  }

  /**
   *  Send a warning message to all listeners.
   *  @param  msg  the message
   */
  private static void distributeWarningMessage(@NotNull String msg)
  {
    sendCollectedStreams();
    final String callerPosition = getCallerPosition();
    for (WarningMessageDebugListener listener : _warnList) {
      listener.receiveWarningMessage(msg, callerPosition);
    }
  }

  /**
   *  Send an error message to all listeners.
   *  @param  msg  the message
   */
  private static void distributeErrorMessage(@NotNull String msg)
  {
    sendCollectedStreams();
    final String callerPosition = getCallerPosition();
    for (ErrorMessageDebugListener listener : _errorList) {
      listener.receiveErrorMessage(msg, callerPosition);
    }
  }

  /**
   *  Send a logging message to all listeners.
   *  @param  msg  the message
   */
  private static void distributeLogMessage(@NotNull String msg)
  {
    sendCollectedStreams();
    final String callerPosition = getCallerPosition();
    for (LogMessageListener listener : _logList) {
      listener.receiveLogMessage(msg, callerPosition);
    }
  }

  /**
   *  Send a fatal error message to all listeners.
   *  If any of them wants to exit, exit afterwards.
   *  @param  msg  the message
   */
  private static void distributeFatalMessage(@NotNull String msg)
  {
    sendCollectedStreams();

    int ret = 0;

    final String callerPosition = getCallerPosition();
    for (FatalMessageDebugListener listener : _fatalList) {
      int retVal = listener.receiveFatalMessage(msg, callerPosition);
      if (ret == 0 && retVal != 0) {
        ret = retVal;
      }
    }

    if (ret != 0) {
      // as we are crashing now, make sure at least the message is visible
      System.err.println(msg);
      System.exit(ret);
    }
  }

  /**
   *  Send an assertion message to all listeners.
   *  @param  msg  the message
   *  @return {@code true} if any of the listeners wants to throw an exception<br>
   *          {@code false} else
   */
  private static boolean distributeAssertionMessage(@NotNull String msg)
  {
    sendCollectedStreams();

    boolean ret = false;

    final String callerPosition = getCallerPosition();
    for (AssertionMessageDebugListener listener : _assertList) {
      boolean retVal = listener.receiveFailedAssertionMessage(msg,
                                                              callerPosition);
      if (!ret && retVal) {
        ret = retVal;
      }
    }

    return ret;
  }

  /**
   *  Create a stack dump from a Throwable.
   *  @param x  the Throwable
   *  @return stack dump
   *  @see #getErrorMessage(Throwable)
   */
  @NotNull
  public static String getStackDump(@NotNull Throwable x)
  {
    final StringWriter writer = new StringWriter();
    final PrintWriter pw = new PrintWriter(writer);
    x.printStackTrace(pw);
    pw.flush();
    return writer.getBuffer().toString();
  }

  /**
   * Get a (halfways) useful message from an exception.
   * @param x exception
   * @return error message
   * @see #getStackDump(Throwable)
   */
  @NotNull
  public static String getErrorMessage(@NotNull Throwable x)
  {
    String msg = x.getMessage();
    if (msg == null) {
      msg = x.getClass().getCanonicalName();
    }
    return msg;
  }

  /**
   *  Print a trace message without further arguments.
   *  @param msg  message
   */
  public static void trace(@NotNull String msg)
  {
    if ((_mask & TRACE_FLAG) != 0) {
      distributeTraceMessage(msg);
    }
  }

  /**
   *  Print a trace message from a throwable.
   *  @param x throwable argument
   */
  public static void trace(@NotNull Throwable x)
  {
    if ((_mask & TRACE_FLAG) != 0) {
      distributeTraceMessage(getStackDump(x));
    }
  }

  /**
   *  Print a trace message without further arguments.
   *  @param obj  Objekt
   */
  public static void trace(@NotNull Object obj)
  {
    if ((_mask & TRACE_FLAG) != 0) {
      distributeTraceMessage(Format.toString(obj));
    }
  }

  /**
   *  Print a trace message with up to nine arguments.
   *  @param msg  formatted message as defined by {@link Format#format(Object, Object...)}
   *  @param args arguments for the formatted message
   */
  public static void trace(@NotNull String msg, Object ... args)
  {
    if ((_mask & TRACE_FLAG) != 0) {
      distributeTraceMessage(Format.format(msg, args));
    }
  }

  /**
   * Print a trace message in the format used by {@link String#format(String, Object...)}.
   * @param format format specifier
   * @param args   further arguments
   */
  public static void tracef(@NotNull String format, Object ... args)
  {
    if ((_mask & TRACE_FLAG) != 0) {
      distributeTraceMessage(String.format(stringFormatLocale, format, args));
    }
  }

  /**
   *  Print a standard message without further arguments.
   *  @param msg  message
   */
  public static void message(@NotNull String msg)
  {
    if ((_mask & MESSAGE_FLAG) != 0) {
      distributeStandardMessage(msg);
    }
  }

  /**
   *  Print a standard message from a throwable.
   *  @param x throwable argument
   */
  public static void message(@NotNull Throwable x)
  {
    if ((_mask & TRACE_FLAG) != 0) {
      distributeStandardMessage(getStackDump(x));
    }
  }

  /**
   *  Print a standard message without further arguments.
   *  @param obj  object
   */
  public static void message(@Nullable Object obj)
  {
    if ((_mask & MESSAGE_FLAG) != 0) {
      distributeStandardMessage(Format.toString(obj));
    }
  }

  /**
   *  Print a standard message with up to nine arguments.
   *  @param msg  formatted message as defined by {@link Format#format(Object, Object...)}
   *  @param args arguments for the formatted message
   */
  public static void message(@NotNull String msg, Object ... args)
  {
    if ((_mask & MESSAGE_FLAG) != 0) {
      distributeStandardMessage(Format.format(msg, args));
    }
  }

  /**
   * Print a trace message in the format used by {@link String#format(String, Object...)}.
   * @param format format specifier
   * @param args   further arguments
   */
  public static void messagef(@NotNull String format, Object ... args)
  {
    if ((_mask & MESSAGE_FLAG) != 0) {
      distributeStandardMessage(String.format(stringFormatLocale, format, args));
    }
  }

  /**
   *  Print a warning message without further arguments.
   *  @param msg  message
   */
  public static void warn(@NotNull String msg)
  {
    if ((_mask & WARNING_FLAG) != 0) {
      distributeWarningMessage(msg);
    }
  }

  /**
   *  Print warning message from a throwable.
   *  @param x throwable argument
   */
  public static void warn(@NotNull Throwable x)
  {
    if ((_mask & WARNING_FLAG) != 0) {
      distributeWarningMessage(getStackDump(x));
    }
  }

  /**
   *  Print a warning message without further arguments.
   *  @param obj  message
   */
  public static void warn(@Nullable Object obj)
  {
    if ((_mask & WARNING_FLAG) != 0) {
      distributeWarningMessage(Format.toString(obj));
    }
  }

  /**
   *  Print a warning message with up to nine arguments.
   *  @param msg  formatted message as defined by {@link Format#format(Object, Object...)}
   *  @param args arguments for the formatted message
   */
  public static void warn(@NotNull String msg, Object ... args)
  {
    if ((_mask & WARNING_FLAG) != 0) {
      distributeWarningMessage(Format.format(msg, args));
    }
  }

  /**
   * Print a trace message in the format used by {@link String#format(String, Object...)}.
   * @param format format specifier
   * @param args   further arguments
   */
  public static void warnf(@NotNull String format, Object ... args)
  {
    if ((_mask & WARNING_FLAG) != 0) {
      distributeWarningMessage(String.format(stringFormatLocale, format, args));
    }
  }

  /**
   *  Print an error message without further arguments.
   *  @param msg  message
   */
  public static void error(@NotNull String msg)
  {
    if ((_mask & ERROR_FLAG) != 0) {
      distributeErrorMessage(msg);
    }
  }

  /**
   *  Print an error message for a throwable.
   *  @param x throwable argument
   */
  public static void error(@NotNull Throwable x)
  {
    if ((_mask & ERROR_FLAG) != 0) {
      distributeErrorMessage(getStackDump(x));
    }
  }

  /**
   *  Print an error message without further arguments.
   *  @param obj  objekt
   */
  public static void error(@Nullable Object obj)
  {
    if ((_mask & ERROR_FLAG) != 0) {
      distributeErrorMessage(Format.toString(obj));
    }
  }

  /**
   *  Print an error message with up to nine arguments.
   *  @param msg  formatted message as defined by {@link Format#format(Object, Object...)}
   *  @param args arguments for the formatted message
   */
  public static void error(@NotNull String msg, Object ... args)
  {
    if ((_mask & ERROR_FLAG) != 0) {
      distributeErrorMessage(Format.format(msg, args));
    }
  }

  /**
   * Print a trace message in the format used by {@link String#format(String, Object...)}.
   * @param format format specifier
   * @param args   further arguments
   */
  public static void errorf(@NotNull String format, Object ... args)
  {
    if ((_mask & ERROR_FLAG) != 0) {
      distributeErrorMessage(String.format(stringFormatLocale, format, args));
    }
  }

  /**
   *  Print a fatal error message without further arguments.
   *  This may end the program if a listener to fatal errors decides
   *  to do so.
   *  @param msg  message
   */
  public static void fatal(@NotNull String msg)
  {
    if ((_mask & FATAL_FLAG) != 0) {
      distributeFatalMessage(msg);
    }
  }

  /**
   *  Print a fatal error message for a throwable.
   *  This may end the program if a listener to fatal errors decides
   *  to do so.
   *  @param x throwable argument
   */
  public static void fatal(@NotNull Throwable x)
  {
    if ((_mask & FATAL_FLAG) != 0) {
      distributeFatalMessage(getStackDump(x));
    }
  }

  /**
   *  Print a fatal error message without further arguments.
   *  This may end the program if a listener to fatal errors decides
   *  to do so.
   *  @param obj   object
   */
  public static void fatal(@Nullable Object obj)
  {
    if ((_mask & FATAL_FLAG) != 0) {
      distributeFatalMessage(Format.toString(obj));
    }
  }

  /**
   *  Print a fatal error message with up to nine arguments.
   *  This may end the program if a listener to fatal errors decides
   *  to do so.
   *  @param msg  formatted message as defined by {@link Format#format(Object, Object...)}
   *  @param args arguments for the formatted message
   */
  public static void fatal(@NotNull String msg, Object ... args)
  {
    if ((_mask & FATAL_FLAG) != 0) {
      distributeFatalMessage(Format.format(msg, args));
    }
  }

  /**
   * Print a trace message in the format used by {@link String#format(String, Object...)}.
   * @param format format specifier
   * @param args   further arguments
   */
  public static void fatalf(@NotNull String format, Object ... args)
  {
    if ((_mask & FATAL_FLAG) != 0) {
      distributeFatalMessage(String.format(stringFormatLocale, format, args));
    }
  }

  /**
   *  Print a logging message without further arguments.
   *  @param msg  message
   */
  public static void log(@NotNull String msg)
  {
    if ((_mask & LOG_FLAG) != 0) {
      distributeLogMessage(msg);
    }
  }

  /**
   *  Print a logging message for a throwable.
   *  @param x throwable argument
   */
  public static void log(@NotNull Throwable x)
  {
    if ((_mask & LOG_FLAG) != 0) {
      distributeLogMessage(getStackDump(x));
    }
  }

  /**
   *  Print a logging message without further arguments.
   *  @param obj  object
   */
  public static void log(@Nullable Object obj)
  {
    if ((_mask & LOG_FLAG) != 0) {
      distributeLogMessage(Format.toString(obj));
    }
  }

  /**
   *  Print a logging message with up to nine arguments.
   *  @param msg  formatted message as defined by {@link Format#format(Object, Object...)}
   *  @param args arguments for the formatted message
   */
  public static void log(@NotNull String msg, Object ... args)
  {
    if ((_mask & LOG_FLAG) != 0) {
      distributeLogMessage(Format.format(msg, args));
    }
  }

  /**
   * Print a logging message in the format used by {@link String#format(String, Object...)}.
   * @param format format specifier
   * @param args   further arguments
   */
  public static void logf(@NotNull String format, Object ... args)
  {
    if ((_mask & LOG_FLAG) != 0) {
      distributeLogMessage(String.format(stringFormatLocale, format, args));
    }
  }

  /**
   *  Test an assertion. Depending on the registered listeners an assertion
   *  failure error is thrown.
   *  @see  AssertionFailedError
   *  @param  condition  if {@code false} the assertion has failed and the
   *                     listeners are informed with a null message
   */
  public static void insist(boolean condition)
  {
    if (!condition   &&   (_mask & ASSERT_FLAG) != 0) {
      if (distributeAssertionMessage("Assertion Failure!")) {
        throw new AssertionFailedError();
      }
    }
  }

  /**
   *  Test an assertion. Depending on the registered listeners an assertion
   *  failure error is thrown.
   *  @see  AssertionFailedError
   *  @param  condition  if {@code false} the assertion has failed and the
   *                     listeners are informed
   *  @param  msg        the message to send to the listeners on failures
   */
  public static void insist(boolean condition, @NotNull String msg)
  {
    if (!condition   &&   (_mask & ASSERT_FLAG) != 0) {
      if (distributeAssertionMessage(msg)) {
        throw new AssertionFailedError(msg);
      }
    }
  }

  /** Class name prefix for classes in the debug package. */
  private static final Package DEBUG_PACKAGE = Debug.class.getPackage();
  private static final String DEBUG_PACKAGE_PREFIX = DEBUG_PACKAGE.getName() + ".";

  /**
   * Get the position of a call to the debug package.
   * @return caller position
   */
  @NotNull
  private static String getCallerPosition()
  {
    final StackTraceElement[] stackTrace = Thread.currentThread().getStackTrace();
    final int len = stackTrace.length;
    for (int s = 1;  s < len;  ++s) {
      if (!stackTrace[s].getClassName().startsWith(DEBUG_PACKAGE_PREFIX)) {
        return stackTrace[s].toString();
      }
    }
    return "???";
  }

  /**
   *  Create a debug window which receives cooked messages.
   *  It contains some simple buttons allowing to change the categories and
   *  the capturing of stdout/stderr globally.
   *  This window does not exit when closing.
   *  @return  win  the window 
   */
  @NotNull
  public static JFrame createDebugWindow()
  {
    return createDebugWindow(false);
  }

  /**
   *  Create a debug window which receives cooked messages.
   *  It contains some simple buttons allowing to change the categories and
   *  the capturing of stdout/stderr globally.
   *  @param   exitOnClose call {@code System.exit()} when window is closing?
   *  @return  win  the window 
   */
  @NotNull
  public static JFrame createDebugWindow(boolean exitOnClose)
  {
    if (_debugWindow != null) {
      if (_debugWindow.isVisible()) {
        _debugWindow.setVisible(false);
      }
    }
    //    return new DebugMessageWindow(true);
    _debugWindow = new FilteringDebugMessageWindow();
    if (exitOnClose) {
      _debugWindow.addWindowListener(new WindowAdapter() {
        @Override
        public void windowClosing(WindowEvent e) {
          System.exit(0);
        }
      });
    }
    return _debugWindow;
  }

  /**
   *  Sends all buffered messages for stdout.
   */
  private static void sendCollectedStdout()
  {
    if (_collectedStdout != null) {
      // somewhat complicated logic allowing to recall this function again
      // in distributeStandardMessage without doing anything more
      String send = _collectedStdout.toString();
      _collectedStdout = null;
      distributeStandardMessage(send);
    }
  }

  /**
   *  Sends all buffered messages for stderr.
   */
  private static void sendCollectedStderr()
  {
    if (_collectedStderr != null) {
      // somewhat complicated logic allowing to recall this function again
      // in distributeErrorMessage without doing anything more
      String send = _collectedStderr.toString();
      _collectedStderr = null;
      distributeErrorMessage(send);
    }
  }

  /**
   *  Sends all buffered messages for stdout and stderr.
   */
  private static void sendCollectedStreams()
  {
    sendCollectedStdout();
    sendCollectedStderr();
  }

  /**
   *  Handle a message to stdout.
   *  @param  str  string written to stdout
   */
  static void writeStdoutMessage(@NotNull String str)
  {
    sendCollectedStderr();
    int length = str.length();
    if (length > 0) {
      if (str.charAt(length-1) == '\n') {
        str = str.substring(0, length-1);
      }

      if (_bufferStreams) {
        if (_collectedStdout == null) {
          _collectedStdout = new StringBuffer();
          str = "[STDOUT>\n"+str;
        }
        _collectedStdout.append(str).append('\n');
      }
      else {
         sendCollectedStdout();
        distributeStandardMessage("[STDERR>\n"+str);
     }
    }
  }

  /**
   *  Handle a message to stderr.
   *  @param  str  string written to stderr
   */
  static void writeStderrMessage(@NotNull String str)
  {
    sendCollectedStdout();
    int length = str.length();
    if (length > 0) {
      if (str.charAt(length-1) == '\n') {
        str = str.substring(0, length-1);
      }

      if (_bufferStreams) {
        if (_collectedStderr == null) {
          _collectedStderr = new StringBuffer();
          str = "[STDERR>\n"+str;
        }
        _collectedStderr.append(str).append('\n');
      }
      else {
        sendCollectedStderr();
        distributeErrorMessage("[STDERR>\n"+str);
      }
    }
  }

  /**
   *  How messages to stdout shall be handled.
   *  @param  capture   {@code true} = messages are captured<br>
   *                    {@code false} = message are handled normally
   */
  public static void setStdoutCaptureMode(boolean capture)
  {
    if (capture != getStdoutCaptureMode()) {
      if (capture) {
        // stdout umleiten
        if (!getStdoutCaptureMode()) {
          _oldStdout = System.out;
          System.setOut(new PrintStream(new DebuggingOutputStream(DebuggingOutputStream.STDOUT)));
        }
      }
      else {
        if (getStdoutCaptureMode()) {
          sendCollectedStdout();
          System.setOut(_oldStdout);
          _oldStdout = null;
        }
      }
      informStdoutCaptureModeListeners(capture);
    }
  }

  /**
   *  Get the actual capturing mode for stdout.
   *  @return {@code true} = stdout is captured<br>
   *          {@code false} = stdout is handled normally
   */
  public static boolean getStdoutCaptureMode()
  {
    return _oldStdout != null;
  }

  /**
   *  Get the stdout stream which prints to the console.
   *  @return the real stdout stream
   */
  public static PrintStream getConsoleStdout() {
    return getStdoutCaptureMode() ? _oldStdout : System.out;
  }

  /**
   *  How messages to stderr shall be handled.
   *  @param  capture   {@code true} = messages are captured<br>
   *                    {@code false} = message are handled normally
   */
  public static void setStderrCaptureMode(boolean capture)
  {
    if (capture != getStderrCaptureMode()) {
      if (capture) {
        // stdout umleiten
        if (!getStderrCaptureMode()) {
          _oldStderr = System.err;
          System.setErr(new PrintStream(new DebuggingOutputStream(DebuggingOutputStream.STDERR)));
        }
      }
      else {
        if (getStderrCaptureMode()) {
          sendCollectedStderr();
          System.setErr(_oldStderr);
          _oldStderr = null;
        }
      }
      informStderrCaptureModeListeners(capture);
    }
  }

  /**
   *  Get the actual capturing mode for stderr.
   *  @return {@code true} = stderr is captured<br>
   *          {@code false} = stderr is handled normally
   */
  public static boolean getStderrCaptureMode()
  {
    return _oldStderr != null;
  }

  /**
   *  Get the stderr stream which prints to the console.
   *  @return the real stderr stream
   */
  public static PrintStream getConsoleStderr() {
    return getStderrCaptureMode() ? _oldStderr : System.err;
  }

  /**
   *  Install a DebugListener which prints all messages in &quot;cooked&quot; form
   *  to the given output stream.
   *  @param  stream output stream
   */
  public static void installCookedOutput(@Nullable OutputStream stream)
  {
    if (stream != null) {
      addCookedMessageDebugListener(new SimpleOutputtingDebugListener(stream));
    }
  }

  /**
   *  Install a DebugListener which prints all messages in &quot;cooked&quot; form
   *  to the console.
   */
  public static void installCookedOutput()
  {
    installCookedOutput(getConsoleStdout());
  }

  /**
   *  Create an in-deep description of the fields of the given object.
   *  @param  obj  object to describe
   *  @return  description of {@code obj}
   */
  public static String getObjectInfo(@NotNull Object obj)
  {
    return ObjectInspector.getInfo(obj, false);
  }

  /**
   *  Create an in-deep description of the fields and the getter methods of the given object.
   *  Beware that all public non parameter methods starting with <tt>get</tt> will be called if
   *  the security manager allows, which might cause side-effects!
   *  @param  obj  object to describe
   *  @return  description of {@code obj}
   */
  public static String getObjectInfoIncludingGetters(@NotNull Object obj)
  {
    return ObjectInspector.getInfo(obj, true);
  }

  /**
   *  Initialize debugging features from properties.
   *  Depending on the settings of certain properties the initial debugging settings are set:
   *  <table border="1" summary="Supported Java Runtime Properties">
   *   <tr><th>Property</th><th>Type</th><th>Description</th></tr>
   *   <tr>
   *     <td>{@code debug.show.window}</td>
   *     <td>{@code boolean}</td>
   *     <td>Display a debug window?</td>
   *   </tr>
   *   <tr>
   *     <td>{@code debug.print.console}</td>
   *     <td>{@code boolean}</td>
   *     <td>Print debug messages to console/stdout?</td>
   *   </tr>
   *   <tr>
   *     <td>{@code debug.mask}</td>
   *     <td>{@code String}</td>
   *     <td>Combination of characters indicating which types of debug messages should be captured:
   *      <table>
   *        <caption>Character to Level Mapping</caption>
   *       <tr><td><tt>T</tt></td><td>Trace messages (lowest priority)</td></tr>
   *       <tr><td><tt>S</tt></td><td>Standard messages</td></tr>
   *       <tr><td><tt>W</tt></td><td>Warning messages</td></tr>
   *       <tr><td><tt>E</tt></td><td>Error messages</td></tr>
   *       <tr><td><tt>F</tt></td><td>Fatal error messages (highest priority)</td></tr>
   *       <tr><td><tt>L</tt></td><td>Log messages</td></tr>
   *       <tr><td><tt>A</tt></td><td>Assertion failure messages</td></tr>
   *      </table>
   *     </td>
   *   </tr>
   *  </table>
   *
   *  If neither a debug window is displayed nor something is printed to the console the mask is ignored and
   *  no type of message is captured.
   *  @param debugMaskDefault     default mask used if property {@code debug.mask} is not set
   *  @param showWindowDefault    default used if property {@code debug.show.window} is not set
   *  @param printConsoleDefault  default used if property {@code debug.print.console} is not set
   */
  public static void initFromProperties(long debugMaskDefault, boolean showWindowDefault, boolean printConsoleDefault)
  {
    boolean showWindow = Utility.getBooleanParameter("debug.show.window", showWindowDefault);
    if (_haveInitializedFromProperties) {
      // no need to do intialize everything twice
      if (showWindow) {
        createDebugWindow();
      }
      return;
    }
    _haveInitializedFromProperties = true;
    boolean printConsole = Utility.getBooleanParameter("debug.print.console", printConsoleDefault);
    if (showWindow || printConsole) {
      long debugMask = 0;
      String mask = Utility.getStringParameter("debug.mask", null);
      if (mask != null) {
        for (StringCharacterIterator it = new StringCharacterIterator(mask);  it.current() != StringCharacterIterator.DONE;  it.next()) {
          char ch = it.current();
          switch (Character.toUpperCase(ch)) {
          case TRACE_CHAR:
            debugMask |= TRACE_FLAG;
            break;

          case MESSAGE_CHAR:
            debugMask |= MESSAGE_FLAG;
            break;

          case WARNING_CHAR:
            debugMask |= WARNING_FLAG;
            break;

          case ERROR_CHAR:
            debugMask |= ERROR_FLAG;
            break;

          case FATAL_CHAR:
            debugMask |= FATAL_FLAG;
            break;

          case LOG_CHAR:
            debugMask |= LOG_FLAG;
            break;

          case ASSERT_CHAR:
            debugMask |= ASSERT_FLAG;
            break;

          case '1':
            setStdoutCaptureMode(true);
            break;

          case '2':
            setStderrCaptureMode(true);
            break;
          }
        }
      }
      else {
        debugMask = debugMaskDefault & DEBUG_ALL_MASK;
      }
      setMask(debugMask);
      if (showWindow) {
        createDebugWindow();
      }
      if (printConsole) {
        installCookedOutput();
      }
    }
    else {
      setMask(NO_DEBUG_MASK);
    }
  }

  /**
   *  Convenience method which calls {@link #initFromProperties(long, boolean, boolean)} with no debug window
   *  and no console output as default.
   *  @param debugMaskDefault compare {@link #initFromProperties(long, boolean, boolean)}
   */
  public static void initFromProperties(long debugMaskDefault)
  {
    initFromProperties(debugMaskDefault, false, false);
  }

  /**
   *  Connect this debugger to the standard Java logging system
   *  located in package java.util.debug and introduced in Java 1.4.
   * <p>
   *  Even if this method is called more than once, only one connector is created.
   *  @return logger connecting the debug system to the logging system
   */
  public static Logger connectToLogging()
  {
    synchronized (_loggerConnectorSyncHelp) {
      if (_loggerConnector == null) {
        _loggerConnector = new LoggerConnector();
        addAnyMessageDebugListener(_loggerConnector);
      }
    }
    return _loggerConnector.getLogger();
  }
}

