// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2015-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.concurrent;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;

import java.util.Iterator;
import java.util.concurrent.*;
import java.util.concurrent.atomic.AtomicLong;

/**
 * Same as thread pool executor, but supporting priorities.
 * <p>
 * Added tasks of the same priority are run in order.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class PriorityThreadPoolExecutor
        implements PriorityExecutor
{
  private final ThreadPoolExecutor threadPoolExecutor;

  /** Sequence number for in order execution. */
  private final AtomicLong sequence = new AtomicLong(Long.MIN_VALUE);
  /** Priority blocking queue. */
  private final PriorityBlockingQueue<PriorityTask> queue = new PriorityBlockingQueue<>();

  /**
   * Constructor.
   * @param corePoolSize    core pool size
   * @param maximumPoolSize maximum pool size
   * @param keepAliveTime   keep alive time
   * @param unit            keep alive time unit
   * @see java.util.concurrent.ThreadPoolExecutor#ThreadPoolExecutor(int, int, long, java.util.concurrent.TimeUnit, java.util.concurrent.BlockingQueue)
   */
  @SuppressWarnings("unchecked")
  public PriorityThreadPoolExecutor(int corePoolSize,
                                    int maximumPoolSize,
                                    long keepAliveTime,
                                    @NotNull TimeUnit unit)
  {
    threadPoolExecutor = new ThreadPoolExecutor(corePoolSize,
                                                maximumPoolSize,
                                                keepAliveTime,
                                                unit,
                                                (BlockingQueue)queue);  // evil, but as the only way an executable is added is
                                                                        // via this class things stay sane, compare execute()
  }

  /**
   * Constructor.
   * @param corePoolSize    core pool size
   * @param maximumPoolSize maximum pool size
   * @param keepAliveTime   keep alive time
   * @param unit            keep alive time unit
   * @param threadFactory   thread creation factory
   * @see java.util.concurrent.ThreadPoolExecutor#ThreadPoolExecutor(int, int, long, java.util.concurrent.TimeUnit, java.util.concurrent.BlockingQueue, java.util.concurrent.ThreadFactory)
   */
  @SuppressWarnings("unchecked")
  public PriorityThreadPoolExecutor(int corePoolSize,
                                    int maximumPoolSize,
                                    long keepAliveTime,
                                    @NotNull TimeUnit unit,
                                    @NotNull ThreadFactory threadFactory)
  {
    threadPoolExecutor = new ThreadPoolExecutor(corePoolSize,
                                                maximumPoolSize,
                                                keepAliveTime,
                                                unit,
                                                (BlockingQueue)queue, // evil, but see explanation in 1st constructor
                                                threadFactory);
  }

  /**
   * Constructor.
   * @param corePoolSize    core pool size
   * @param maximumPoolSize maximum pool size
   * @param keepAliveTime   keep alive time
   * @param unit            keep alive time unit
   * @param rejectedExecutionHandler rejected execution handler
   * @see java.util.concurrent.ThreadPoolExecutor#ThreadPoolExecutor(int, int, long, java.util.concurrent.TimeUnit, java.util.concurrent.BlockingQueue, java.util.concurrent.ThreadFactory, java.util.concurrent.RejectedExecutionHandler)
   */
  @SuppressWarnings("unchecked")
  public PriorityThreadPoolExecutor(int corePoolSize,
                                    int maximumPoolSize,
                                    long keepAliveTime,
                                    @NotNull TimeUnit unit,
                                    @NotNull RejectedExecutionHandler rejectedExecutionHandler)
  {
    threadPoolExecutor = new ThreadPoolExecutor(corePoolSize,
                                                maximumPoolSize,
                                                keepAliveTime,
                                                unit,
                                                (BlockingQueue)queue, // evil, but see explanation in 1st constructor
                                                rejectedExecutionHandler);
  }

  /**
   * Constructor.
   * @param corePoolSize    core pool size
   * @param maximumPoolSize maximum pool size
   * @param keepAliveTime   keep alive time
   * @param unit            keep alive time unit
   * @param threadFactory   thread creation factory
   * @param rejectedExecutionHandler rejected execution handler
   * @see java.util.concurrent.ThreadPoolExecutor#ThreadPoolExecutor(int, int, long, java.util.concurrent.TimeUnit, java.util.concurrent.BlockingQueue, java.util.concurrent.ThreadFactory, java.util.concurrent.RejectedExecutionHandler)
   */
  @SuppressWarnings("unchecked")
  public PriorityThreadPoolExecutor(int corePoolSize,
                                    int maximumPoolSize,
                                    long keepAliveTime,
                                    @NotNull TimeUnit unit,
                                    @NotNull ThreadFactory threadFactory,
                                    @NotNull RejectedExecutionHandler rejectedExecutionHandler)
  {
    threadPoolExecutor = new ThreadPoolExecutor(corePoolSize,
                                                maximumPoolSize,
                                                keepAliveTime,
                                                unit,
                                                (BlockingQueue)queue, // evil, but see explanation in 1st constructor
                                                threadFactory,
                                                rejectedExecutionHandler);
  }

  /**
   * Execute a task with the given priority.
   * @param priority priority of task
   * @param mark     mark which allows access to objects in queue.
   *                 Not all implementation need to take care of this.
   * @param task     task to run
   * @return an id allowing later access to this task
   * @see #removeTasksWithMark(Object)
   *
   */
  @Override
  public long execute(@NotNull Priority priority, @Nullable Object mark, @NotNull Runnable task)
  {

    final long id = sequence.getAndIncrement();
    threadPoolExecutor.execute(new PriorityTask(priority,
                                                mark,
                                                null,
                                                id,
                                                task));
    return id;
  }

  /**
   * Execute a task with the given priority.
   *
   * @param priority priority of task
   * @param mark     mark which allows access to objects in queue, and also gets informed
   *                 when the task is enqueued, started and finished.
   *                 Not all implementation need to take care of this.
   * @param task     task to run
   * @return an id allowing later access to this task, or {@link #NO_ID}
   * if access is no longer possible
   */
  @Override
  public long executeWithLifeCycle(@NotNull Priority priority, @NotNull TaskLifeCycleListener mark,
                                   @NotNull Runnable task)
  {
    final long id = sequence.getAndIncrement();
    threadPoolExecutor.execute(new PriorityTask(priority,
                                                mark,
                                                mark,
                                                id,
                                                task));
    return id;
  }

  /**
   * Remove all tasks which where started with the given mark.
   * @param mark mark to check for
   * @return number of removed tasks if removing was possible, or {@link #CANT_REMOVE}
   *         if the executor does not allow removing
   */
  @Override
  public int removeTasksWithMark(@NotNull Object mark)
  {
    int count = 0;
    for (Iterator<PriorityTask> it = queue.iterator();
            it.hasNext(); ) {
      if (it.next().isMarked(mark)) {
        it.remove();
        ++count;
      }
    }
    return count;
  }

  /**
   * Remove the task with the given id.
   * @param id id
   * @return {@code true} if a rask with the given id was found and removed<br>
   *         {@code false}: if none was found
   */
  @Override
  public boolean removeTaskWithId(long id)
  {
    // silently accepting NO_ID which possibly could happen after 2^64 tasks
    for (Iterator<PriorityTask> it = queue.iterator();
         it.hasNext();  ) {
      if (it.next().sequenceNumber == id) {
        it.remove();
        return true;
      }
    }
    return false;
  }

  /**
   *  Helper class for priority sorting.
   */
  private static final class PriorityTask
          implements Comparable<PriorityTask>,
                     Runnable
  {
    @NotNull
    private final Priority priority;

    private final long sequenceNumber;
    @NotNull
    private final Runnable task;
    @Nullable
    private final Object mark;
    @Nullable
    private final TaskLifeCycleListener listener;

    /**
     * Constructor.
     * @param priority       priority of task
     * @param mark           task mark
     * @param listener       task life cycle listener
     * @param sequenceNumber sequence number of task
     * @param task           task to run
     */
    PriorityTask(@NotNull Priority priority,
                 @Nullable Object mark,
                 @Nullable TaskLifeCycleListener listener,
                 long sequenceNumber,
                 @NotNull Runnable task)
    {
      this.priority = priority;
      this.mark = mark;
      this.listener = listener;
      this.sequenceNumber = sequenceNumber;
      this.task = task;
      if (listener != null) {
        listener.taskEnqueued();
      }
    }

    /**
     * Is this task tagged with the given mark.
     * @param checkMark mark to check
     * @return {@code true}: if this task has a non-null mark and this equals the given mark<br>
     *         {@code false}; if this task has no mark or is tagged with another mark
     */
    boolean isMarked(@NotNull Object checkMark)
    {
      return checkMark.equals(mark);
    }

    /**
     * Run the underlying task.
     */
    @Override
    public void run()
    {
      if (listener != null) {
        listener.taskStarting();
        try {
          task.run();
        } finally {
          listener.taskFinished();
        }
      }
      else {
        task.run();
      }
    }

    /**
     * Compares this object with the specified object for order.
     * The comparison is done in a way that priority counts most,
     * but for the same priority tasks are ordered by their sequence number.
     *
     * @param o the object to be compared.
     * @return a negative integer, zero, or a positive integer as this object
     * is less than, equal to, or greater than the specified object.
     * @throws ClassCastException if the specified object's type prevents it
     *                            from being compared to this object.
     */
    @Override
    public int compareTo(@NotNull PriorityTask o)
    {
      int result = o.priority.compareTo(priority);
      if (result != 0) {
        return result;
      }
      return Long.compare(sequenceNumber, o.sequenceNumber);
    }

    @Override
    public String toString()
    {
      return "PriorityTask{" +
             "priority=" + priority +
             ", sequenceNumber=" + sequenceNumber +
             ", mark=" + mark +
             ", task=" + task +
             '}';
    }
  }
}
