// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2015-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.concurrent;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;

/**
 * Executor which executes a task with a given priority.
 * On heavy load situations it is possible that low priority tasks starve
 * depending on the implementation of the executor.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public interface PriorityExecutor
{
  /** Return value for {@link #execute(Priority, Object, Runnable)} indicating no valid id. */
  long NO_ID = -1L;
  /** Return value for {@link #removeTasksWithMark(Object)} if removing is not supported. */
  int CANT_REMOVE = -1;

  /**
   * Priority executor which just executes the task directly without caring for priority.
   * As there is no queue, it neither cares for priority nor for mark.
   */
  PriorityExecutor DIRECT_EXECUTOR = new PriorityExecutor()
  {
    @Override
    public long execute(@NotNull Priority priority,
                        @Nullable Object mark, @NotNull Runnable task)
    {
      task.run();
      return NO_ID;
    }

    @Override
    public long executeWithLifeCycle(@NotNull Priority priority, @NotNull TaskLifeCycleListener mark, @NotNull Runnable task)
    {
      mark.taskEnqueued();
      mark.taskStarting();
      try {
        task.run();
      } finally {
        mark.taskFinished();
      }
      return 0;
    }

    @Override
    public int removeTasksWithMark(@NotNull Object mark)
    {
      return CANT_REMOVE;
    }

    @Override
    public boolean removeTaskWithId(long id)
    {
      return false;
    }
  };

  /**
   * Execute a task with the given priority.
   * @param priority priority of task
   * @param mark     mark which allows access to objects in queue.
   *                 Not all implementation need to take care of this.
   * @param task     task to run
   * @return an id allowing later access to this task, or {@link #NO_ID}
   *         if access is no longer possible
   */
  long execute(@NotNull Priority priority,
               @Nullable Object mark,
               @NotNull Runnable task);

  /**
   * Execute a task with the given priority.
   * @param priority priority of task
   * @param mark     mark which allows access to objects in queue, and also gets informed
   *                 when the task is enqueued, started and finished.
   *                 Not all implementation need to take care of this.
   * @param task     task to run
   * @return an id allowing later access to this task, or {@link #NO_ID}
   *         if access is no longer possible
   */
  long executeWithLifeCycle(@NotNull Priority priority,
                            @NotNull TaskLifeCycleListener mark,
                            @NotNull Runnable task);

  /**
   * Remove all tasks which where started with the given mark.
   * @param mark mark to check for
   * @return number of removed tasks if removing was possibe, or {@link #CANT_REMOVE}
   *         if the executor does not allow removing
   */
  int removeTasksWithMark(@NotNull Object mark);

  /**
   * Remove the task with the given id.
   * @param id id
   * @return {@code true} if a rask with the given id was found and removed<br>
   *         {@code false}: if none was found
   */
  boolean removeTaskWithId(long id);
}
