// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2015-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.concurrent;

import de.caff.annotation.NotNull;

/**
 * Execution priority.
 * <p>
 * In simple environments it is recommended to use only
 * {@link de.caff.util.concurrent.Priority#Low}
 * and
 * {@link de.caff.util.concurrent.Priority#High}
 * as it leaves finer granularity possibilities if required
 * later.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public enum Priority
{
  /** Lowest possible priority. */
  Lowest,
  /** Lower than low priority. */
  Lower,
  /** Low priority. */
  Low,
  /** Lower than medium priority. */
  MediumLow,
  /** Medium priority. */
  Medium,
  /** Higher than medium priority. */
  MediumHigh,
  /** High priority. */
  High,
  /** Higher than high priority. */
  Higher,
  /** Highest priority. */
  Highest;

  /** Cached values. */
  private static final Priority[] VALUES = values();

  /**
   * Get a priority which is increased one step if this
   * is not already the {@link #Highest highest} priority.
   * @return increased priority
   */
  @NotNull
  public Priority increase()
  {
    return increased(this);
  }

  /**
   * Get a priority which is decreased one step if this
   * is not already the {@link #Lowest lowest} priority.
   * @return decreased priority
   */
  @NotNull
  public Priority decrease()
  {
    return decreased(this);
  }

  /**
   * Get an increased priority.
   * @param priority priority to increase
   * @return priority increased by one step if possible,
   *         otherwise same priority as {@code priority}
   */
  @NotNull
  public static Priority increased(@NotNull Priority priority)
  {
    int v = priority.ordinal() + 1;
    if (v > VALUES.length) {
      return priority;
    }
    return VALUES[v];
  }

  /**
   * Get a decreased priority.
   * @param priority priority to decrease
   * @return priority decreased by one step if possible,
   *         otherwise same priority as {@code priority}
   */
  @NotNull
  public static Priority decreased(@NotNull Priority priority)
  {
    int v = priority.ordinal() - 1;
    if (v < 0) {
      return priority;
    }
    return VALUES[v];
  }
}
