// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2016-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.concurrent;

import de.caff.annotation.NotNull;

import java.util.concurrent.*;

/**
 * A future which is already existing in the present.
 * <p>
 * This is thought for situations where normally a Future is awaited,
 * but sometimes the object is already given. It is just a small wrapper
 * around the object.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since August.02, 2016
 */
public class PresentFuture<V>
        implements Future<V>
{
  private final V value;

  /**
   * Constructor.
   * @param value value which is returned by get()
   */
  public PresentFuture(V value)
  {
    this.value = value;
  }

  /**
   * Attempts to cancel execution of this task.  This attempt will
   * fail if the task has already completed, has already been cancelled,
   * or could not be cancelled for some other reason. If successful,
   * and this task has not started when <tt>cancel</tt> is called,
   * this task should never run.  If the task has already started,
   * then the <tt>mayInterruptIfRunning</tt> parameter determines
   * whether the thread executing this task should be interrupted in
   * an attempt to stop the task.
   *
   * <p>After this method returns, subsequent calls to {@link #isDone} will
   * always return <tt>true</tt>.  Subsequent calls to {@link #isCancelled}
   * will always return <tt>true</tt> if this method returned <tt>true</tt>.
   *
   * @param mayInterruptIfRunning <tt>true</tt> if the thread executing this
   *                              task should be interrupted; otherwise, in-progress tasks are allowed
   *                              to complete
   * @return <tt>false</tt> if the task could not be cancelled,
   * typically because it has already completed normally;
   * <tt>true</tt> otherwise
   */
  @Override
  public boolean cancel(boolean mayInterruptIfRunning)
  {
    return false;
  }

  /**
   * Returns <tt>true</tt> if this task was cancelled before it completed
   * normally.
   *
   * @return <tt>true</tt> if this task was cancelled before it completed
   */
  @Override
  public boolean isCancelled()
  {
    return false;
  }

  /**
   * Returns <tt>true</tt> if this task completed.
   * <p>
   * Completion may be due to normal termination, an exception, or
   * cancellation -- in all of these cases, this method will return
   * <tt>true</tt>.
   *
   * @return <tt>true</tt> if this task completed
   */
  @Override
  public boolean isDone()
  {
    return true;
  }

  /**
   * Waits if necessary for the computation to complete, and then
   * retrieves its result.
   *
   * @return the computed result
   * @throws CancellationException if the computation was cancelled
   * @throws ExecutionException    if the computation threw an
   *                               exception
   * @throws InterruptedException  if the current thread was interrupted
   *                               while waiting
   */
  @Override
  public V get() throws InterruptedException, ExecutionException
  {
    return value;
  }

  /**
   * Waits if necessary for at most the given time for the computation
   * to complete, and then retrieves its result, if available.
   *
   * @param timeout the maximum time to wait
   * @param unit    the time unit of the timeout argument
   * @return the computed result
   * @throws CancellationException if the computation was cancelled
   * @throws ExecutionException    if the computation threw an
   *                               exception
   * @throws InterruptedException  if the current thread was interrupted
   *                               while waiting
   * @throws TimeoutException      if the wait timed out
   */
  @Override
  public V get(long timeout, @NotNull TimeUnit unit) throws InterruptedException, ExecutionException, TimeoutException
  {
    return value;
  }
}
