// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2016-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.concurrent;

import de.caff.annotation.NotNull;
import de.caff.generics.OrderedPair;
import de.caff.generics.Types;

import java.util.*;
import java.util.concurrent.*;

/**
 * Parallel sorting.
 * This sort algorithm uses multiple threads to sort a list.
 * Whether it is faster or slower than normal sorting is hard to predict,
 * and depends on various factors like cache size etc.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class ParallelSort
{
  /**
   * Sort a list of comparable items in their natural order using multiple threads.
   * This will use all threads of the executor pool.
   * @param list             list to sort. Doas not need to be threadsafe unless it is also changed o
   *                         concurrently outside of this method.
   * @param executor         thread pool executor for parallel sort tasks
   * @param minDivisionSize  minimal size of a sublist before standard standard sorting is used
   * @param <T> list type
   * @throws ExecutionException on executor errors
   * @throws InterruptedException if execution was interrupted
   */
  public static <T extends Comparable<T>> void sort(@NotNull List<T> list,
                                                    @NotNull ThreadPoolExecutor executor,
                                                    int minDivisionSize) throws ExecutionException, InterruptedException
  {
    sort(list, executor, executor.getMaximumPoolSize(), minDivisionSize);
  }

  /**
   * Sort a list of comparable items in their natural order using multiple threads.
   * @param list             list to sort. Doas not need to be threadsafe unless it is also changed o
   *                         concurrently outside of this method.
   * @param executor         thread pool executor for parallel sort tasks
   * @param maxThreads       maximum number of parallel sort threads
   * @param minDivisionSize  minimal size of a sublist before standard standard sorting is used
   * @param <T> list type
   * @throws ExecutionException on executor errors
   * @throws InterruptedException if execution was interrupted
   */
  public static <T extends Comparable<T>> void sort(@NotNull List<T> list,
                                                    @NotNull ThreadPoolExecutor executor,
                                                    int maxThreads,
                                                    int minDivisionSize) throws ExecutionException, InterruptedException
  {
    sort(list,
         Types.naturalOrder(),
         executor,
         maxThreads,
         minDivisionSize);
  }

  /**
   * Sort a list by a given comparator using multiple threads.
   * This will use all threads of the executor pool.
   * @param list             list to sort. Doas not need to be threadsafe unless it is also changed o
   *                         concurrently outside of this method.
   * @param comparator       comparator defining the sort order, has to be threadsafe
   * @param executor         thread pool executor for parallel sort tasks
   * @param minDivisionSize  minimal size of a sublist before standard standard sorting is used
   * @param <T> list type
   * @throws ExecutionException on executor errors
   * @throws InterruptedException if execution was interrupted
   */
  public static <T> void sort(@NotNull List<T> list,
                              @NotNull Comparator<? super T> comparator,
                              @NotNull ThreadPoolExecutor executor,
                              int minDivisionSize) throws ExecutionException, InterruptedException
  {
    sort(list, comparator, executor, executor.getMaximumPoolSize(), minDivisionSize);
  }

  /**
   * Sort a list by a given comparator using multiple threads.
   * @param list             list to sort. Doas not need to be threadsafe unless it is also changed o
   *                         concurrently outside of this method.
   * @param comparator       comparator defining the sort order, has to be threadsafe
   * @param executor         thread pool executor for parallel sort tasks
   * @param maxThreads       maximum number of parallel sort threads, a positive integer
   * @param minDivisionSize  minimal size of a sublist before standard standard sorting is used
   * @param <T> list type
   * @throws ExecutionException on executor errors
   * @throws InterruptedException if execution was interrupted
   */
  public static <T> void sort(@NotNull List<T> list,
                              @NotNull Comparator<? super T> comparator,
                              @NotNull ThreadPoolExecutor executor,
                              int maxThreads,
                              int minDivisionSize) throws ExecutionException, InterruptedException
  {
    final int size = list.size();
    if (size < 2 * minDivisionSize) {
      Collections.sort(list, comparator);
      return;
    }
    if (size / maxThreads > minDivisionSize) {
      // use maximum number of threads
      doSort(list, comparator, executor, size / maxThreads);
    }
    else {
      doSort(list, comparator, executor, minDivisionSize);
    }
  }

  @SuppressWarnings("unchecked") // due to tricky but safe conversions
  private static <T> void doSort(@NotNull final List<T> list,
                                 @NotNull final Comparator<? super T> comparator,
                                 @NotNull ThreadPoolExecutor executor,
                                 final int divisionSize) throws ExecutionException, InterruptedException
  {
    final int numDivisions = list.size() / divisionSize;
    final List<Future<List<T>>> futures = new ArrayList<>(numDivisions);
    for (int d = 0;  d < numDivisions;  ++d) {
      final int start = d * divisionSize;
      futures.add(executor.submit(() -> {
        Object[] a = list.subList(start, Math.min(list.size(), start+divisionSize)).toArray();
        Arrays.sort(a, (Comparator)comparator);
        return (List<T>)Types.asList(a);
      }));
    }
    final List<OrderedPair<T, Iterator<T>>> iterators = new ArrayList<>(numDivisions);
    for (Future<List<T>> f : futures) {

      final Iterator<T> iterator = f.get().iterator();
      iterators.add(OrderedPair.create(iterator.next(), iterator));
    }
    ListIterator<T> setter = list.listIterator();
    while (iterators.size() > 1) {
      OrderedPair<T, Iterator<T>> minPair = null;
      for (OrderedPair<T, Iterator<T>> pair : iterators) {
        if (minPair == null) {
          minPair = pair;
        }
        else {
          if (comparator.compare(pair.first, minPair.first) < 0) {
            minPair = pair;
          }
        }
      }
      assert(minPair != null);
      setter.next();
      setter.set(minPair.first);
      if (minPair.second.hasNext()) {
        minPair.first = minPair.second.next();
      }
      else {
        iterators.remove(minPair);
      }
    }
    OrderedPair<T, Iterator<T>> lastPair = iterators.get(0);
    setter.next();
    setter.set(lastPair.first);
    while (lastPair.second.hasNext()) {
      setter.next();
      setter.set(lastPair.second.next());
    }
  }
}
