// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2016-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.concurrent;

import de.caff.annotation.NotNull;
import de.caff.util.Utility;

import java.security.AccessControlContext;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.util.*;
import java.util.concurrent.*;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * Class with static helper methods for thread pool execution.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since August 16, 2022
 */
public final class Exec
{
  /**
   * The default for the maximum allowed number of threads.
   * This can be changed by setting Java property {@code "caff.exec.maxthreads"},
   * preferably at startup via command line, but at least before this class is loaded, e.g. statically
   * in the startup class or early in the startup method.
   */
  public static final int DEFAULT_MAX_THREADS = Math.max(1, Utility.getIntParameter("caff.exec.maxthreads", Integer.MAX_VALUE));
  /** Default wait-to-compute ratio. */
  public static final double DEFAULT_WAIT_TO_COMPUTE_RATIO = 0.1;
  /** CPU utilization factor, leave 5% for computer and GUI. */
  public static final double DEFAULT_CPU_UTILIZATION_FACTOR = 0.95;

  /** Don't create */
  private Exec() {}

  /**
   * Calculate a useful number of threads for a task.
   * <p>
   * This uses the number of CPUs as a base together with the
   * parameters, to calculate a useful number of threads for
   * a thread pool.
   * @param cpuUtilizationFactor how much of the CPU should be used to run the threads,
   *                             a number between {@code 0.0} (none) and {@code 1.0} (everything).
   *                             Leave a bit of room for other tasks your computer has to perform.
   * @param waitToComputeRatio   value which defines the ratio of the time your tasks are waiting for input
   *                             against the time they spend actual computing. A number between
   *                             {@code 0.0} (no waiting, only computing) and {@code 1.0} (no computing,
   *                             only waiting).
   * @param maxThreads           upper limit to the number of threads, at least 1
   * @return useful value for the number of threads in a thread pool
   */
  public static int calcNumThreads(double cpuUtilizationFactor,
                                   double waitToComputeRatio,
                                   int maxThreads)
  {
   if (cpuUtilizationFactor < 0.0  ||  cpuUtilizationFactor > 1.0) {
     throw new IllegalArgumentException("cpuUtilizationFactor has to be between 0.0 (no CPU usage) and 1.0 (exclusive CPU usage), but is "+cpuUtilizationFactor);
   }
   if (waitToComputeRatio < 0.0  ||  waitToComputeRatio > 1.0) {
     throw new IllegalArgumentException("waitToComputeRatio has to be between 0.0 (no waiting, only computing) and 1.0 (no computing, only waiting), but is "+waitToComputeRatio);
   }
   if (maxThreads < 1) {
     throw new IllegalArgumentException("maxThreads has to be at least 1 (although that does not make much sense), but is "+maxThreads);
   }

   final int nCpu = Runtime.getRuntime().availableProcessors();
   return Math.min(maxThreads,
                   Math.max(1, (int)Math.round(nCpu * cpuUtilizationFactor * (1 + waitToComputeRatio))));
  }

  /**
   * Create a default thread pool executor with a useful number of threads,
   * a keep alive time of 0, a linked blocking queue, and a rejection handler
   * which discards all unexecutable threads.
   * @param cpuUtilizationFactor how much of the CPU should be used to run the threads,
   *                             a number between {@code 0.0} (none) and {@code 1.0} (everything).
   *                             Leave a bit of room for other tasks your computer has to perform.
   * @param waitToComputeRatio   value which defines the ratio of the time your tasks are waiting for input
   *                             against the time they spend actual computing. A number between
   *                             {@code 0.0} (no waiting, only computing) and {@code 1.0} (no computing,
   *                             only waiting).
   * @param maxThreads           upper limit to the number of threads, at least 1
   * @return prepared thread pool executor
   */
  @NotNull
  public static ThreadPoolExecutor createDefaultExecutor(double cpuUtilizationFactor,
                                                         double waitToComputeRatio,
                                                         int maxThreads)
  {
    final int nThreads = Exec.calcNumThreads(cpuUtilizationFactor, waitToComputeRatio, maxThreads);
    final ThreadPoolExecutor exec = new ThreadPoolExecutor(nThreads, nThreads, 0L, TimeUnit.MICROSECONDS,
                                                           new LinkedBlockingQueue<>());
    exec.setRejectedExecutionHandler(new ThreadPoolExecutor.DiscardPolicy());
    return exec;
  }

  /**
   * Create a default thread pool executor with a restricted but default number of threads,
   * a keep-alive time of 0, a linked blocking queue, and a rejection handler
   * which discards all unexecutable threads.
   * This calls {@link #createDefaultExecutor(double,double,int)} with
   * {@link #DEFAULT_CPU_UTILIZATION_FACTOR} and {@link #DEFAULT_WAIT_TO_COMPUTE_RATIO}.
   * @param maxThreads upper limit to the created number of threads
   * @return prepared thread pool executor
   */
  @NotNull
  public static ThreadPoolExecutor createDefaultExecutor(int maxThreads)
  {
    return createDefaultExecutor(DEFAULT_CPU_UTILIZATION_FACTOR, DEFAULT_WAIT_TO_COMPUTE_RATIO, maxThreads);
  }

  /**
   * Create a default thread pool executor with a default number of threads,
   * a keep-alive time of 0, a linked blocking queue, and a rejection handler
   * which discards all unexecutable threads.
   * This calls {@link #createDefaultExecutor(double,double,int)} with
   * {@link #DEFAULT_CPU_UTILIZATION_FACTOR}, {@link #DEFAULT_WAIT_TO_COMPUTE_RATIO},
   * and {@link #DEFAULT_MAX_THREADS}.
   * @return prepared thread pool executor
   */
  @NotNull
  public static ThreadPoolExecutor createDefaultExecutor()
  {
    return createDefaultExecutor(DEFAULT_MAX_THREADS);
  }

  /**
   * Create a default thread pool executor with a useful number of threads,
   * a keep-alive time of 0, a linked blocking queue, and a rejection handler
   * which discards all unexecutable threads.
   * @param basicThreadName      basic name used for threads, see {@link #defaultNamedThreadFactory(String)}
   * @param cpuUtilizationFactor how much of the CPU should be used to run the threads,
   *                             a number between {@code 0.0} (none) and {@code 1.0} (everything).
   *                             Leave a bit of room for other tasks your computer has to perform.
   * @param waitToComputeRatio   value which defines the ratio of the time your task are waiting for input
   *                             against the time they spend actual computing. A number between
   *                             {@code 0.0} (no waiting, only computing) and {@code 1.0} (no computing,
   *                             only waiting).
   * @param maxThreads           upper limit to the number of threads, at least 1
   * @return prepared thread pool executor
   */
  @NotNull
  public static ThreadPoolExecutor createDefaultExecutor(@NotNull String basicThreadName,
                                                         double cpuUtilizationFactor,
                                                         double waitToComputeRatio,
                                                         int maxThreads)
  {
    final int nThreads = Exec.calcNumThreads(cpuUtilizationFactor, waitToComputeRatio, maxThreads);
    final ThreadPoolExecutor exec = new ThreadPoolExecutor(nThreads, nThreads, 0L,
                                                           TimeUnit.MICROSECONDS,
                                                           new LinkedBlockingQueue<>(),
                                                           defaultNamedThreadFactory(basicThreadName));
    exec.setRejectedExecutionHandler(new ThreadPoolExecutor.DiscardPolicy());
    return exec;
  }

  /**
   * Get a default thread factory where threads get an individual name
   * made up from the given basic name.
   * This is basically the same as {@link Executors#defaultThreadFactory()},
   * only you can set a dedicated name.
   * <p>
   * Threads will get individual names made up from the given {@code baseName},
   * a pool number for the factory and a number for each thread in the individual pool
   * in the format {@code <baseName>-<poolnr>-<threadnr>}.
   * @param baseName basic name
   * @return thread factory which creates threads with names based on the given {@code baseName}
   */
  @NotNull
  public static ThreadFactory defaultNamedThreadFactory(@NotNull String baseName)
  {
    return new DefaultThreadFactory(baseName);
  }

  /**
   * Get a privileged thread factory where threads get an individual name
   * made up from the given basic name.
   * This is basically the same as {@link Executors#privilegedThreadFactory()},
   * only you can set a dedicated name.
   * <p>
   * Threads will get individual names made up from the given {@code baseName},
   * a pool number for the factory and a number for each thread in the individual pool
   * in the format {@code <baseName>-<poolnr>-<threadnr>}.
   * @param baseName basic name
   * @return thread factory which creates threads with names based on the given {@code baseName}
   */
  @NotNull
  public static ThreadFactory privilegedNamedThreadFactory(@NotNull String baseName)
  {
    return new PrivilegedThreadFactory(baseName);
  }

  /**
   * Get a privileged thread factory where threads get an individual name
   * made up from the given basic name.
   * This is basically the same as {@link Executors#privilegedThreadFactory()},
   * only you can set a dedicated name and a dedicated class loader.
   * <p>
   * Threads will get individual names made up from the given {@code baseName},
   * a pool number for the factory and a number for each thread in the individual pool
   * in the format {@code <baseName>-<poolnr>-<threadnr>}.
   * @param baseName basic name
   * @param classLoader class loader set as context class loader in the created threads
   * @return thread factory which creates threads with names based on the given {@code baseName}
   */
  @NotNull
  public static ThreadFactory privilegedNamedThreadFactory(@NotNull String baseName,
                                                           @NotNull ClassLoader classLoader)
  {
    return new PrivilegedThreadFactory(baseName, classLoader);
  }

  /**
   * This is a wrapper for a thread-pool executor which accepts runnable tasks
   * and can wait until all of them are finished.
   */
  public static class FinishingThreadPoolExecutor
          implements ExecutorService
  {
    @NotNull
    private final ThreadPoolExecutor executor;
    @NotNull
    private final Deque<Future<?>> queue = new ConcurrentLinkedDeque<>();

    /**
     * Constructor.
     *
     * @param executor wrapped executor
     */
    public FinishingThreadPoolExecutor(@NotNull ThreadPoolExecutor executor)
    {
      this.executor = executor;
    }

    @Override
    public void execute(@NotNull Runnable command)
    {
      submit(command);
    }

    @Override
    public void shutdown()
    {
      executor.shutdown();
    }

    @NotNull
    @Override
    public List<Runnable> shutdownNow()
    {
      return executor.shutdownNow();
    }

    @Override
    public boolean isShutdown()
    {
      return executor.isShutdown();
    }

    @Override
    public boolean isTerminated()
    {
      return executor.isTerminated();
    }

    @Override
    public boolean awaitTermination(long timeout, @NotNull TimeUnit unit) throws InterruptedException
    {
      return executor.awaitTermination(timeout, unit);
    }

    @NotNull
    @Override
    public Future<?> submit(@NotNull Runnable task)
    {
      final Future<?> future = executor.submit(task);
      queue.add(future);
      return future;
    }

    @NotNull
    @Override
    public <T> Future<T> submit(@NotNull Callable<T> task)
    {
      final Future<T> future = executor.submit(task);
      queue.add(future);
      return future;
    }

    @NotNull
    @Override
    public <T> Future<T> submit(@NotNull Runnable task, T result)
    {
      final Future<T> future = executor.submit(task, result);
      queue.add(future);
      return future;
    }

    @NotNull
    @Override
    public <T> List<Future<T>> invokeAll(@NotNull Collection<? extends Callable<T>> tasks) throws InterruptedException
    {
      final List<Future<T>> futures = executor.invokeAll(tasks);
      queue.addAll(futures);
      return futures;
    }

    @NotNull
    @Override
    public <T> List<Future<T>> invokeAll(@NotNull Collection<? extends Callable<T>> tasks, long timeout, @NotNull TimeUnit unit)
            throws InterruptedException
    {
      final List<Future<T>> futures = executor.invokeAll(tasks, timeout, unit);
      queue.addAll(futures);
      return futures;
    }

    @NotNull
    @Override
    public <T> T invokeAny(@NotNull Collection<? extends Callable<T>> tasks) throws InterruptedException, ExecutionException
    {
      return executor.invokeAny(tasks);
    }

    @Override
    public <T> T invokeAny(@NotNull Collection<? extends Callable<T>> tasks, long timeout, @NotNull TimeUnit unit)
            throws InterruptedException, ExecutionException, TimeoutException
    {
      return executor.invokeAny(tasks, timeout, unit);
    }

    /**
     * Wait until this executor is finished.
     * @throws ExecutionException   wrapped exception from one of the tasks
     * @throws InterruptedException if this executor was interrupted
     */
    public void waitUntilFinished() throws ExecutionException, InterruptedException
    {
      while (!queue.isEmpty()) {
        queue.removeFirst().get();
      }
    }
  }

  /**
   * Default thread factory.
   * This follows {@code Executors#DefaultThreadFactory} closely
   * to avoid unexpected behavior.
   */
  static class DefaultThreadFactory
          implements ThreadFactory
  {
    /**
     * Pool number.
     * Increased for each factory created.
     */
    @NotNull
    private static final AtomicInteger poolNumber = new AtomicInteger(1);
    @NotNull
    private final ThreadGroup threadGroup;
    /**
     * Thread number.
     * Increased for each thread created by this factory.
     */
    @NotNull
    private final AtomicInteger threadNumber = new AtomicInteger(1);
    @NotNull
    private final String namePrefix;

    /**
     * Constructor.
     * @param baseName basic name used for threads created by this factory
     */
    DefaultThreadFactory(@NotNull String baseName)
    {
      final SecurityManager sm = System.getSecurityManager();
      threadGroup = sm != null
              ? sm.getThreadGroup()
              : Thread.currentThread().getThreadGroup();
      this.namePrefix = baseName + "-" + poolNumber.getAndIncrement() + "-thread-";
    }

    @Override
    @NotNull
    public Thread newThread(@NotNull Runnable r)
    {
      final Thread thread = new Thread(threadGroup, r, namePrefix + threadNumber.getAndIncrement(), 0);
      if (thread.isDaemon()) {
        thread.setDaemon(false);
      }
      if (thread.getPriority() != Thread.NORM_PRIORITY) {
        thread.setPriority(Thread.NORM_PRIORITY);
      }
      return thread;
    }
  }

  /**
   * Privileged thread factor.
   * This follows {@code Executors#PrivilegedThreadFactory} closely
   * to avoid unexpected behavior.
   */
  static class PrivilegedThreadFactory
          extends DefaultThreadFactory
  {
    private final AccessControlContext accessControlContext;
    private final ClassLoader classLoader;

    /**
     * Constructor.
     * @param baseName basic name used for threads created by this factory
     */
    public PrivilegedThreadFactory(@NotNull String baseName)
    {
      super(baseName);
      accessControlContext = AccessController.getContext();
      classLoader = Thread.currentThread().getContextClassLoader();
    }

    /**
     * Constructor allowing to set an individual class loader.
     * @param baseName     basic name used for threads created by this factory
     * @param classLoader  class loader to use for threads
     */
    public PrivilegedThreadFactory(@NotNull String baseName,
                                   @NotNull ClassLoader classLoader)
    {
      super(baseName);
      final SecurityManager sm = System.getSecurityManager();
      if (sm != null) {
        // Fail now and not late on thread creation
        sm.checkPermission(new RuntimePermission("setContextClassLoader"));
      }
      accessControlContext = AccessController.getContext();
      this.classLoader = classLoader;
    }

    @Override
    @NotNull
    public Thread newThread(@NotNull Runnable r)
    {
      return super.newThread(() -> AccessController.doPrivileged((PrivilegedAction<Void>)() -> {
        Thread.currentThread().setContextClassLoader(classLoader);
        r.run();
        return null;
      }, accessControlContext));
    }
  }

}
