//=============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2005-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================

package de.caff.util.args;

import de.caff.annotation.NotNull;

/**
 * Exception thrown if a long switch comes with an appended argument,
 * although it does not expect one.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since January 27, 2019
 */
public class UnexpectedArgumentException
        extends Exception
{
  private static final long serialVersionUID = -817156488481847130L;

  @NotNull
  private final String longSwitch;
  @NotNull
  private final String argument;

  /**
   * Constructor.
   * @param longSwitch long switch (including {@code --} prefix)
   * @param argument argument appended to switch
   */
  public UnexpectedArgumentException(@NotNull String longSwitch, @NotNull String argument)
  {
    super(String.format("Found %s=%s, but %s does not have arguments!", longSwitch, argument, longSwitch));
    this.longSwitch = longSwitch;
    this.argument = argument;
  }

  /**
   * Get the long switch for which an unexpected argument was defined.
   * @return long switch
   */
  @NotNull
  public String getLongSwitch()
  {
    return longSwitch;
  }

  /**
   * Get the unexpected argument of the long switch.
   * @return argument
   */
  @NotNull
  public String getArgument()
  {
    return argument;
  }
}
