//=============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2005-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.args;

import de.caff.annotation.NotNull;

/**
 * A command line argument or a command line switch.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since January 27, 2019
 */
public interface SwitchOrArgument
{
  /**
   * Is this a switch?
   * @return {@code true}: this is a switch and can be casted to {@link Switch},
   *         {@code false}: this is a argument and can be casted to {@link Argument}
   */
  boolean isSwitch();

  /**
   * Is this argument optional?
   * For switches the order is basically ignored, but it is not allowed
   * that non-optional arguments appear after optional ones
   * @return {@code true} when this switch or argument is optional, <br>
   *         {@code false} if it is required
   */
  boolean isOptional();

  /**
   * Does this argument or switch have a variable length?
   * This is only allowed for the last argument/switch in the command line.
   * @return {@code true}: this argument or switch consumes the remaining items in the command line<br>
   *         {@code false}: this argument or switch just consumes 1 item (for arguments) or 2 items (for switches) from the command line
   */
  boolean isVarLength();

  /**
   * Get the appearance of this switch or argument on the command line.
   * @return appearance
   */
  @NotNull
  String getAppearance();

  /**
   * Get a short description for this switch.
   * @return short description
   */
  @NotNull
  String getDescription();

  /**
   * Read further arguments from the command line.
   * The default implementation does not remove any arguments.
   * @param arg argument
   * @return if more arguments may be consumed
   */
  boolean consumeArgument(@NotNull String arg);
}
