//=============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2005-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.args;

import de.caff.annotation.NotNull;

import java.util.Collection;

/**
 * A basic command line switch.
 * <p>
 * A switch can appear in a short one-character form
 * like {@code -k} or a long form like {@code --kill-all-idiots}.
 * Long forms can have an argument directly append separated by
 * an equals-sign ({@code =}), or as next command line argument.
 * Short forms only accept the latter.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since January 27, 2019
 */
public interface Switch
        extends SwitchOrArgument
{
  @Override
  default boolean isSwitch()
  {
    return true;
  }

  /**
   * Get the short one character forms influencing this switch.
   * <p>
   * If the command line switch is {@code -k}, this method
   * should return a singleton collection with the character
   * {@code k}.
   * @return short one character forms, may be empty
   */
  @NotNull
  Collection<Character> getShortForms();

  /**
   * Get the long forms influencing this switch.
   * <p>
   * The return strings must not include the {@code --} prefix,
   * and have to consist of characters, digits and the minus
   * sign ({@code -}.
   * <p>
   * So if the command line switch is {@code --kill-all-idiots},
   * this method should return a singleton collection with the
   * string {@code kill-all-idiots}.
   * @return long switches. may be empty
   */
  @NotNull
  Collection<String> getLongForms();

  /**
   * Is this an optional switch?
   * Non-optional switches have to appear on the command line.
   * The default implementation returns {@code true}.
   * @return {@code true}: if this switch is optional<br>
   *         {@code false}: if this switch is required
   */
  default boolean isOptional()
  {
    return true;
  }

  /**
   * This switch appears in the command line in the given short form.
   * @param shortSwitch short form in which this switch appeared
   */
  void found(char shortSwitch);

  /**
   * This switch appears in the command line in the given long form.
   * @param longSwitch long form
   */
  void found(@NotNull String longSwitch);

  /**
   * Does this switch require arguments?
   * Only if it requires arguments, the {@link #consumeArgument(String)}
   * method will be called.
   * The default implementation returns {@code false}, so always both
   * this method and the {@link #consumeArgument(String)} method need to
   * be overridden.
   * @return {@code true} if this switch requires arguments<br>
   *         {@code false} if this switch does not require arguments
   * @see #consumeArgument(String)
   */
  default boolean needsArguments()
  {
    return false;
  }

  /**
   * Read further arguments from the command line.
   *
   * @param arg argument
   * @return if more arguments may be consumed
   * @see #needsArguments()
   */
  default boolean consumeArgument(@NotNull String arg)
  {
    return false;
  }
}
