// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2005-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.args;

import de.caff.annotation.NotNull;

import static de.caff.util.args.CommandLine.NON_BREAKING_SPACE;

/**
 * Switch expecting a string argument.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since January 28, 2019
 */
public class StringArgSwitch
        extends AbstractSwitch
{
  @NotNull
  private final String argName;
  @NotNull
  private String argument;

  /**
   *
   * @param shortForm       short form of the switch
   * @param longForm        long form of the switch
   * @param argName         name used for the argument
   * @param defaultArgument default value for the argument
   * @param description     description of the switch
   */
  public StringArgSwitch(@NotNull Character shortForm,
                         @NotNull String longForm,
                         @NotNull String argName,
                         @NotNull String defaultArgument,
                         @NotNull String description)
  {
    super(shortForm, longForm, description);
    this.argName = argName;
    this.argument = defaultArgument;
  }

  @Override
  public boolean isVarLength() {
    return false;
  }

  /**
   * This switch appears in the command line in the given short form.
   *
   * @param shortSwitch short form in which this switch appeared
   */
  @Override
  public void found(char shortSwitch)
  {
  }

  /**
   * This switch appears in the command line in the given long form.
   *
   * @param longSwitch long form
   */
  @Override
  public void found(@NotNull String longSwitch)
  {
  }

  /**
   * Get the appearance of this switch or argument on the command line.
   *
   * @return appearance
   */
  @NotNull
  @Override
  public String getAppearance()
  {
    return String.format("-%s"+NON_BREAKING_SPACE+"<%s>|--%s=<%s>", shortForm, argName, longForm, argName);
  }

  /**
   * Does this switch require arguments?
   * Only if it requires arguments, the {@link #consumeArgument(String)}
   * method will be called.
   * The default implementation returns {@code false}, so always both
   * this method and the {@link #consumeArgument(String)} method need to
   * be overridden.
   *
   * @return {@code true} if this switch requires arguments<br>
   * {@code false} if this switch does not require arguments
   * @see #consumeArgument(String)
   */
  @Override
  public boolean needsArguments()
  {
    return true;
  }

  /**
   * Read further arguments from the command line.
   *
   * @param arg argument
   * @return if more arguments may be consumed
   * @see #needsArguments()
   */
  @Override
  public boolean consumeArgument(@NotNull String arg)
  {
    argument = arg;
    return false;
  }

  /**
   * Get the argument.
   * @return argument
   */
  @NotNull
  public String getArgument()
  {
    return argument;
  }
}
