// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.args;

import de.caff.annotation.NotNull;

/**
 * A simple optional switch which either appears on the command line or not.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since January 27, 2019
 */
public class SimpleSwitch
        extends AbstractSwitch
{
  private boolean appeared;

  /**
   * Constructor.
   * @param shortForm short form of switch
   * @param longForm  long form of switch
   * @param description description of this switch
   */
  public SimpleSwitch(@NotNull Character shortForm,
                      @NotNull String longForm,
                      @NotNull String description)
  {
    super(shortForm, longForm, description);
  }

  @Override
  public boolean isVarLength() {
    return false;
  }

  @NotNull
  @Override
  public String getAppearance()
  {
    return String.format("-%s|--%s", shortForm, longForm);
  }

  @Override
  public void found(char shortSwitch)
  {
    appeared = true;
  }

  @Override
  public void found(@NotNull String longSwitch)
  {
    appeared = true;
  }

  /**
   * Has the switch appeared on the command line?
   * @return {@code true}: the switch was included on the command line<br>
   *         {@code false}: the switch was not found on the command line
   */
  public boolean hasAppeared()
  {
    return appeared;
  }

  /**
   * Set whether this switch has appeared.
   * Usually this should be set automatically via the {@code found()}
   * methods.
   * @param appeared {@code true}: the switch was included on the command line<br>
   *                 {@code false}: the switch was not found on the command line
   */
  public void setAppeared(boolean appeared)
  {
    this.appeared = appeared;
  }

}
